package com.xforceplus.ultraman.transfer.client;

import com.fasterxml.jackson.databind.JsonNode;
import com.xforceplus.ultraman.transfer.client.config.BocpClientSetting;
import com.xforceplus.ultraman.transfer.client.config.OqsSdkProperties;
import com.xforceplus.ultraman.transfer.client.listener.IBocpServerMessageListener;
import com.xforceplus.ultraman.transfer.common.context.MetadataContextHolder;
import com.xforceplus.ultraman.transfer.common.util.JsonUtils;
import com.xforceplus.ultraman.transfer.common.util.MessageUtils;
import com.xforceplus.ultraman.transfer.domain.entity.TransferMessage;
import com.xforceplus.ultraman.transfer.domain.enums.MessageType;
import com.xforceplus.ultraman.transfer.storage.http.BocpOkHttpClient;
import com.xforceplus.ultraman.transfer.storage.http.interceptor.MetadataHeadInterceptor;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.lang3.StringUtils;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.*;

/**
 * http
 */

@Slf4j
public class BocpHttpClient extends BocpOkHttpClient implements IBocpClient {

    public final String GET_METADATA_PATH = "%smetadatas/app-versions";
    private String bocpHost;
    private Boolean useSsl;
    private String apiPrefix;
    
    private OqsSdkProperties properties;
    
    private BocpClientSetting bocpClientSetting;
    private final OkHttpClient client;

    private List<IBocpServerMessageListener> listeners;

    private CountDownLatch latch = new CountDownLatch(1);
    
    private boolean initIsTriggered = false;

    ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(1);

    public BocpHttpClient(boolean useSSl, BocpClientSetting bocpClientSetting, OqsSdkProperties oqsSdkProperties, List<IBocpServerMessageListener> listeners) {
        this.useSsl = useSSl;
        this.client = buildClient(useSsl, Collections.emptyList());
        this.properties = oqsSdkProperties;
        this.bocpClientSetting = bocpClientSetting;
        this.listeners = listeners;
    }
    
    public BocpHttpClient( String bocpHost,
                           Boolean useSsl,
                           String apiPrefix,
                           OqsSdkProperties oqsSdkProperties,
                           BocpClientSetting bocpClientSetting,
                           MetadataHeadInterceptor metadataHeadInterceptor, List<IBocpServerMessageListener> listeners) {
        this.bocpClientSetting = bocpClientSetting;
        this.bocpHost = bocpHost;
        this.useSsl = useSsl;
        this.apiPrefix = apiPrefix;
        this.client = buildClient(useSsl, Arrays.asList(metadataHeadInterceptor));
        this.listeners = listeners;
        this.properties = oqsSdkProperties;
        init();
    }
    
    protected void init() {
        String appId = properties.getAuth().getAppId();
        String env = properties.getAuth().getEnv();
        RetryConfig config = RetryConfig.custom()
                .maxAttempts(bocpClientSetting.getBocp().getFailureThreshold())
                .waitDuration(Duration.ofMillis(100)).build();
        Retry requestRetry = Retry.of("httpRequest", config);

        Long appIdLong;
        Long envLong;
        try {
            appIdLong = Long.parseLong(appId);
            envLong = Long.parseLong(env);

            try {
                requestRetry.executeRunnable(() -> requestLastMetadata(appIdLong, envLong));
            } catch (Throwable throwable) {
                if(bocpClientSetting.getBocp().isSupportOffline()) {
                    if (!initIsTriggered) {
                        TransferMessage transferMessage = new TransferMessage();
                        transferMessage.setAppId(Long.parseLong(properties.getAuth().getAppId()));
                        transferMessage.setMessageType(MessageType.CLIENT_CONNECT);
                        transferMessage.setHandleSuccess(true);

                        Optional.ofNullable(listeners)
                                .orElseGet(Collections::emptyList).forEach(listener -> {
                                    try {
                                        initIsTriggered = true;
                                        listener.onTransferMessage(transferMessage).thenAccept(x -> latch.countDown());
                                    } catch (Throwable e) {
                                        log.error("execute message task failed!", e);
                                    }
                                });
                    }
                }
            }
            scheduledExecutorService.scheduleAtFixedRate(() -> requestLastMetadata(appIdLong, envLong)
                    , 3L, 1L, TimeUnit.MINUTES);
            try {
                latch.await();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        } catch (Throwable throwable) {
            throw throwable;
        }
    }
    
    private void requestLastMetadata(Long appId, Long envId) {
        Request request = buildRequest(appId, envId);
        try (Response resp = client.newCall(request).execute()) {
            if (!resp.isSuccessful()) {
                throw new RuntimeException("调用获取元数据接口失败, msg = " + resp.message());
            }
            if(resp.body() != null) {
                String body = resp.body().string();
                JsonNode jsonNode = JsonUtils.readTree(body);
                JsonNode transferBody = jsonNode.get("data");
                if(transferBody != null) {
                    String text = transferBody.toString();
                    TransferMessage transferMessages = JsonUtils.json2Object(text, TransferMessage.class);
                    String versionStr = transferMessages.getVersion();
                    String version = MetadataContextHolder.currentVersion();
                    transferMessages.setMessageType(MessageType.CLIENT_CONNECT);
                    transferMessages.setHandleSuccess(true);
                    
                    if(!versionStr.equalsIgnoreCase(version)) {
                        Optional.ofNullable(listeners)
                                .orElseGet(Collections::emptyList).forEach(listener -> {
                                    try {
                                        listener.onTransferMessage(transferMessages).thenAccept(x -> latch.countDown());
                                    } catch (Throwable e) {
                                        log.error("execute message task failed!", e);
                                    }
                                }); 
                        }
                    }
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    protected Request buildRequest(Long appId, Long envId) {
        String realApiPrefix = StringUtils.isBlank(apiPrefix) ? "" : apiPrefix + "/";
        String realApiPath = String.format(GET_METADATA_PATH, realApiPrefix, envId);
        String url = MessageUtils.getUrl(bocpHost, useSsl);
        HttpUrl hostUrl = HttpUrl.get(url);

        HttpUrl httpUrl = new HttpUrl.Builder()
                .scheme(MessageUtils.getSchema(useSsl))
                .host(hostUrl.host())
                .port(hostUrl.port())
                .addPathSegments(realApiPath)  // 添加路径段，如果有多个路径段，可以多次调用这个方法
                .addQueryParameter("appId", appId.toString())  // 添加查询参数// 添加查询参数
                .addQueryParameter("envId", envId.toString())  // 添加查询参数// 添加查询参数
//                .addQueryParameter("secret", "A0AAE12DB94B4212872F437BA168D7DB")  // 添加查询参数// 添加查询参数
                .build();
        return new Request.Builder()
                .url(httpUrl)
                .build();
    }

    @Override
    public void sendMessage(String message) {
        //do nothing
        log.warn("SendMessage not supported");
    }
}
