package com.xforceplus.ultraman.transfer.client.controller;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xforceplus.ultraman.metadata.jsonschema.enums.SchemaMetadataType;
import com.xforceplus.ultraman.transfer.client.BocpClient;
import com.xforceplus.ultraman.transfer.client.config.OqsSdkProperties;
import com.xforceplus.ultraman.transfer.common.context.MetadataContextHolder;
import com.xforceplus.ultraman.transfer.common.context.MetadataI18nContextHolder;
import com.xforceplus.ultraman.transfer.common.util.JsonUtils;
import com.xforceplus.ultraman.transfer.domain.entity.DeployDetail;
import com.xforceplus.ultraman.transfer.domain.entity.TransferMessage;
import com.xforceplus.ultraman.transfer.domain.enums.MessageType;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.net.URL;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

/**
 * 用户维护及查看socket信息
 *
 * @author leo
 * @version 0.1 2022/11/30 09:26
 * @since 1.8
 */
@RequestMapping("/bocpclient")
@Slf4j
public class BocpClientController {

    private OqsSdkProperties oqsSdkProperties;
    private BocpClient bocpClient;

    public BocpClientController(OqsSdkProperties oqsSdkProperties, BocpClient bocpClient) {
        this.oqsSdkProperties = oqsSdkProperties;
        this.bocpClient = bocpClient;
    }

    @PostMapping("/send")
    @ResponseBody
    public void sendReplayMessage(@RequestBody TransferMessage replyMessage) {
        replyMessage.setReplyMessage(true);
        bocpClient.sendMessage(JsonUtils.object2Json(replyMessage));
    }

    @GetMapping("/pull")
    @ResponseBody
    public void pull() {
        TransferMessage message = new TransferMessage();
        message.setAppId(Long.valueOf(oqsSdkProperties.getAuth().getAppId()));
        message.setEnv(oqsSdkProperties.getAuth().getEnv());
        message.setMessageType(MessageType.CLIENT_PULL);
        message.setForceUpdate(true);
        bocpClient.sendMessage(JsonUtils.object2Json(message));
    }

    @GetMapping("/current")
    @ResponseBody
    public ResponseEntity current() {
        return ResponseEntity.ok(buildCurrentInfo());
    }

    @GetMapping("/current/details")
    @ResponseBody
    public ResponseEntity currentDetails(@RequestParam(required = false) String[] types) {
        CurrentInfo currentInfo = buildCurrentInfo();
        currentInfo.setMetadataMap(buildMetadataMap(types));
        return ResponseEntity.ok(currentInfo);
    }

    @GetMapping("/current/i18nResources")
    @ResponseBody
    public ResponseEntity currentI18nResource(@RequestParam String resourceCode, @RequestParam String tenantCode) {
        return ResponseEntity.ok(MetadataI18nContextHolder.getTenantI18nResourceMap(resourceCode, tenantCode));
    }

    @GetMapping("/dependency")
    @ResponseBody
    public ResponseEntity dependency() throws IOException {
        List<Map> dependencyVersions = getDependencyVersions();
        return ResponseEntity.ok(dependencyVersions);
    }

    private List<Map> getDependencyVersions() throws IOException {
        ClassLoader classLoader = this.getClass().getClassLoader();

        Enumeration<URL> resources = classLoader.getResources("META-INF/MANIFEST.MF");
        List<Map> result = Lists.newArrayList();

        while (resources.hasMoreElements()) {
            try {
                URL url = resources.nextElement();
                Manifest manifest = new Manifest(url.openStream());
                Attributes attributes = manifest.getMainAttributes();
                String implementationTitle = attributes.getValue("Implementation-Title");
                String implementationVersion = attributes.getValue("Implementation-Version");

                if (implementationTitle != null && implementationVersion != null) {
                    Map item = Maps.newHashMap();
                    item.put("dependency", implementationTitle);
                    item.put("version", implementationVersion);
                    result.add(item);
                }
            } catch (Exception e) {
                log.error("获取依赖版本信息失败！", e);
            }
        }
        return result;
    }

    @Data
    public static class CurrentInfo {
        private Long appId = null;
        private String env = null;
        private String appCode = null;
        private String appCodeForDB = null;
        private String version = StringUtils.EMPTY;
        private DeployDetail deployDetail = null;
        private Map<SchemaMetadataType, Object> metadataMap = Maps.newConcurrentMap();
    }

    private CurrentInfo buildCurrentInfo() {
        CurrentInfo currentInfo = new CurrentInfo();
        currentInfo.setEnv(oqsSdkProperties.getAuth().getEnv());
        currentInfo.setVersion(MetadataContextHolder.currentVersion());
        if ("99".equals(oqsSdkProperties.getAuth().getEnv())) {
            String version = MetadataContextHolder.currentVersion();
            String[] split = version.split("\\.");
            String timestamp = split[1];
            String format = LocalDateTime.ofInstant(Instant.ofEpochMilli(Long.valueOf(timestamp)),
                    ZoneId.systemDefault()).format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
            currentInfo.setVersion(split[0] + "." + format);
        }
        currentInfo.setAppCode(MetadataContextHolder.appCode());
        currentInfo.setAppCodeForDB(MetadataContextHolder.appCodeForDB());
        currentInfo.setAppId(MetadataContextHolder.appId());
        currentInfo.setDeployDetail(MetadataContextHolder.getDeployDetail());
        return currentInfo;
    }

    private Map<SchemaMetadataType, Object> buildMetadataMap(String[] types) {
        Map<SchemaMetadataType, Object> map = Maps.newHashMap();
        if (null == types || types.length == 0) {
            types = new String[] {
                    SchemaMetadataType.DICT.code(),
                    SchemaMetadataType.ENTITY.code(),
                    SchemaMetadataType.SDK_SETTING.code(),
                    SchemaMetadataType.ACTION.code(),
                    SchemaMetadataType.APP_EVENT.code(),
                    SchemaMetadataType.APP_I18N_LOCALE.code()
            };
        }
        Arrays.stream(types).forEach(type -> {
            SchemaMetadataType schemaMetadataType = SchemaMetadataType.fromCode(type);
            if (SchemaMetadataType.DICT.equals(schemaMetadataType)) {
                map.put(SchemaMetadataType.DICT, MetadataContextHolder.getDictMetadata());
            } else if (SchemaMetadataType.ENTITY.equals(schemaMetadataType)) {
                map.put(SchemaMetadataType.ENTITY, MetadataContextHolder.getEntityMetadata());
            } else if (SchemaMetadataType.SDK_SETTING.equals(schemaMetadataType)) {
                map.put(SchemaMetadataType.SDK_SETTING, MetadataContextHolder.getSdkSettingMetadata());
            } else if (SchemaMetadataType.ACTION.equals(schemaMetadataType)) {
                map.put(SchemaMetadataType.ACTION, MetadataContextHolder.getActionMetadata());
            } else if (SchemaMetadataType.APP_EVENT.equals(schemaMetadataType)) {
                map.put(SchemaMetadataType.APP_EVENT, MetadataContextHolder.getAppEventMetadata());
            } else if (SchemaMetadataType.APP_I18N_LOCALE.equals(schemaMetadataType)) {
                map.put(SchemaMetadataType.APP_I18N_LOCALE, MetadataContextHolder.getAppI18nLocaleMetadata());
            }
        });
        return map;
    }

}
