package com.xforceplus.ultraman.transfer.client;

import com.xforceplus.ultraman.transfer.client.config.BocpClientSetting;
import com.xforceplus.ultraman.transfer.client.config.OqsSdkProperties;
import com.xforceplus.ultraman.transfer.client.listener.IBocpServerMessageListener;
import com.xforceplus.ultraman.transfer.common.constant.TransferConstant;
import com.xforceplus.ultraman.transfer.common.util.JsonUtils;
import com.xforceplus.ultraman.transfer.common.util.MessageUtils;
import com.xforceplus.ultraman.transfer.common.util.TransferUtils;
import com.xforceplus.ultraman.transfer.domain.entity.TransferMessage;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

/**
 * sync metadata from BOCP server.
 *
 * @author leo
 * @version 0.1 2022/11/7 17:42
 * @since 1.8
 */
@Slf4j
public class BocpClient extends WebSocketListener {

    private static final Integer INTERVAL_IN_MILLI_SECONDS = 15000;
    /**
     * CLIENT ID
     */
    private String clientId;
    /**
     * OkHttpClient
     */
    private OkHttpClient mOkHttpClient;
    /**
     * 当前链接对象
     */
    private WebSocket webSocket;
    /**
     * WebSocket连接请求
     */
    private Request mRequest;

    private BocpClientSetting bocpClientSetting;
    private OqsSdkProperties oqsSdkProperties;
    private volatile boolean isConnected = false;

    private List<IBocpServerMessageListener> listeners;

    private CountDownLatch latch = new CountDownLatch(1);

    public BocpClient(BocpClientSetting bocpClientSetting, OqsSdkProperties oqsSdkProperties, List<IBocpServerMessageListener> listeners) {
        this.oqsSdkProperties = oqsSdkProperties;
        this.bocpClientSetting = bocpClientSetting;
        this.clientId = TransferUtils.getAppEnvClientKey(oqsSdkProperties.getAuth().getAppId(), oqsSdkProperties.getAuth().getEnv(), UUID.randomUUID().toString());
        this.mOkHttpClient = buildClient();
        this.mRequest = buildRequest();
        if(listeners == null) {
            this.listeners = Collections.emptyList();
        } else {
            this.listeners = listeners;
        }

        this.listeners.forEach(x -> x.setBocpClient(this));

        log.debug("request info {}", this.mRequest);
        this.connect();
        try {
            latch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onOpen(WebSocket webSocket, Response response) {
        isConnected = true;
        log.info("success to create websocket connection {}", response.body());
    }

    @Override
    public void onMessage(WebSocket webSocket, String message) {
        log.debug("receive message :{}", message);
        TransferMessage transferMessage = JsonUtils.json2Object(message, TransferMessage.class);
        Optional.ofNullable(listeners)
                .orElseGet(Collections::emptyList).forEach(listener -> {
                    try {
                        listener.onTransferMessage(transferMessage).thenAccept(x -> latch.countDown());
                    } catch (Throwable e) {
                        log.error("execute message task failed!", e);
                    }
                });

    }

    @Override
    public void onClosed(WebSocket webSocket, int code, String reason) {
        isConnected = false;
        log.warn("websocket connection closed, code :{}, reason:{}", code, reason);
    }

    @Override
    public void onFailure(WebSocket webSocket, Throwable t, Response response) {
        isConnected = false;
        log.warn("failed to create websocket connection", t);
        this.retry();
    }

    /**
     * 发送消息
     *
     * @param message
     */
    public void sendMessage(String message) {
        if (!Optional.ofNullable(this.webSocket).isPresent()) {
            this.connect();
        }
        boolean success = this.webSocket.send(message);
        log.debug("websocket send message {}: {}", success ? "success" : "failed", message);
    }

    private synchronized void connect() {
        if (webSocket == null || !isConnected) {
            log.debug("start to create websocket connection");
            if (mRequest == null) {
                mRequest = buildRequest();
            }
            this.mOkHttpClient.dispatcher().cancelAll();
            try {
                webSocket = mOkHttpClient.newWebSocket(mRequest, this);
                isConnected = true;
            } catch (Exception e) {
                isConnected = false;
            }
        }
    }

    private synchronized void retry() {
        try {
            Thread.sleep(INTERVAL_IN_MILLI_SECONDS);
            log.info("retry to create websocket connection");
            connect();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    @SneakyThrows
    private OkHttpClient buildClient() {
        OkHttpClient.Builder builder = new OkHttpClient().newBuilder()
                .retryOnConnectionFailure(true)
                .readTimeout(60, TimeUnit.SECONDS)
                .writeTimeout(60, TimeUnit.SECONDS)
                .pingInterval(10, TimeUnit.SECONDS);
        if (bocpClientSetting.getBocp().isUseSsl()) {
            //默认信任所有证书
            final X509TrustManager trustManager = new X509TrustManager() {
                @Override
                public void checkServerTrusted(X509Certificate[] chain, String authType) {
                }

                @Override
                public void checkClientTrusted(X509Certificate[] x509Certificates, String s) {
                }

                @Override
                public X509Certificate[] getAcceptedIssuers() {
                    return new X509Certificate[0];
                }
            };
            SSLContext sslContext = SSLContext.getInstance("TLS");
            sslContext.init(null, new TrustManager[]{trustManager}, new SecureRandom());
            return builder.sslSocketFactory(sslContext.getSocketFactory(), trustManager)
                    .hostnameVerifier((s, sslSession) -> true).build();
        } else {
            return builder.build();
        }
    }

    private Request buildRequest() {
        return new Request.Builder()
                .url(MessageUtils.getRealWsUrl(bocpClientSetting.getBocp().getHost(), bocpClientSetting.getBocp().isUseSsl()))
                .addHeader(TransferConstant.WEBSOCKET_AUTH_TOKEN, TransferConstant.authToken)
                .addHeader(TransferConstant.APP_ID, oqsSdkProperties.getAuth().getAppId())
                .addHeader(TransferConstant.ENV, oqsSdkProperties.getAuth().getEnv())
                .addHeader(TransferConstant.CLIENT_ID, this.clientId)
                .build();
    }

}
