package com.xforceplus.ultraman.transfer.client.controller;

import com.google.common.collect.Maps;
import com.xforceplus.ultraman.metadata.jsonschema.enums.SchemaMetadataType;
import com.xforceplus.ultraman.transfer.client.BocpClient;
import com.xforceplus.ultraman.transfer.client.config.OqsSdkProperties;
import com.xforceplus.ultraman.transfer.common.context.MetadataContextHolder;
import com.xforceplus.ultraman.transfer.common.util.JsonUtils;
import com.xforceplus.ultraman.transfer.domain.entity.TransferMessage;
import com.xforceplus.ultraman.transfer.domain.enums.MessageType;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Map;

/**
 * 用户维护及查看socket信息
 *
 * @author leo
 * @version 0.1 2022/11/30 09:26
 * @since 1.8
 */
@RequestMapping("/bocpclient")
public class BocpClientController {

    private OqsSdkProperties oqsSdkProperties;
    private BocpClient bocpClient;

    public BocpClientController(OqsSdkProperties oqsSdkProperties, BocpClient bocpClient) {
        this.oqsSdkProperties = oqsSdkProperties;
        this.bocpClient = bocpClient;
    }

    @PostMapping("/send")
    @ResponseBody
    public void sendReplayMessage(@RequestBody TransferMessage replyMessage) {
        bocpClient.sendMessage(JsonUtils.object2Json(replyMessage));
    }


    @GetMapping("/pull")
    @ResponseBody
    public void pull() {
        TransferMessage message = new TransferMessage();
        message.setAppId(Long.valueOf(oqsSdkProperties.getAuth().getAppId()));
        message.setEnv(oqsSdkProperties.getAuth().getEnv());
        message.setMessageType(MessageType.CLIENT_PULL);
        message.setForceUpdate(true);
        bocpClient.sendMessage(JsonUtils.object2Json(message));
    }

    @GetMapping("/current")
    @ResponseBody
    public ResponseEntity current() {
        CurrentInfo currentInfo = new CurrentInfo();
        currentInfo.setEnv(oqsSdkProperties.getAuth().getEnv());
        currentInfo.setVersion(MetadataContextHolder.currentVersion());
        if ("99".equals(oqsSdkProperties.getAuth().getEnv())) {
            String version = MetadataContextHolder.currentVersion();
            String[] split = version.split("\\.");
            String timestamp = split[1];
            String format = LocalDateTime.ofInstant(Instant.ofEpochMilli(Long.valueOf(timestamp)),
                    ZoneId.systemDefault()).format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
            currentInfo.setVersion(split[0] + "." + format);
        }
        currentInfo.setAppCode(MetadataContextHolder.appCode());
        currentInfo.setAppCodeForDB(MetadataContextHolder.appCodeForDB());
        currentInfo.setAppId(MetadataContextHolder.appId());
        return ResponseEntity.ok(currentInfo);
    }

    @Data
    public static class CurrentInfo {
        private Long appId = null;
        private String env = null;
        private String appCode = null;
        private String appCodeForDB = null;
        private String version = StringUtils.EMPTY;
        private Map<SchemaMetadataType, Object> metadataMap = Maps.newConcurrentMap();

    }
}
