package com.xforceplus.ultraman.transfer.client.listener.impl;

import com.google.common.collect.Lists;
import com.xforceplus.ultraman.metadata.jsonschema.pojo.*;
import com.xforceplus.ultraman.transfer.client.IBocpClient;
import com.xforceplus.ultraman.transfer.client.config.BocpClientSetting;
import com.xforceplus.ultraman.transfer.client.config.OqsSdkProperties;
import com.xforceplus.ultraman.transfer.common.context.MetadataContextHolder;
import com.xforceplus.ultraman.transfer.client.listener.IBocpServerMessageListener;
import com.xforceplus.ultraman.transfer.client.thread.MessageWorkerManager;
import com.xforceplus.ultraman.transfer.common.event.publisher.EventStream;
import com.xforceplus.ultraman.transfer.common.util.MessageUtils;
import com.xforceplus.ultraman.transfer.common.util.JsonUtils;
import com.xforceplus.ultraman.transfer.common.util.VersionUtils;
import com.xforceplus.ultraman.transfer.domain.dto.AppInfo;
import com.xforceplus.ultraman.transfer.domain.entity.DeployDetail;
import com.xforceplus.ultraman.transfer.domain.entity.TenantAppDeployInfo;
import com.xforceplus.ultraman.transfer.domain.enums.MessageType;
import com.xforceplus.ultraman.metadata.jsonschema.enums.SchemaMetadataType;
import com.xforceplus.ultraman.transfer.domain.entity.TransferMessage;
import com.xforceplus.ultraman.transfer.common.event.SDKMetadataEvent;

import com.xforceplus.ultraman.transfer.storage.aggregator.strategy.MetadataStorageRepository;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

/**
 * .
 *
 * @author leo
 * @version 0.1 2022/11/7 19:50
 * @since 1.8
 */
@Slf4j
public class BocpServerMessageListenerImpl implements IBocpServerMessageListener {

    private final Integer GET_METADATA_RETRY_TIMES = 3;
    private final Integer GET_METADATA_WAIT_TIME = 5000;
    private IBocpClient bocpClient;
    private BocpClientSetting bocpClientSetting;
    private OqsSdkProperties oqsSdkProperties;
    private MetadataStorageRepository metadataStorageRepository;
    private MessageWorkerManager messageWorkerManager;
    private EventStream<SDKMetadataEvent> eventEventStream;

    public BocpServerMessageListenerImpl(
            BocpClientSetting bocpClientSetting,
            OqsSdkProperties oqsSdkProperties,
            MetadataStorageRepository metadataStorageRepository,
            MessageWorkerManager messageWorkerManager,
            EventStream<SDKMetadataEvent> eventEventStream) {
        this.bocpClientSetting = bocpClientSetting;
        this.oqsSdkProperties = oqsSdkProperties;
        this.metadataStorageRepository = metadataStorageRepository;
        this.messageWorkerManager = messageWorkerManager;
        this.eventEventStream = eventEventStream;
    }

    @Override
    public void setBocpClient(IBocpClient bocpClient) {
        this.bocpClient = bocpClient;
    }

    @Override
    public CompletableFuture<Void> onTransferMessage(TransferMessage message) {
        Runnable messageWorker = () -> {
            log.debug("handle message : {}", message);
            if (VersionUtils.compare(message.getVersion(), MetadataContextHolder.currentVersion()) == 0 && !message.isForceUpdate()) {
                // 过期版本且非强制推送则丢弃消息
                log.debug("the message has the same version as the current version {}. discard it!", message.getVersion());
                return;
            }
            if (MessageType.CLIENT_CONNECT.equals(message.getMessageType())) {
                dealWithMessageTriggerByClient(message);
            } else if (MessageType.CLIENT_PULL.equals(message.getMessageType())) {
                //多节点情况下优化，随机延时，避免并发操作
                randomDelay();
                dealWithMessageTriggerByClient(message);

            } else if (MessageType.BOCP_DEPLOY.equals(message.getMessageType())) {
                //多节点情况下优化，随机延时，避免并发操作
                randomDelay();
                dealWithDeployMessage(message);
            } else if (MessageType.BOCP_DEPLOY_DDL.equals(message.getMessageType())) {
                dealWithDeployDdlMessage(message);
            } else {
                log.warn("unknown message type : {}, discard!", message.getMessageType());
            }
        };

        return messageWorkerManager.execute(messageWorker);
    }

    /**
     * 主动拉取消息（当连接上或者手动触发）
     *
     * @param message
     */
    private void dealWithMessageTriggerByClient(TransferMessage message) {
        try {
            if (!message.isHandleSuccess()) {
                log.warn(String.format("handle message failed, %s", message.getHandleMessage()));
                return;
            }
            log.info("begin pulling metadata for the specified app version {} {}", message.getVersion(), null == message.getDeployDetail() ? "" : message.getDeployDetail());
            SchemaApp schemaApp = prepareMetadata(message);
            publishMetadataForSdk(message, schemaApp);
        } catch (Throwable throwable) {
            log.error("the metadata retrieval failed when the client connected or attempted to pull!", throwable);
        }
    }

    /**
     * 处理版本部署消息（包括服务端部署和客户端主动拉两种消息）
     *
     * @param message
     */
    private void dealWithDeployMessage(TransferMessage message) {
        TransferMessage replyMessage = null;
        try {
            log.info("begin updating metadata for the specified app version {} {}", message.getVersion(), null == message.getDeployDetail() ? "" : message.getDeployDetail());

            SchemaApp schemaApp = prepareMetadata(message);

            publishMetadataForSdk(message, schemaApp);

            replyMessage = MessageUtils.buildReplyMessage(message, true, "版本部署成功");
        } catch (RuntimeException re) {
            log.error("the metadata update failed: {} ", re.getMessage());
            replyMessage = MessageUtils.buildReplyMessage(message, false, re.getMessage());
        } catch (Throwable throwable) {
            log.error("the metadata update failed!", throwable);
            replyMessage = MessageUtils.buildReplyMessage(message, false, ExceptionUtils.getStackTrace(throwable));
        } finally {
            bocpClient.sendMessage(JsonUtils.object2Json(replyMessage));
        }
    }

    private void dealWithDeployDdlMessage(TransferMessage message) {
        TransferMessage replyMessage = null;
        try {
            //TODO 如何执行DDL

            replyMessage = MessageUtils.buildReplyMessage(message, true, "DDL执行成功");
        } catch (RuntimeException re) {
            log.error("the metadata update failed: {} ", re.getMessage());
            replyMessage = MessageUtils.buildReplyMessage(message, false, re.getMessage());
        } catch (Throwable throwable) {
            log.error("the metadata update failed!", throwable);
            replyMessage = MessageUtils.buildReplyMessage(message, false, ExceptionUtils.getStackTrace(throwable));
        } finally {
//            bocpClient.sendMessage(JsonUtils.object2Json(replyMessage));
        }
    }


    /**
     * 收到应用版本信息后，准备元数据信息；如果获取的实体元数据为空，则等待5s，重新获取
     *
     * @param message
     * @throws InterruptedException
     */
    private SchemaApp prepareMetadata(TransferMessage message) throws InterruptedException {
        List<SchemaBo> entitys = null;
        int count = 0;
        List<TenantAppDeployInfo> tenantApps = Optional.ofNullable(message.getDeployDetail()).map(DeployDetail::getTenantApps).orElse(Lists.newArrayList());
        String version = message.getVersion();
        if (version == null) {
            AppInfo lastestAppInfo = metadataStorageRepository.getLastestAppInfo(message.getAppId());
            version = lastestAppInfo.getVersion();
            message.setVersion(version);
            message.setAppCodeForDB(lastestAppInfo.getAppCodeForDb());
            message.setAppId(lastestAppInfo.getAppId());
            message.setAppCode(lastestAppInfo.getAppCode());
        } else {
            metadataStorageRepository.saveAppInfo(message);
        }

        while (count < GET_METADATA_RETRY_TIMES) {
            entitys = metadataStorageRepository.getMetadatas(
                    Long.valueOf(oqsSdkProperties.getAuth().getAppId()),
                    version, SchemaMetadataType.ENTITY, tenantApps);
            if (null != entitys && !entitys.isEmpty()) {
                break;
            }
            Thread.sleep(GET_METADATA_WAIT_TIME);
            log.debug("retry to retrieve entity metadata");
            count++;
        }
        if (null == entitys || entitys.isEmpty()) {
            throw new RuntimeException("retry to retrieve entity metadata failed");
        }
        log.info(String.format("entity metadata received, num: %d", entitys.size()));
        List<SchemaDict> dicts = metadataStorageRepository.getMetadatas(Long.valueOf(oqsSdkProperties.getAuth().getAppId()), version, SchemaMetadataType.DICT, tenantApps);
        log.info(String.format("dict metadata received, num: %d", null == dicts ? 0 : dicts.size()));
        List<SchemaAction> actions = metadataStorageRepository.getMetadatas(Long.valueOf(oqsSdkProperties.getAuth().getAppId()), version, SchemaMetadataType.ACTION);
        log.info(String.format("action metadata received, num: %d", null == actions ? 0 : actions.size()));
        List<SchemaAppEvent> appEvents = metadataStorageRepository.getMetadatas(Long.valueOf(oqsSdkProperties.getAuth().getAppId()), version, SchemaMetadataType.APP_EVENT);
        log.info(String.format("appEvent metadata received, num: %d", null == appEvents ? 0 : appEvents.size()));

        List<SchemaSdkSetting> sdkSettings = new ArrayList<>();
        if (bocpClientSetting.getBocp().getEnableSDKConfig()) {
            sdkSettings = metadataStorageRepository.getMetadatas(Long.valueOf(oqsSdkProperties.getAuth().getAppId()), version, SchemaMetadataType.SDK_SETTING);
            log.info(String.format("sdkSetting metadata received, num: %d", null == sdkSettings ? 0 : sdkSettings.size()));
        } else {
            log.info("sdkSetting metadata disabled");
        }

        List<SchemaFlow> flows = metadataStorageRepository.getMetadatas(Long.valueOf(oqsSdkProperties.getAuth().getAppId()), version, SchemaMetadataType.FLOW, tenantApps);
        log.info(String.format("flow metadata received, num: %d", null == flows ? 0 : flows.size()));

        SchemaApp schemaApp = new SchemaApp();
        schemaApp.setBos(entitys);
        schemaApp.setDicts(dicts);
        schemaApp.setActions(actions);
        schemaApp.setAppEvents(appEvents);
        schemaApp.setSdkSettings(sdkSettings);
        schemaApp.setFlowSettings(flows);

        //其它可选项
        if (bocpClientSetting.getBocp().getExtraMetadataTypes().contains(SchemaMetadataType.PAGE.code())) {
            List<SchemaPage> pages = metadataStorageRepository.getMetadatas(
                    Long.valueOf(oqsSdkProperties.getAuth().getAppId()), version, SchemaMetadataType.PAGE, tenantApps);
            log.info(String.format("page metadata received, num: %d", null == pages ? 0 : pages.size()));
            schemaApp.setPages(pages);
        }
        if (bocpClientSetting.getBocp().getExtraMetadataTypes().contains(SchemaMetadataType.FORM.code())) {
            List<SchemaForm> forms = metadataStorageRepository.getMetadatas(
                    Long.valueOf(oqsSdkProperties.getAuth().getAppId()), version, SchemaMetadataType.FORM, tenantApps);
            log.info(String.format("form metadata received, num: %d", null == forms ? 0 : forms.size()));
            schemaApp.setForms(forms);
        }
        if (bocpClientSetting.getBocp().getExtraMetadataTypes().contains(SchemaMetadataType.PAGE_SETTING.code())) {
            List<SchemaPageSetting> pageSettings = metadataStorageRepository.getMetadatas(
                    Long.valueOf(oqsSdkProperties.getAuth().getAppId()), version, SchemaMetadataType.PAGE_SETTING, tenantApps);
            log.info(String.format("page setting metadata received, num: %d", null == pageSettings ? 0 : pageSettings.size()));
            schemaApp.setPageSettings(pageSettings);
        }
        return schemaApp;
    }

    /**
     * 包括更新元数据上下文信息以及发布元数据更新事件，供sdk捕获
     *
     * @param message
     * @param schemaApp
     */
    private void publishMetadataForSdk(TransferMessage message, SchemaApp schemaApp) {
        MetadataContextHolder.update(oqsSdkProperties.getAuth().getAppId(), message, schemaApp);
        SDKMetadataEvent sdkMetadataEvent = new SDKMetadataEvent(
                Long.valueOf(oqsSdkProperties.getAuth().getAppId()),
                Long.valueOf(oqsSdkProperties.getAuth().getEnv()),
                MetadataContextHolder.appCode(),
                MetadataContextHolder.appCodeForDB(),
                message.getVersion(),
                schemaApp.getBos(),
                schemaApp.getDicts(),
                schemaApp.getFlowSettings(),
                schemaApp.getActions(),
                schemaApp.getAppEvents(),
                schemaApp.getSdkSettings(),
                schemaApp.getPages(),
                schemaApp.getForms(),
                schemaApp.getPageSettings(),
                message.getMessageType());
        eventEventStream.offer(sdkMetadataEvent);
    }

    /**
     * 多节点情况下优化，随机延时，避免并发操作
     */
    private void randomDelay() {
        int RANDOM_RANGE = 1000;
        int randomSleep = (int) (Math.random() * RANDOM_RANGE);
        try {
            Thread.sleep(randomSleep);
        } catch (InterruptedException e) {
            log.error("sleep interrupted", e);
        }
    }
}
