package com.xforceplus.ultraman.transfer.storage.aliyunoss.api;

import com.aliyun.oss.OSS;
import com.aliyun.oss.common.utils.IOUtils;
import com.aliyun.oss.model.ListObjectsRequest;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.ObjectListing;
import com.xforceplus.ultraman.metadata.jsonschema.enums.SchemaMetadataType;
import com.xforceplus.ultraman.transfer.domain.dto.AppInfo;
import com.xforceplus.ultraman.transfer.domain.dto.MetadataEntityDTO;
import com.xforceplus.ultraman.transfer.domain.dto.MetadataStr;
import com.xforceplus.ultraman.transfer.storage.aliyunoss.constant.AliyunOSSConstant;
import com.xforceplus.ultraman.transfer.storage.api.IMetadataStorage;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.CharEncoding;
import org.apache.commons.lang3.StringUtils;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

/**
 * CopyRight: 上海云砺信息科技有限公司
 * User: youyifan
 * DateTime: 2023/4/20 17:53
 * Description:
 * History:
 */
@Slf4j
public class AliyunOssMetadataStorage implements IMetadataStorage {

    private final OSS ossClient;

    public AliyunOssMetadataStorage(OSS ossClient) {
        this.ossClient = ossClient;
    }

    @Override
    public boolean isPersistent() {
        return false;
    }

    @Override
    public AppInfo getLatestAppInfo(Long appId) {
        return null;
    }

    @Override
    public void saveMetadata(MetadataEntityDTO dto, String jsonStr) {
        if (StringUtils.isBlank(dto.getTenantCode())) {
            saveMetadata(dto.getAppId(), dto.getVersion(), dto.getMetadataType(), jsonStr);
        } else {
            String filePath = AliyunOSSConstant.getTenantMetadataFilePath(
                    dto.getAppId(), dto.getTenantCode(), dto.getVersion(), dto.getMetadataType());
            ossClient.putObject(AliyunOSSConstant.BUCKET_NAME, filePath, new ByteArrayInputStream(jsonStr.getBytes(StandardCharsets.UTF_8)));
        }
    }

    @Override
    public void saveMetadata(MetadataEntityDTO dto, Map<String, String> jsonStrMap) {
        if (StringUtils.isBlank(dto.getTenantCode())) {
            saveMetadata(dto.getAppId(), dto.getVersion(), dto.getMetadataType(), jsonStrMap);
        } else {
            Optional.ofNullable(jsonStrMap).ifPresent(map ->
                    map.keySet().forEach(key -> {
                        String filePath = AliyunOSSConstant.getTenantMetadatasFilePath(
                                dto.getAppId(), dto.getTenantCode(), dto.getVersion(), dto.getMetadataType(), key);
                        ossClient.putObject(AliyunOSSConstant.BUCKET_NAME, filePath, new ByteArrayInputStream(jsonStrMap.get(key).getBytes(StandardCharsets.UTF_8)));
                    }));
        }
    }

    @Override
    public void saveMetadata(Long appId, String version, SchemaMetadataType metadataType, Map<String, String> jsonStrMap) {
        Optional.ofNullable(jsonStrMap).ifPresent(map ->
                map.keySet().forEach(key -> {
                    String filePath = AliyunOSSConstant.getMetadatasFilePath(appId, version, metadataType, key);
                    ossClient.putObject(AliyunOSSConstant.BUCKET_NAME, filePath, new ByteArrayInputStream(jsonStrMap.get(key).getBytes(StandardCharsets.UTF_8)));
                }));
    }

    @Override
    public void saveMetadata(Long appId, String version, SchemaMetadataType metadataType, String jsonStr) {
        String filePath = AliyunOSSConstant.getMetadataFilePath(appId, version, metadataType);
        ossClient.putObject(AliyunOSSConstant.BUCKET_NAME, filePath, new ByteArrayInputStream(jsonStr.getBytes(StandardCharsets.UTF_8)));
    }

    @Override
    public MetadataStr loadMetadataStr(Long appId, String version, SchemaMetadataType metadataType) {
        String filePath = AliyunOSSConstant.getMetadataFilePath(appId, version, metadataType);
        return getDataStrFromOSS(filePath);
    }

    @Override
    public List<MetadataStr> loadMetadataStrs(Long appId, String version, SchemaMetadataType metadataType) {
        Map<String, String> contentMap = loadMetadataMap(appId, version, metadataType);
        return contentMap.entrySet().stream().map(x -> {
            MetadataStr metadataStr = new MetadataStr();
            metadataStr.setMetadataId(x.getKey());
            metadataStr.setJsonContent(x.getValue());
            return metadataStr;
        }).collect(Collectors.toList());
    }

    public Map<String, String> loadMetadataMap(Long appId, String version, SchemaMetadataType metadataType) {
        String filePathDir = AliyunOSSConstant.getMetadatasDirPath(appId, version, metadataType);
        return getDataMapFromOSS(filePathDir);
    }

    @Override
    public MetadataStr loadMetadataStr(Long stdAppId, Long tenantAppId, String tenantCode, String tenantAppVersion, SchemaMetadataType metadataType) {
        String filePath = AliyunOSSConstant.getTenantMetadataFilePath(stdAppId, tenantCode, tenantAppVersion, metadataType);
        return getDataStrFromOSS(filePath);
    }

    @Override
    public List<MetadataStr> loadMetadataStrs(Long stdAppId, Long tenantAppId, String tenantCode, String tenantAppVersion, SchemaMetadataType metadataType) {
        Map<String, String> contentMap = loadMetadataMap(stdAppId, tenantAppId, tenantCode, tenantAppVersion, metadataType);

        return contentMap.entrySet().stream().map(x -> {
                    MetadataStr metadataStr = new MetadataStr();
                    metadataStr.setMetadataId(x.getKey());
                    metadataStr.setJsonContent(x.getValue());
                    return metadataStr;
                }).collect(Collectors.toList());
    }

    public Map<String, String> loadMetadataMap(Long stdAppId, Long tenantappId, String tenantCode, String tenantAppVersion, SchemaMetadataType metadataType) {
        String filePathDir = AliyunOSSConstant.getTenantMetadatasDirPath(stdAppId, tenantCode, tenantAppVersion, metadataType);
        return getDataMapFromOSS(filePathDir);
    }

    private MetadataStr getDataStrFromOSS(String filePath) {
        try {
            if(!ossClient.doesObjectExist(AliyunOSSConstant.BUCKET_NAME, filePath)) {
                log.debug("filePath {} not found in oss", filePath);
                return null;
            }
            OSSObject ossObject = ossClient.getObject(AliyunOSSConstant.BUCKET_NAME, filePath);
            String jsonContent = IOUtils.readStreamAsString(ossObject.getObjectContent(), CharEncoding.UTF_8);
            MetadataStr metadataStr = new MetadataStr();
            metadataStr.setJsonContent(jsonContent);
            metadataStr.setMetadataId(null);
            return metadataStr;
        } catch (Exception e) {
            log.error("could not load tenant metadata", e);
        }
        return null;
    }

    private Map<String, String> getDataMapFromOSS(String filePathDir) {
        Map<String, String> contentMap = new HashMap<>();
        try {
            // 列举文件。
            ListObjectsRequest request = new ListObjectsRequest();
            request.setBucketName(AliyunOSSConstant.BUCKET_NAME);
            request.setPrefix(filePathDir);
            request.setMaxKeys(AliyunOSSConstant.MAX_RETURNED_KEYS_LIMIT);
            ObjectListing objectListing = ossClient.listObjects(request);
            List<OSSObjectSummary> sums = objectListing.getObjectSummaries();
            for (OSSObjectSummary s : sums) {
                // 获取文件。
                OSSObject ossObject = ossClient.getObject(AliyunOSSConstant.BUCKET_NAME, s.getKey());
                // 读取文件内容。
                contentMap.put(AliyunOSSConstant.getFileName(s.getKey()), IOUtils.readStreamAsString(ossObject.getObjectContent(), CharEncoding.UTF_8));
            }
        } catch (Exception e) {
            log.error("could not load metadata", e);
        }
        return contentMap;
    }
}
