package com.xforceplus.ultraman.transfer.storage.aliyunoss.api;

import com.aliyun.oss.OSS;
import com.aliyun.oss.common.utils.IOUtils;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.ObjectListing;
import com.xforceplus.ultraman.metadata.jsonschema.enums.SchemaMetadataType;
import com.xforceplus.ultraman.transfer.storage.aliyunoss.constant.AliyunOSSConstant;
import com.xforceplus.ultraman.transfer.storage.api.IMetadataStorage;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.CharEncoding;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

/**
 * CopyRight: 上海云砺信息科技有限公司
 * User: youyifan
 * DateTime: 2023/4/20 17:53
 * Description:
 * History:
 */
@Slf4j
public class AliyunOssMetadataStorage implements IMetadataStorage {

    private OSS ossClient;

    public AliyunOssMetadataStorage(OSS ossClient) {
        this.ossClient = ossClient;
    }

    @Override
    public void saveMetadata(Long appId, String version, SchemaMetadataType metadataType, Map<String, String> jsonStrMap) {
        Optional.ofNullable(jsonStrMap).ifPresent(map ->
                        map.keySet().forEach(key -> {
                            String filePath = AliyunOSSConstant.getMetadatasFilePath(appId, version, metadataType, key);
                            ossClient.putObject(AliyunOSSConstant.BUCKET_NAME, filePath, new ByteArrayInputStream(jsonStrMap.get(key).getBytes(StandardCharsets.UTF_8)));
                        }));
    }

    @Override
    public void saveMetadata(Long appId, String version, SchemaMetadataType metadataType, String jsonStr) {
        String filePath = AliyunOSSConstant.getMetadataFilePath(appId, version, metadataType);
        ossClient.putObject(AliyunOSSConstant.BUCKET_NAME, filePath, new ByteArrayInputStream(jsonStr.getBytes(StandardCharsets.UTF_8)));
    }

    @Override
    public String loadMetadataStr(Long appId, String version, SchemaMetadataType metadataType) {
        try {
            String filePath = AliyunOSSConstant.getMetadataFilePath(appId, version, metadataType);
            if(!ossClient.doesObjectExist(AliyunOSSConstant.BUCKET_NAME, filePath)) {
                log.debug("app {} version {} metadata {} not found in oss", appId, version, metadataType.code());
                return null;
            }
            OSSObject ossObject = ossClient.getObject(AliyunOSSConstant.BUCKET_NAME, filePath);
            return IOUtils.readStreamAsString(ossObject.getObjectContent(), CharEncoding.UTF_8);
        } catch (Exception e) {
            log.error("could not load metadata", e);
        }
        return null;
    }

    @Override
    public List<String> loadMetadataStrs(Long appId, String version, SchemaMetadataType metadataType) {
        Map<String, String> contentMap = loadMetadataMap(appId, version, metadataType);
        return contentMap.values().stream().collect(Collectors.toList());
    }

    @Override
    public Map<String, String> loadMetadataMap(Long appId, String version, SchemaMetadataType metadataType) {
        Map<String, String> contentMap = new HashMap<>();
        try {
            String filePathDir = AliyunOSSConstant.getMetadatasDirPath(appId, version, metadataType);
            // 列举文件。
            ObjectListing objectListing = ossClient.listObjects(AliyunOSSConstant.BUCKET_NAME, filePathDir);
            List<OSSObjectSummary> sums = objectListing.getObjectSummaries();
            for (OSSObjectSummary s : sums) {
                // 获取文件。
                OSSObject ossObject = ossClient.getObject(AliyunOSSConstant.BUCKET_NAME, s.getKey());
                // 读取文件内容。
                contentMap.put(AliyunOSSConstant.getFileName(s.getKey()), IOUtils.readStreamAsString(ossObject.getObjectContent(), CharEncoding.UTF_8));
            }

        } catch (Exception e) {
            log.error("could not load metadata", e);
        }
        return contentMap;
    }
}
