package com.xforceplus.ultraman.transfer.storage.http.api;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.xforceplus.ultraman.transfer.common.util.JsonUtils;
import com.xforceplus.ultraman.transfer.common.util.MessageUtils;
import com.xforceplus.ultraman.metadata.jsonschema.enums.SchemaMetadataType;
import com.xforceplus.ultraman.transfer.domain.dto.AppInfo;
import com.xforceplus.ultraman.transfer.domain.dto.MetadataStr;
import com.xforceplus.ultraman.transfer.storage.api.IMetadataStorage;
import com.xforceplus.ultraman.transfer.storage.http.BocpOkHttpClient;
import com.xforceplus.ultraman.transfer.storage.http.dto.CustomResponse;
import com.xforceplus.ultraman.transfer.storage.http.interceptor.MetadataHeadInterceptor;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * CopyRight: 上海云砺信息科技有限公司
 * User: youyifan
 * DateTime: 2023/5/15 13:33
 * Description:
 * History:
 */
@Slf4j
public class HttpMetadataStorage extends BocpOkHttpClient implements IMetadataStorage {

    public final String GET_METADATA_PATH = "%smetadatas/versions/%s";
    private String bocpHost;
    private Boolean useSsl;
    private String apiPrefix;
    private final OkHttpClient client;
    
    public HttpMetadataStorage(Boolean useSSl) {
        this.useSsl = useSSl;
        this.client = buildClient(useSsl, Collections.emptyList());
    }
    
    public HttpMetadataStorage(
            String bocpHost,
            Boolean useSsl,
            String apiPrefix,
            MetadataHeadInterceptor metadataHeadInterceptor) {
        this.bocpHost = bocpHost;
        this.useSsl = useSsl;
        this.apiPrefix = apiPrefix;
        this.client = buildClient(useSsl, Arrays.asList(metadataHeadInterceptor));
    }

    @Override
    public boolean isPersistent() {
        return false;
    }

    @Override
    public AppInfo getLatestAppInfo(Long appId) {
        return null;
    }

    @Override
    public void saveMetadata(Long appId, String version, SchemaMetadataType metadataType, String jsonStr) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void saveMetadata(Long appId, String version, SchemaMetadataType metadataType, Map<String, String> jsonStrs) {
        throw new UnsupportedOperationException();
    }

    @Override
    public MetadataStr loadMetadataStr(Long appId, String version, SchemaMetadataType metadataType) {
        String result =  requestMetadata(appId, version, metadataType);
        log.info("调用获取元数据接口成功 {}", metadataType.code());
        MetadataStr metadataStr = new MetadataStr();
        metadataStr.setJsonContent(result);
        return metadataStr;
    }

    /**
     * http请求到的flows列表是在一个json里面，不需要该方法
     * @param appId
     * @param version
     * @param metadataType
     * @return
     */
    @Override
    public List<MetadataStr> loadMetadataStrs(Long appId, String version, SchemaMetadataType metadataType) {
        MetadataStr metadataStr = this.loadMetadataStr(appId, version, metadataType);
        if(metadataStr == null) {
            return Collections.emptyList();
        }
        JsonNode jsonNode = JsonUtils.readTree(metadataStr.getJsonContent());
        if(jsonNode.isArray()) {
            List<MetadataStr> metadataStrs = new ArrayList<>();
            jsonNode.forEach(x -> {
                MetadataStr metadata = new MetadataStr();
                if(x instanceof ObjectNode) {
                    JsonNode idNode = x.get("id");
                    if (idNode != null) {
                        metadata.setMetadataId(idNode.asText());
                    }
                }
                metadata.setJsonContent(x.toString());
                metadataStrs.add(metadata);
            });
            return metadataStrs;
        } else {
           return Collections.singletonList(metadataStr);
        }
    }

    @Override
    public MetadataStr loadMetadataStr(Long stdAppId, Long tenantAppId, String tenantCode, String tenantAppVersion
            , SchemaMetadataType metadataType) {
        String result = requestMetadata(tenantAppId, tenantAppVersion, metadataType);
        log.info("调用获取元数据接口成功 {} {}", metadataType.code(), Optional.ofNullable(tenantCode).orElse(""));
        MetadataStr metadataStr = new MetadataStr();
        metadataStr.setJsonContent(result);
        return metadataStr;
    }

    @Override
    public List<MetadataStr> loadMetadataStrs(Long stdAppId, Long tenantAppId, String tenantCode, String tenantAppVersion, SchemaMetadataType metadataType) {
        MetadataStr metadataStr = this.loadMetadataStr(stdAppId, tenantAppId, tenantCode, tenantAppVersion, metadataType);
        
        if(metadataStr == null) {
            return Collections.emptyList();
        }
        
        JsonNode jsonNode = JsonUtils.readTree(metadataStr.getJsonContent());
        if(jsonNode.isArray()) {
            List<MetadataStr> metadataStrs = new ArrayList<>();
            jsonNode.forEach(x -> {
                MetadataStr metadata = new MetadataStr();
                metadata.setMetadataId(null);
                metadata.setJsonContent(x.toString());
                metadataStrs.add(metadata);
            });
            return metadataStrs;
        } else {
            return Collections.singletonList(metadataStr);
        }
    }

    protected String requestMetadata(Long appId, String version, SchemaMetadataType metadataType) {
        Request request = buildRequest(appId, version, metadataType);
        try (Response resp = client.newCall(request).execute()) {
            if (!resp.isSuccessful()) {
                throw new RuntimeException("调用获取元数据接口失败, msg = " + resp.message());
            }
            CustomResponse response = JsonUtils.json2Object(resp.body().string(), CustomResponse.class);
            if (isFail(response)) {
                throw new RuntimeException("调用获取元数据接口失败, msg = " + resp.message());
            }
            return response.getData();
        } catch (Exception e) {
//            log.error("调用获取元数据接口失败", e);
            throw new RuntimeException(e);
        }
    }

    protected Request buildRequest(Long appId, String version, SchemaMetadataType metadataType) {
        String realApiPrefix = StringUtils.isBlank(apiPrefix) ? "" : apiPrefix + "/";
        String realApiPath = String.format(GET_METADATA_PATH, realApiPrefix, version);
        String url = MessageUtils.getUrl(bocpHost, useSsl);
        HttpUrl hostUrl = HttpUrl.get(url);

        HttpUrl httpUrl = new HttpUrl.Builder()
                .scheme(MessageUtils.getSchema(useSsl))
                .host(hostUrl.host())
                .port(hostUrl.port())
                .addPathSegments(realApiPath)  // 添加路径段，如果有多个路径段，可以多次调用这个方法
                .addQueryParameter("appId", appId.toString())  // 添加查询参数
                .addQueryParameter("metadataType", metadataType.code())// 添加查询参数
                .build();
        return new Request.Builder()
                .url(httpUrl)
                .build();
    }

}
