package com.xforceplus.ultraman.cdc;

import com.alibaba.otter.canal.common.utils.NamedThreadFactory;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.cdc.core.local.EmbedCDCServer;
import com.xforceplus.ultraman.cdc.core.remote.DispatcherCDCServer;
import com.xforceplus.ultraman.cdc.dto.enums.CDCStatus;
import com.xforceplus.ultraman.cdc.lock.CDCDestinationLock;
import com.xforceplus.ultraman.cdc.processor.DataProcessor;
import com.xforceplus.ultraman.cdc.reader.CDCPropertyPackage;
import com.xforceplus.ultraman.cdc.reader.CDCPropertyPackageInternal;
import com.xforceplus.ultraman.cdc.reader.CanalPropertiesReader;
import com.xforceplus.ultraman.cdc.utils.TimeWaitUtils;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.*;

/**
 * Created by justin.xu on 06/2023.
 *
 * @since 1.8
 */
public class CDCStarter implements CDCLifeCycle {

    private static final Logger logger = LoggerFactory.getLogger(CDCStarter.class);

    private CDCServer cdcServer;

    private volatile boolean isShutdown = false;

    private ScheduledExecutorService executor = null;

    private CDCDestinationLock lock = null;

    public CDCStarter(String key, CDCPropertyPackage cdcPropertyPackage, DataProcessor dataProcessor, CDCDestinationLock lock) {
        this.executor = Executors.newScheduledThreadPool(cdcPropertyPackage.readers().size(),
                new NamedThreadFactory("cdc-server-metrics"));

        if (cdcPropertyPackage.model().equals(CDCPropertyPackageInternal.Model.REMOTE)) {

            if (lock != null) {
                lock.run(key, () -> {
                    cdcServer = new DispatcherCDCServer(cdcPropertyPackage, dataProcessor);
                },  () -> {
                    cdcServer = new DispatcherCDCServer(cdcPropertyPackage, dataProcessor);
                    this.init();
                });
            } else {
                cdcServer = new DispatcherCDCServer(cdcPropertyPackage, dataProcessor);
            }
        } else if (cdcPropertyPackage.model().equals(CDCPropertyPackageInternal.Model.LOCAL)) {
            if (lock != null) {
                lock.run(key, () -> {
                    cdcServer = new EmbedCDCServer(cdcPropertyPackage, dataProcessor);
                },  () -> {
                    cdcServer = new EmbedCDCServer(cdcPropertyPackage, dataProcessor);
                    this.init();
                });
            } else {
                cdcServer = new EmbedCDCServer(cdcPropertyPackage, dataProcessor);
            }
        }
    }

    @PostConstruct
    @Override
    public void init() {
        if (cdcServer != null) {
            cdcServer.init();
            isShutdown = false;
            cdcServer.execute();
            executor.submit(this::metrics);
        }
    }

    @PreDestroy
    @Override
    public void destroy() {
        if (cdcServer != null) {
            cdcServer.destroy();
            isShutdown = true;
            executor.shutdown();
        }
    }

    public CDCServer cdcServer() {
        return cdcServer;
    }


    private void metrics() {
        while (!isShutdown) {

            Map<String, CDCStatus> cdcStatusMap =
                    cdcServer.metrics();

            if (null != cdcStatusMap) {
                try {
                    logger.info("cdc status : {}", JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(cdcStatusMap));
                } catch (JsonProcessingException e) {
                    throw new RuntimeException(e);
                }

                TimeWaitUtils.wakeupAfter(30, TimeUnit.SECONDS);
            }
        }
    }

}
