package com.xforceplus.ultraman.cdc.core.remote.connect;

import static com.xforceplus.ultraman.cdc.dto.constant.CDCConstant.DEFAULT_BATCH_SIZE;

import com.alibaba.otter.canal.client.CanalConnector;
import com.alibaba.otter.canal.client.CanalConnectors;
import com.alibaba.otter.canal.protocol.Message;
import com.xforceplus.ultraman.cdc.reader.CanalPropertiesReader;
import java.net.InetSocketAddress;
import java.sql.SQLException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 非集群模式连接器.
 *
 * @author xujia 2020/11/5
 * @since : 1.8
 */
public class RemoteCDCConnector implements CDCConnector {

    private Logger logger = LoggerFactory.getLogger(RemoteCDCConnector.class);

    protected CanalConnector canalConnector;

    //  canal的基本信息
    protected CanalPropertiesReader canalPropertiesReader;

    private int batchSize = DEFAULT_BATCH_SIZE;

    private boolean isClosed = true;

    public RemoteCDCConnector(CanalPropertiesReader canalPropertiesReader) {

        this.canalPropertiesReader = canalPropertiesReader;
        if (null != canalPropertiesReader && canalPropertiesReader.getBatchSize() > 0) {
            this.batchSize = batchSize;
        }
    }

    @Override
    public void init() {
        if (null == canalPropertiesReader) {
            throw new RuntimeException("canal properties could not be null.");
        }

        canalConnector = CanalConnectors.newSingleConnector(new InetSocketAddress(canalPropertiesReader.getMasterHost(),
                canalPropertiesReader.getMasterPort()), canalPropertiesReader.getDestination(),
            canalPropertiesReader.getMasterUser(), canalPropertiesReader.getMasterPasswd());
    }

    @Override
    public void destroy() {
        try {
            if (canalConnector.checkValid()) {
                close();
            }
        } catch (Exception e) {
            logger.warn("[cdc-connector {}] shutdown error, message : {}", canalPropertiesReader.getDestination(), e.getMessage());
        }
    }

    @Override
    public void open() {
        if (null != canalConnector) {
            //  连接CanalServer
            canalConnector.connect();
            //  订阅destination
            canalConnector.subscribe();
            logger.info("[cdc-connector {}] connect to canal server...", canalPropertiesReader.getDestination());
            isClosed = false;
        }
    }

    @Override
    public void close() {
        if (null != canalConnector && !isClosed) {
            try {

                //  关闭连接CanalServer
                canalConnector.disconnect();
                logger.info("[cdc-connector {}] close canal connector...", canalPropertiesReader.getDestination());
            } catch (Exception e) {
                logger.error("[cdc-connector {}] close error, ex : {}", canalPropertiesReader.getDestination(), e.getMessage());
            } finally {
                isClosed = true;
            }
        }
    }

    @Override
    public void rollback() {
        if (null != canalConnector) {
            canalConnector.rollback();
        }
    }

    @Override
    public void ack(long batchId) throws SQLException {
        if (null == canalConnector) {
            notInitException(canalPropertiesReader.getDestination());
        }
        canalConnector.ack(batchId);
    }

    @Override
    public Message getMessageWithoutAck() throws SQLException {
        if (null == canalConnector) {
            notInitException(canalPropertiesReader.getDestination());
        }
        //  获取数据
        return canalConnector.getWithoutAck(batchSize);
    }

    @Override
    public int getBatchSize() {
        return batchSize;
    }

    public String name() {
        return canalPropertiesReader.getDestination();
    }

    private void notInitException(String destination) throws SQLException {
        throw new SQLException("[cdc-connector {}] canal connector not init.", destination);
    }
}
