package com.xforceplus.ultraman.cdc.processor.impl;


import static com.xforceplus.ultraman.metadata.helper.OriginEntityUtils.attributesToMap;
import static com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment.DEL_UID_KEY;
import static com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment.DEL_UNAME_KEY;
import static com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment.GROUP_KEY;
import static com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment.ROOT;

import com.alibaba.otter.canal.protocol.CanalEntry;
import com.alibaba.otter.canal.protocol.Message;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.cdc.adapter.CDCBeforeCallback;
import com.xforceplus.ultraman.cdc.adapter.EngineAdapterService;
import com.xforceplus.ultraman.cdc.context.ParserContext;
import com.xforceplus.ultraman.cdc.dto.ParseResult;
import com.xforceplus.ultraman.cdc.dto.constant.CDCConstant;
import com.xforceplus.ultraman.cdc.processor.DataProcessor;
import com.xforceplus.ultraman.cdc.processor.EventQueue;
import com.xforceplus.ultraman.cdc.utils.BinLogParseUtils;
import com.xforceplus.ultraman.extensions.cdc.status.StatusService;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.helper.OriginEntityUtils;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.sdk.core.event.EntityCreated;
import com.xforceplus.ultraman.sdk.core.event.EntityDeleted;
import com.xforceplus.ultraman.sdk.core.event.EntityUpdated;
import com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment;
import com.xforceplus.ultraman.sdk.infra.event.EventPublisher;
import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import io.micrometer.core.annotation.Timed;

import java.sql.SQLException;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Created by justin.xu on 08/2022.
 *
 * @since 1.8
 */
@Slf4j
public class DefaultDataProcessor implements DataProcessor {

    private static long time_total = 0;
    final Logger logger = LoggerFactory.getLogger(DefaultDataProcessor.class);
    @Resource
    private EntityClassEngine engine;
    @Autowired(required = false)
    private StatusService statusService;
    @Autowired(required = false)
    private List<CDCBeforeCallback> cdcBeforeCallbacks = new ArrayList<>();
    @Resource
    private EventPublisher publisher;
    @Resource
    private EngineAdapterService engineAdapterService;
    @Resource
    private EventQueue eventQueue;
    @Resource
    private ExecutorService eventThreadPool;
    @Resource
    private ExecutorService workThreadPool;

    public ExecutorService getEventThreadPool() {
        return eventThreadPool;
    }

    public void setEventThreadPool(ExecutorService eventThreadPool) {
        this.eventThreadPool = eventThreadPool;
    }

    public void setEngine(EntityClassEngine engine) {
        this.engine = engine;
    }

    public void setStatusService(StatusService statusService) {
        this.statusService = statusService;
    }

    public void setCdcBeforeCallbacks(List<CDCBeforeCallback> cdcBeforeCallbacks) {
        this.cdcBeforeCallbacks = cdcBeforeCallbacks;
    }

    public void setPublisher(EventPublisher publisher) {
        this.publisher = publisher;
    }

    public void setEngineAdapterService(EngineAdapterService engineAdapterService) {
        this.engineAdapterService = engineAdapterService;
    }

    public void setEventQueue(EventQueue eventQueue) {
        this.eventQueue = eventQueue;
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"initiator", "cdc", "action", "consume"}
    )
    @Override
    public boolean onProcess(Message message) throws Exception {
        return messageProcess(message) > 0;
    }

    private int messageProcess(Message message) {
        try {
            if (message.getEntries().size() == 0) {
                if(statusService != null) {
                    statusService.onEmpty();
                }
                return 0;
            }
            final AtomicInteger handleMessageEventCount = new AtomicInteger(0);
            //  设置异步处理写事件
            workThreadPool.submit(() -> handleMessageEvent(message));
            //  设置异步处理写索引
            long start = System.currentTimeMillis();
            handleMessageEventCount.set(handleWriteIndex(message));
            long endTime = System.currentTimeMillis() - start;
            time_total += endTime;
            log.info("计算单位(毫秒) Message当前拉取批次大小:{},CDC处理总消耗时间:{},CDC当时处理批次消耗时间:{}", message.getEntries().size(), time_total, endTime);

            return handleMessageEventCount.get();

        } catch (Exception e) {
            logger.error("batchId : {}, consume message failed, message : {}", message.getId(), e.getMessage());
            throw e;
        }
    }

    private boolean handleMessageEvent(Message message) {
        ParserContext parserContext = new ParserContext(message.getId());
        parseCanalEntries(message.getEntries(), parserContext, true);
        return true;
    }

    private int handleWriteIndex(Message message) {
        ParserContext parserContext = new ParserContext(message.getId());
        ParseResult result = parseCanalEntries(message.getEntries(), parserContext, false);
        mutateEntries(result);
        Map<String, Map<Long, OqsEngineEntity>> batches = new LinkedHashMap<>();
        result.getFinishEntries().forEach((k, v) -> {
            batches.computeIfAbsent(v.getTable(), k1 -> new LinkedHashMap<>()).put(v.getId(), v);
        });
        AtomicInteger atomicInteger = new AtomicInteger(0);
        batches.forEach(
                (k, v) -> {
                    boolean r = engineAdapterService.batchUpsertOperation(v.values());
                    if (statusService != null) {
                        statusService.clearStatus(v.values());
                    }
                    atomicInteger.addAndGet(v.size());
                    if (!r) {
                        throw new RuntimeException("Adapter Service return False");
                    }
                }
        );

        return atomicInteger.get();
    }


    /**
     * 解析入口函数,对一个批次进行解析.
     *
     * @param entries 完整的批次信息.
     * @return 成功条数.
     */
    private ParseResult parseCanalEntries(List<CanalEntry.Entry> entries, ParserContext parserContext, boolean handleEvent) {
        ParseResult parseResult = new ParseResult();
        for (CanalEntry.Entry entry : entries) {
            //  不是TransactionEnd/RowData类型数据, 将被过滤
            switch (entry.getEntryType()) {
                case TRANSACTIONBEGIN:
                case TRANSACTIONEND:
                    break;
                case ROWDATA:
                    rowDataParse(entry, parserContext, parseResult, handleEvent);
                    break;
                default: {
                }
            }
        }

        return parseResult;
    }

    private void mutateEntries(ParseResult parseResult) {

        //  批次数据整理完毕，开始执行index写操作。
        if (!parseResult.getFinishEntries().isEmpty()) {
            Optional.ofNullable(cdcBeforeCallbacks)
                    .orElseGet(Collections::emptyList)
                    .stream().sorted(Comparator.comparingInt(CDCBeforeCallback::getOrder))
                    .forEach(x -> {
                        try {
                            x.mutate(toMutateEntities(parseResult.getFinishEntries()));
                        } catch (Throwable throwable) {
                            logger.error("CDC callback ERROR name:{} , ex:{}", x.name(), throwable);
                        }
                    });
        }
    }

    private OqsEngineEntity cloneEntity(OqsEngineEntity src) {
        OqsEngineEntity target = new OqsEngineEntity();
        target.setEntityClassRef(src.getEntityClassRef());
        target.setId(src.getId());
        target.setVersion(src.getVersion());

        Map<String, Object> attributes = src.getAttributes();
        try {
            String attr = JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(attributes);
            Map<String, Object> newAttr = JacksonDefaultMapper.OBJECT_MAPPER.readValue(attr, Map.class);
            target.setAttributes(newAttr);
        } catch (Throwable throwable) {
            logger.error("{}", throwable);
        }

        target.setUpdateTime(src.getUpdateTime());
        target.setFather(src.getFather());
        target.setDeleted(src.isDeleted());
        return target;
    }

    /**
     * 对rowData进行解析，rowData为对一张表的CUD操作，记录条数1～N.
     *
     * @param entry         canal对象同步实例.
     * @param parserContext 上下文.
     */
    private void rowDataParse(CanalEntry.Entry entry, ParserContext parserContext,
                              ParseResult parseResult, boolean handleEvent) {

        String tableName = entry.getHeader().getTableName();
        if (tableName.isEmpty()) {
            logger.error("batch : {}, table name could not be Null, [{}]",
                    parserContext.getBatchId(), entry.getStoreValue());
            return;
        }
        Optional<IEntityClass> entityClassOp = foundEntityClassFromTableName(tableName);
        if (!entityClassOp.isPresent()) {
            logger.error("batch : {}, tableName : {}, entityClass could not be Null.",
                    parserContext.getBatchId(), tableName);
            return;
        }

        CanalEntry.RowChange rowChange = null;
        try {
            rowChange = CanalEntry.RowChange.parseFrom(entry.getStoreValue());
        } catch (Exception e) {
            logger.error("batch : {}, parse entry value failed, [{}], [{}]",
                    parserContext.getBatchId(), entry.getStoreValue(), e.getMessage());
            return;
        }

        CanalEntry.EventType eventType = rowChange.getEventType();
        //  遍历RowData
        for (CanalEntry.RowData rowData : rowChange.getRowDatasList()) {

            //  获取一条完整的更新前的columns
            Map<String, CanalEntry.Column> beforeColumnsMap = new HashMap();
            rowData.getBeforeColumnsList().forEach(column -> beforeColumnsMap.put(column.getName(), column));

            //  获取一条完整的更新后的columns
            Map<String, CanalEntry.Column> afterColumnsMap = new HashMap();
            rowData.getAfterColumnsList().forEach(column -> afterColumnsMap.put(column.getName(), column));

            OqsEngineEntity beforeEntity = null;
            OqsEngineEntity afterEntity = null;

            SystemAttachment systemAttachment = null;
            try {
                switch (eventType) {
                    case INSERT: {
                        afterEntity = oneRowParser(afterColumnsMap, false, entityClassOp.get(), parseResult);
                        if (handleEvent) {
                            systemAttachment = getAttachment(afterColumnsMap);
                        }
                        mergeEntityToResult(parseResult, afterEntity);
                        break;
                    }
                    case UPDATE: {
                        afterEntity = oneRowParser(afterColumnsMap, false, entityClassOp.get(), parseResult);
                        mergeEntityToResult(parseResult, afterEntity);
                        if (handleEvent) {
                            try {
                                beforeEntity = oneRowParser(beforeColumnsMap, false, entityClassOp.get(), parseResult);
                            } catch (Exception e) {
                                logger.warn(
                                        "beforeEntity in update is null, will be ignore, but it will influence event");
                            }

                            systemAttachment = getAttachment(afterColumnsMap);
                        }
                        break;
                    }
                    case DELETE: {
                        beforeEntity = oneRowParser(beforeColumnsMap, true, entityClassOp.get(), parseResult);
                        if (handleEvent) {
                            systemAttachment = getAttachment(beforeColumnsMap);
                        }

                        mergeEntityToResult(parseResult, beforeEntity);
                        break;
                    }
                }

                //  处理事件
                if (handleEvent) {
                    OqsEngineEntity beforeClone = null;
                    OqsEngineEntity afterClone = null;

                    if (beforeEntity != null) {
                        beforeClone = cloneEntity(beforeEntity);
                    }

                    if (afterEntity != null) {
                        afterClone = cloneEntity(afterEntity);
                    }

                    if (systemAttachment == null) {
                        systemAttachment = new SystemAttachment();
                    }
                    eventHandler(eventType, systemAttachment, entityClassOp.get(), beforeClone, afterClone);
                }
            } catch (Exception e) {
                //  对于数据层面的错误，只记录，不回滚批次.
                logger.warn("parse entity error, message : {}", e.getMessage());
            }
        }
    }

    private SystemAttachment getAttachment(Map<String, CanalEntry.Column> columns) {
        String attrStr = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.DYNAMIC_FIELD);
        SystemAttachment attachment = new SystemAttachment();
        try {
            if (attrStr.isEmpty()) {
                return attachment;
            } else {
                Map<String, Object> map = attributesToMap(attrStr);
                Object o = map.get(ROOT);
                if (o != null) {
                    Map<String, Object> attachmentMap = (Map<String, Object>) o;
                    Object grouped = attachmentMap.get(GROUP_KEY);
                    if (grouped != null) {
                        attachment.setGrouped(Integer.parseInt(grouped.toString()));
                    }

                    Object delUid = attachmentMap.get(DEL_UID_KEY);
                    if (delUid != null) {
                        attachment.setDelUId(Long.parseLong(delUid.toString()));
                    }

                    Object delUname = attachmentMap.get(DEL_UNAME_KEY);
                    if (delUname != null) {
                        attachment.setDelUname(delUname.toString());
                    }
                }
                return attachment;
            }
        } catch (Throwable throwable) {
            return attachment;
        }
    }

    private Optional<IEntityClass> foundEntityClassFromTableName(String tableName) {
        String[] tableSplit = tableName.split("_");
        if (tableSplit.length < 3) {
            return Optional.empty();
        }

        String code = tableSplit[2];
        String profile = null;
        if (tableSplit.length > 3) {
            profile = tableSplit[3];
        }

        return engine.loadByCode(code, profile);
    }

    private List<OqsEngineEntity> toMutateEntities(Map<String, OqsEngineEntity> entityMap) {
        Map<Long, OqsEngineEntity> mutates = new HashMap<>();

        entityMap.forEach(
                (k, v) -> {
                    OqsEngineEntity engineEntity = mutates.get(v.getId());
                    if (null != engineEntity) {
                        engineEntity.getAttributes().putAll(v.getAttributes());
                        if (v.isDeleted()) {
                            engineEntity.setDeleted(true);
                            if (v.getFather() > 0) {
                                engineEntity.setFather(v.getFather());
                            }
                            if (v.getRelationTables() != null) {
                                engineEntity.getRelationTables().putAll(v.getRelationTables());
                            }
                        }
                        engineEntity.updateIntact();
                        engineEntity.setUpdateTime(v.getUpdateTime());
                    } else {
                        mutates.put(v.getId(), v);
                    }
                }
        );

        return new ArrayList<>(mutates.values());
    }

    //  将合并值到attributes中
    private void mergeEntityToResult(ParseResult parseResult, OqsEngineEntity entity) {

        String key = entity.getId() + "@@" + entity.getTable();

        OqsEngineEntity target = parseResult.getFinishEntries().get(key);
        if (null == target) {
            parseResult.getFinishEntries().put(key, entity);
            return;
        }
        target.getAttributes().putAll(entity.getAttributes());

        if (entity.isDeleted()) {
            target.setDeleted(true);
        }

        if (entity.getFather() > 0 && target.getFather() == 0) {
            target.setFather(entity.getFather());
        }

        if (entity.getUpdateTime() >= target.getUpdateTime()) {
            target.setUpdateTime(entity.getUpdateTime());
        }
    }

    private OqsEngineEntity oneRowParser(Map<String, CanalEntry.Column> columns, boolean isDelete,
                                         IEntityClass tableEntityClass,
                                         ParseResult parseResult)
            throws SQLException, JsonProcessingException {
        try {
            long entityClassId = BinLogParseUtils.getLongFromColumn(columns, SystemColumn.SYS_ENTITY_CLASS);
            String profile = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.SYS_PROFILE);
            long id = BinLogParseUtils.getLongFromColumn(columns, SystemColumn.ID);

            IEntityClass iEntityClass = tableEntityClass;
            IEntityClass relationTableEntityClass = engine.load(String.valueOf(entityClassId), profile).get();
            List<IEntityClass> tableEntityClasses = relationTableEntityClass.getEntityTables().stream().collect(Collectors.toList());
            OqsEngineEntity.Builder builder = new OqsEngineEntity.Builder();
            builder.withDeleted(isDelete);
            builder.withEntityClassRef(new EntityClassRef(entityClassId, "", "", profile));
            builder.withId(id);
            builder.withRelationTables(tableEntityClass.id(), tableEntityClass.code());
            builder.withTableEntityClass(tableEntityClasses);

            if (entityClassId != tableEntityClass.id()) {
                builder.withFather(tableEntityClass.id());
            } else {
                IEntityClass ec = tableEntityClass.extendEntityClass();
                if (null != ec) {
                    builder.withFather(ec.id());
                } else {
                    builder.withFather(0);
                }
            }

            builder.withAttribute("id", id);
            /**取出元数据**/
            Map<String, IEntityField> iEntityFieldMap = new HashMap<>();
            engine.describe(iEntityClass, profile).getAllFields().stream().forEach(p -> iEntityFieldMap.put(p.name().replace(".", "_"), p));

            for (CanalEntry.Column column : columns.values()) {
                if (column.getName().equals(SystemColumn.SYS_OPERATE_TIME)) {
                    //  update time.
                    long updateTime = Long.parseLong(column.getValue());
                    builder.withUpdateTime(updateTime);
                } else if (column.getName().equals(SystemColumn.SYS_VERSION)) {
                    //  update version.
                    if (!column.getValue().isEmpty()) {
                        int version = Integer.parseInt(column.getValue());
                        builder.withVersion(version);
                    }
                } else if (column.getName().equals(SystemColumn.ID) ||
                        column.getName().equals(SystemColumn.SYS_ENTITY_CLASS) ||
                        column.getName().equals(SystemColumn.SYS_PROFILE) ||
                        column.getName().equals(SystemColumn.SYS_DELETED)) {
                    //  do nothing
                } else if (column.getName().equals(SystemColumn.DYNAMIC_FIELD)) {
                    builder.withAttributes(OriginEntityUtils.parserDynamic(iEntityClass, BinLogParseUtils.getStringFromColumn(columns, SystemColumn.DYNAMIC_FIELD), !StringUtils.isEmpty(tableEntityClass.realProfile())));
                } else {
                    String name = column.getName();
                    String value = column.getValue();

                    if (StringUtils.isEmpty(value) || StringUtils.equalsIgnoreCase("null", value)) {
                        builder.withAttribute(name, null);
                        continue;
                    }

                    IEntityField field = iEntityFieldMap.get(name);
                    if (null == field) {
                        logger.warn("entityField can not be null, column name {}", name);
                        continue;
                    }

                    OriginEntityUtils.formatFiledValues(builder, field, name, value);
                }
            }

            if (parseResult.getStartId() == CDCConstant.NOT_INIT_START_ID) {
                parseResult.setStartId(id);
            }

            return builder.build();
        } catch (Exception e) {
            throw e;
        }
    }


    private void eventHandler(CanalEntry.EventType eventType, SystemAttachment attachment, IEntityClass entityClass, OqsEngineEntity before,
                              OqsEngineEntity after) {
        //TO publish event
        switch (eventType) {
            case INSERT: {
                publishCreatedEvent(entityClass, attachment, after);
                break;
            }
            case UPDATE: {
                if (attachment.getDelUname() != null && attachment.getDelUId() > 0) {
                    //skip del event
                    break;
                }
                publishUpdatedEvent(entityClass, attachment, before, after);
                break;
            }
            case DELETE: {
                publishDeletedEvent(entityClass, attachment, before);
                break;
            }
            default:
                //do nothing
        }
    }

    private void fillAttachment(SystemAttachment systemAttachment, Map<String, Object> context) {
        context.put("attachment", systemAttachment);
    }

    private void publishDeletedEvent(IEntityClass entityClass, SystemAttachment attachment, OqsEngineEntity b) {
        eventQueue.feedDelete(attachment, b)
                .thenAcceptOnce(x -> {
                    try {
                        if (publisher != null) {
                            Map<String, Object> context = new HashMap<>();
                            fillAttachment(attachment, context);
                            publisher.publishTransactionEvent(new EntityDeleted(entityClass.code()
                                    , x.getId(), x.getAttributes(), false, context));
                        }
                    } catch (Throwable throwable) {
                        logger.error("{}", throwable);
                    }
                }, eventThreadPool);
    }

    private void publishUpdatedEvent(IEntityClass entityClass, SystemAttachment attachment, OqsEngineEntity b, OqsEngineEntity a) {
        eventQueue.feedUpdate(attachment, b, a)
                .thenAcceptOnce(x -> {
                    try {
                        if (publisher != null) {
                            OqsEngineEntity before = x._1;
                            OqsEngineEntity after = x._2;
                            Map<String, Object> context = new HashMap<>();
                            fillAttachment(attachment, context);
                            EntityUpdated entityUpdated = new EntityUpdated(entityClass.code()
                                    , before.getId(), before.getAttributes(), after.getAttributes(), false, context);
                            int version = after.getVersion();
                            entityUpdated.setVer(version);
                            publisher.publishTransactionEvent(entityUpdated);
                        }
                    } catch (Throwable throwable) {
                        logger.error("{}", throwable);
                    }
                }, eventThreadPool);
    }

    private void publishCreatedEvent(IEntityClass entityClass, SystemAttachment attachment, OqsEngineEntity after) {
        eventQueue.feedCreate(attachment, after)
                .thenAcceptOnce(x -> {
                    try {
                        if (publisher != null) {
                            Map<String, Object> context = new HashMap<>();
                            fillAttachment(attachment, context);
                            publisher.publishTransactionEvent(new EntityCreated(entityClass.code()
                                    , x.getId(), x.getAttributes(), false, context));
                        }
                    } catch (Throwable throwable) {
                        logger.error("{}", throwable);
                    }
                }, eventThreadPool);
    }

}
