package com.xforceplus.ultraman.cdc;

import com.alibaba.otter.canal.common.utils.NamedThreadFactory;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.cdc.core.local.EmbedCDCServer;
import com.xforceplus.ultraman.cdc.core.remote.DispatcherCDCServer;
import com.xforceplus.ultraman.cdc.dto.enums.CDCStatus;
import com.xforceplus.ultraman.cdc.lock.CDCDestinationLock;
import com.xforceplus.ultraman.cdc.processor.DataProcessor;
import com.xforceplus.ultraman.cdc.reader.CDCPropertyPackage;
import com.xforceplus.ultraman.cdc.reader.CDCPropertyPackageInternal;
import com.xforceplus.ultraman.cdc.utils.TimeWaitUtils;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by justin.xu on 06/2023.
 *
 * @since 1.8
 */
public class CDCStarter implements CDCLifeCycle {

    private static final Logger logger = LoggerFactory.getLogger(CDCStarter.class);

    private CDCServer cdcServer;

    private volatile boolean isShutdown = false;

    private ScheduledExecutorService executor = null;

    private CDCDestinationLock lock = null;
    
    private String OQS_CDC_STATUS = "oqs.cdc.status";
    
//    private AtomicInteger gauge = Metrics.gauge("oqs.cdc.status", new AtomicInteger());

    public CDCStarter(String key, CDCPropertyPackage cdcPropertyPackage, DataProcessor dataProcessor, CDCDestinationLock lock) {
        this.executor = Executors.newScheduledThreadPool(cdcPropertyPackage.readers().size(),
                new NamedThreadFactory("cdc-server-metrics"));

        if (cdcPropertyPackage.model().equals(CDCPropertyPackageInternal.Model.REMOTE)) {

            if (lock != null) {
                lock.run(key, () -> {
                    cdcServer = new DispatcherCDCServer(cdcPropertyPackage, dataProcessor);
                },  () -> {
                    cdcServer = new DispatcherCDCServer(cdcPropertyPackage, dataProcessor);
                    this.init();
                });
            } else {
                cdcServer = new DispatcherCDCServer(cdcPropertyPackage, dataProcessor);
            }
        } else if (cdcPropertyPackage.model().equals(CDCPropertyPackageInternal.Model.LOCAL)  ) {
            if (lock != null) {
                lock.run(key, () -> {
                    cdcServer = new EmbedCDCServer(cdcPropertyPackage, dataProcessor);
                },  () -> {
                    cdcServer = new EmbedCDCServer(cdcPropertyPackage, dataProcessor);
                    this.init();
                });
            } else {
                cdcServer = new EmbedCDCServer(cdcPropertyPackage, dataProcessor);
            }
        }
    }

    @PostConstruct
    @Override
    public void init() {
        if (cdcServer != null) {
            cdcServer.init();
            isShutdown = false;
            cdcServer.execute();
            this.metrics();
//            executor.submit(this::metrics);
        }
    }

    @PreDestroy
    @Override
    public void destroy() {
        if (cdcServer != null) {
            cdcServer.destroy();
            isShutdown = true;
            executor.shutdown();
        }
    }

    public CDCServer cdcServer() {
        return cdcServer;
    }

    private void metrics() {
        Map<String, CDCStatus> cdcStatusMap =
                cdcServer.metrics();
        if (null != cdcStatusMap) {
            cdcStatusMap.forEach((k,v) -> {
                Gauge.builder(OQS_CDC_STATUS, () -> {
                            return cdcStatusMap.get(k).ordinal();
                        })
                        .tag("destination", k)
                        .register(Metrics.globalRegistry);
            });
        }
    }
}
