package com.xforceplus.ultraman.cdc.processor.impl;


import com.alibaba.otter.canal.protocol.CanalEntry;
import com.alibaba.otter.canal.protocol.Message;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.cdc.adapter.CDCBeforeCallback;
import com.xforceplus.ultraman.cdc.adapter.EngineAdapterService;
import com.xforceplus.ultraman.cdc.context.ParserContext;
import com.xforceplus.ultraman.cdc.dto.ParseResult;
import com.xforceplus.ultraman.cdc.dto.constant.CDCConstant;
import com.xforceplus.ultraman.cdc.processor.DataProcessor;
import com.xforceplus.ultraman.cdc.processor.EventQueue;
import com.xforceplus.ultraman.cdc.processor.SystemAttachment;
import com.xforceplus.ultraman.cdc.utils.BinLogParseUtils;
import com.xforceplus.ultraman.extensions.cdc.status.StatusService;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.plus.common.metrics.MetricsDefine;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.sdk.core.event.EntityCreated;
import com.xforceplus.ultraman.sdk.core.event.EntityDeleted;
import com.xforceplus.ultraman.sdk.core.event.EntityUpdated;
import com.xforceplus.ultraman.sdk.infra.event.EventPublisher;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import io.micrometer.core.annotation.Timed;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;

import static com.xforceplus.ultraman.metadata.values.DateTimeValue.ZONE_ID;

/**
 * Created by justin.xu on 08/2022.
 *
 * @since 1.8
 */
public class DefaultDataProcessor implements DataProcessor {

    final Logger logger = LoggerFactory.getLogger(DefaultDataProcessor.class);
    @Resource
    private EntityClassEngine engine;
    @Autowired(required = false)
    private StatusService statusService;

    @Autowired(required = false)
    private List<CDCBeforeCallback> cdcBeforeCallbacks;

    @Resource
    private EventPublisher publisher;
    @Resource
    private EngineAdapterService engineAdapterService;
    @Resource
    private EventQueue eventQueue;

    /**
     * 属性字符串表示解析为实际对象列表.
     *
     * @param attrStr 属性的字符串表示.
     * @return 解析结果.
     * @throws JsonProcessingException JSON解析失败.
     */
    public static Map<String, Object> attributesToMap(String attrStr) throws JsonProcessingException {
        return JacksonDefaultMapper.OBJECT_MAPPER.readValue(attrStr, Map.class);
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"initiator", "cdc", "action", "consume"}
    )
    @Override
    public boolean onProcess(Message message, long batchId) throws SQLException {
        try {
            messageProcess(message, batchId);

            return true;
        } catch (Exception e) {
            logger.error("batchId : {}, consume message failed, message : {}", message.getId(),
                    e.getMessage());
            throw e;
        }
    }

    private void messageProcess(Message message, long batchId) throws SQLException {
        //  同步逻辑.
        parseCanalEntries(message.getEntries(), batchId);
    }

    /**
     * 解析入口函数,对一个批次进行解析.
     *
     * @param entries 完整的批次信息.
     * @return 成功条数.
     */
    private int parseCanalEntries(List<CanalEntry.Entry> entries, long batchId) throws SQLException {
        //  初始化上下文
        ParserContext parserContext =
                new ParserContext(batchId);

        ParseResult parseResult = new ParseResult();

        for (CanalEntry.Entry entry : entries) {

            //  不是TransactionEnd/RowData类型数据, 将被过滤
            switch (entry.getEntryType()) {
                case TRANSACTIONBEGIN:
                case TRANSACTIONEND:
                    eventHandler(entry.getEntryType());
                    break;
                case ROWDATA:
                    rowDataParse(entry, parserContext, parseResult);
                    break;
                default: {
                }
            }
        }

        //  批次数据整理完毕，开始执行index写操作。
        if (!parseResult.getFinishEntries().isEmpty()) {
            try {
                List<OqsEngineEntity> oqsEngineEntities = new ArrayList<>(
                        parseResult.getFinishEntries().values());
                //  通过执行器执行index同步
                cdcBeforeCallbacks.forEach(x -> {
                    try {
                        x.mutate(oqsEngineEntities);
                    } catch (Throwable throwable) {
                        logger.error("CDC callback ERROR name:{} , ex:{}", x.name(), throwable);
                    }
                });
                boolean flag = engineAdapterService.batchUpsertOperation(oqsEngineEntities);
                if (flag) {
                    Instant instant = LocalDateTime.now().atZone(ZONE_ID).toInstant();
                    long currentTime = instant.toEpochMilli();
                    if (statusService != null) {
                        statusService.clearStatus(oqsEngineEntities, currentTime);
                    }
                } else {
                    throw new RuntimeException("");
                }
            } catch (Exception e) {
                throw new SQLException(String.format("write sphinx-batch error, startId : %s, message : %s",
                        parseResult.getStartId(), e.getMessage()));
            }
        }

        return parseResult.getFinishEntries().size();
    }

    private OqsEngineEntity cloneEntity(OqsEngineEntity src) {
        OqsEngineEntity target = new OqsEngineEntity();
        target.setEntityClassRef(src.getEntityClassRef());
        target.setId(src.getId());
        target.setVersion(src.getVersion());

        Map<String, Object> attributes = src.getAttributes();
        try {
            String attr = JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(attributes);
            Map<String, Object> newAttr = JacksonDefaultMapper.OBJECT_MAPPER.readValue(attr, Map.class);
            target.setAttributes(newAttr);
        } catch (Throwable throwable) {
            logger.error("{}", throwable);
        }

        target.setUpdateTime(src.getUpdateTime());
        target.setFather(src.getFather());
        target.setDeleted(src.isDeleted());
        return target;
    }

    /**
     * 对rowData进行解析，rowData为对一张表的CUD操作，记录条数1～N.
     *
     * @param entry         canal对象同步实例.
     * @param parserContext 上下文.
     */
    private void rowDataParse(CanalEntry.Entry entry, ParserContext parserContext,
                              ParseResult parseResult) {

        String tableName = entry.getHeader().getTableName();
        if (tableName.isEmpty()) {
            logger.error("batch : {}, table name could not be Null, [{}]",
                    parserContext.getBatchId(), entry.getStoreValue());
            return;
        }
        Optional<IEntityClass> entityClassOp =
                foundEntityClassFromTableName(tableName);
        if (!entityClassOp.isPresent()) {
            logger.error("batch : {}, entityClass could not be Null, [{}]",
                    parserContext.getBatchId(), entry.getStoreValue());
            return;
        }

        CanalEntry.RowChange rowChange = null;
        try {
            rowChange = CanalEntry.RowChange.parseFrom(entry.getStoreValue());
        } catch (Exception e) {
            logger.error("batch : {}, parse entry value failed, [{}], [{}]",
                    parserContext.getBatchId(), entry.getStoreValue(), e);
            return;
        }

        CanalEntry.EventType eventType = rowChange.getEventType();
        //  遍历RowData
        for (CanalEntry.RowData rowData : rowChange.getRowDatasList()) {

            //  获取一条完整的更新前的columns
            List<CanalEntry.Column> beforeColumns = rowData.getBeforeColumnsList();

            //  获取一条完整的更新后的columns
            List<CanalEntry.Column> afterColumns = rowData.getAfterColumnsList();

            OqsEngineEntity beforeEntity = null;
            OqsEngineEntity afterEntity = null;

            String profile = null;
            SystemAttachment systemAttachment = null;
            try {
                switch (eventType) {
                    case INSERT: {
                        afterEntity = oneRowParser(afterColumns, false, entityClassOp.get(), parseResult);
                        //parseResult.getFinishEntries().put(afterEntity.getId(), afterEntity);
                        systemAttachment = getAttachment(afterColumns);
                        mergeEntityToResult(parseResult, afterEntity);
                        break;
                    }
                    case UPDATE: {
                        try {
                            beforeEntity = oneRowParser(beforeColumns, false, entityClassOp.get(), parseResult);
                        } catch (Exception e) {
                            //  beforeEntity will be ignore...
                            logger.warn(
                                    "beforeEntity in update is null, will be ignore, but it will influence event");
                        }
                        afterEntity = oneRowParser(afterColumns, false, entityClassOp.get(), parseResult);
                        systemAttachment = getAttachment(afterColumns);
                        mergeEntityToResult(parseResult, afterEntity);
                        break;
                    }
                    case DELETE: {
                        beforeEntity = oneRowParser(beforeColumns, true, entityClassOp.get(), parseResult);
                        systemAttachment = getAttachment(beforeColumns);
                        mergeEntityToResult(parseResult, beforeEntity);
                        break;
                    }
                }

                OqsEngineEntity beforeClone = null;
                OqsEngineEntity afterClone = null;

                if (beforeEntity != null) {
                    beforeClone = cloneEntity(beforeEntity);
                }

                if (afterEntity != null) {
                    afterClone = cloneEntity(afterEntity);
                }
                //  todo to handle event;
                if (systemAttachment == null) {
                    systemAttachment = new SystemAttachment();
                }
                eventHandler(eventType, systemAttachment, entityClassOp.get(), beforeClone, afterClone);
            } catch (Exception e) {
                //  对于数据层面的错误，只记录，不回滚批次.
                logger.warn("parse entity error, message : {}", e.getMessage());
            }
        }
    }

    private SystemAttachment getAttachment(List<CanalEntry.Column> columns) {
        String attrStr = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.DYNAMIC_FIELD);
        SystemAttachment attachment = new SystemAttachment();
        try {
            if (attrStr.isEmpty()) {
                return attachment;
            } else {
                Map<String, Object> map = attributesToMap(attrStr);
                Object o = map.get("#a");
                if (o != null) {
                    Map<String, Object> attachmentMap = (Map<String, Object>) o;
                    Object grouped = attachmentMap.get("grouped");
                    if (grouped != null) {
                        attachment.setGrouped(Integer.parseInt(grouped.toString()));
                    }
                }
                return attachment;
            }
        } catch (Throwable throwable) {
            return attachment;
        }
    }

    private Optional<IEntityClass> foundEntityClassFromTableName(String tableName) {
        String[] tableSplit = tableName.split("_");
        if (tableSplit.length < 3) {
            return Optional.empty();
        }

        String code = tableSplit[2];
        String profile = null;
        if (tableSplit.length > 3) {
            profile = tableSplit[3];
        }

        return engine.loadByCode(code, profile);
    }

    //  将合并值到attributes中
    private void mergeEntityToResult(ParseResult parseResult, OqsEngineEntity entity) {

        OqsEngineEntity target = parseResult.getFinishEntries().get(entity.getId());
        if (null == target) {
            parseResult.getFinishEntries().put(entity.getId(), entity);
            return;
        }

        target.getAttributes().putAll(entity.getAttributes());

        if (entity.isDeleted()) {
            target.setDeleted(true);
        }

        if (entity.getFather() > 0 && target.getFather() == 0) {
            target.setFather(entity.getFather());
        }

        target.setUpdateTime(entity.getUpdateTime());
    }

    private OqsEngineEntity oneRowParser(List<CanalEntry.Column> columns, boolean isDelete,
                                         IEntityClass tableEntityClass,
                                         ParseResult parseResult)
            throws SQLException, JsonProcessingException {
        try {
            long entityClass = BinLogParseUtils.getLongFromColumn(columns, SystemColumn.SYS_ENTITY_CLASS);
            String profile = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.SYS_PROFILE);
            long id = BinLogParseUtils.getLongFromColumn(columns, SystemColumn.ID);

            IEntityClass iEntityClass = tableEntityClass;

            OqsEngineEntity.Builder builder = new OqsEngineEntity.Builder();
            builder.withDeleted(isDelete);
            builder.withEntityClassRef(new EntityClassRef(entityClass, "", "", profile));
            builder.withId(id);

            if (entityClass != tableEntityClass.id()) {
                builder.withFather(tableEntityClass.id());
            } else {
                IEntityClass ec = tableEntityClass.extendEntityClass();
                if (null != ec) {
                    builder.withFather(ec.id());
                } else {
                    builder.withFather(0);
                }
            }

            builder.withAttribute("id", id);

            for (CanalEntry.Column column : columns) {
                if (column.getName().equals(SystemColumn.SYS_OPERATE_TIME)) {
                    //  update time.
                    long updateTime = Long.parseLong(column.getValue());
                    builder.withUpdateTime(updateTime);
                } else if (column.getName().equals(SystemColumn.SYS_VERSION)) {
                    //  update version.
                    if (!column.getValue().isEmpty()) {
                        int version = Integer.parseInt(column.getValue());
                        builder.withVersion(version);
                    }
                } else if (column.getName().equals(SystemColumn.ID) ||
                        column.getName().equals(SystemColumn.SYS_ENTITY_CLASS) ||
                        column.getName().equals(SystemColumn.SYS_PROFILE) ||
                        column.getName().equals(SystemColumn.SYS_DELETED)) {
                    //  do nothing

                } else if (column.getName().equals(SystemColumn.DYNAMIC_FIELD)) {
                    builder.withAttributes(attrCollection(iEntityClass, columns));
                } else {
                    String name = column.getName();
                    String value = column.getValue();
                    IEntityField field = engine.describe(iEntityClass, profile).getAllFields().stream()
                            .filter(p -> p.name().replace(".", "_").equalsIgnoreCase(name))
                            .findFirst().orElse(null);

                    if (null == field) {
                        logger.warn("entityField can not be null, column name {}", name);
                        continue;
                    }

                    if (value.isEmpty()) {
                        builder.withAttribute(name, null);
                    } else {
                        if (field.type() == FieldType.LONG || field.type() == FieldType.DATETIME) {
                            builder.withAttribute(name, Long.parseLong(value));
                        } else if (field.type() == FieldType.BOOLEAN) {
                            long result = Long.parseLong(value);
                            builder.withAttribute(name, result != 0);
                        } else if (field.type() == FieldType.DECIMAL) {
                            builder.withAttribute(name, new BigDecimal(value));
                        } else if (field.type() == FieldType.STRINGS) {
                            try {
                                List<String> multiValues = JacksonDefaultMapper.OBJECT_MAPPER.readValue(value, JacksonDefaultMapper.LIST_TYPE_REFERENCE);
                                builder.withAttribute(name, multiValues);
                            } catch (Throwable throwable) {
                                logger.error("{}", throwable);
                                builder.withAttribute(name, value);
                            }
                        } else {
                            builder.withAttribute(name, value);
                        }
                    }
                }
            }

            if (parseResult.getStartId() == CDCConstant.NOT_INIT_START_ID) {
                parseResult.setStartId(id);
            }

            return builder.build();
        } catch (Exception e) {
            throw e;
        }
    }

    private void eventHandler(CanalEntry.EntryType entryType) {

    }

    private void eventHandler(CanalEntry.EventType eventType, SystemAttachment attachment, IEntityClass entityClass, OqsEngineEntity before,
                              OqsEngineEntity after) {
        //TO publish event
        attachment.setEntityClass(entityClass);
        switch (eventType) {
            case INSERT: {
                publishCreatedEvent(attachment, after);
                break;
            }
            case UPDATE: {
                publishUpdatedEvent(attachment, before, after);
                break;
            }
            case DELETE: {
                publishDeletedEvent(attachment, before);
                break;
            }
            default:
                //do nothing
        }
    }

    private void publishDeletedEvent(SystemAttachment attachment, OqsEngineEntity b) {
        eventQueue.feedDelete(attachment, b)
                .thenAcceptOnce(x -> {
                    if (publisher != null) {
                        publisher.publishTransactionEvent(new EntityDeleted(attachment.getEntityClass().code()
                                , x.getId(), x.getAttributes(), false, Collections.emptyMap()));
                    }
                });
    }

    private void publishUpdatedEvent(SystemAttachment attachment, OqsEngineEntity b, OqsEngineEntity a) {
        eventQueue.feedUpdate(attachment, b, a)
                .thenAcceptOnce(x -> {
                    if (publisher != null) {
                        OqsEngineEntity before = x._1;
                        OqsEngineEntity after = x._2;
                        publisher.publishTransactionEvent(new EntityUpdated(attachment.getEntityClass().code()
                                , before.getId(), before.getAttributes(), after.getAttributes(), false, Collections.emptyMap()));
                    }
                });
    }

    private void publishCreatedEvent(SystemAttachment attachment, OqsEngineEntity after) {
        eventQueue.feedCreate(attachment, after)
                .thenAcceptOnce(x -> {
                    if (publisher != null) {
                        publisher.publishTransactionEvent(new EntityCreated(attachment.getEntityClass().code()
                                , x.getId(), x.getAttributes(), false, Collections.emptyMap()));
                    }
                });
    }

    /**
     * 转换attribute.
     *
     * @param columns 原始数据集.
     * @return 对象键值对.
     */
    private Map<String, Object> attrCollection(IEntityClass entityClass,
                                               List<CanalEntry.Column> columns)
            throws JsonProcessingException {

        Map<String, Object> result;
        String attrStr = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.DYNAMIC_FIELD);
        if (attrStr.isEmpty()) {
            result = new HashMap<>();
        } else {
            result = attributesToMap(attrStr);
        }

        entityClass.selfFields().stream().filter(IEntityField::isDynamic).forEach(
                f -> {
                    if (!result.containsKey(f.name())) {
                        result.put(f.name(), null);
                    } else if (f.type().equals(FieldType.BOOLEAN)) {
                        result.computeIfPresent(
                                f.name(),
                                (k, s) -> {
                                    return ((int) s) != 0;
                                }
                        );

                    }
                }
        );
        return result;
    }

}
