package com.xforceplus.ultraman.cdc.core.local;

import static com.xforceplus.ultraman.cdc.dto.constant.CDCConstant.DEFAULT_BATCH_SIZE;

import com.alibaba.otter.canal.common.utils.NamedThreadFactory;
import com.alibaba.otter.canal.protocol.ClientIdentity;
import com.alibaba.otter.canal.protocol.Message;
import com.xforceplus.ultraman.cdc.CDCServer;
import com.xforceplus.ultraman.cdc.core.local.embed.EmbedDataConsumer;
import com.xforceplus.ultraman.cdc.core.local.embed.EmbedDataProducer;
import com.xforceplus.ultraman.cdc.core.local.embed.SourceDataConsumer;
import com.xforceplus.ultraman.cdc.core.local.embed.SourceDataProducer;
import com.xforceplus.ultraman.cdc.dto.enums.CDCStatus;
import com.xforceplus.ultraman.cdc.processor.DataProcessor;
import com.xforceplus.ultraman.cdc.reader.CDCPropertyPackage;
import com.xforceplus.ultraman.cdc.reader.CanalPropertiesReader;
import com.xforceplus.ultraman.cdc.utils.TimeWaitUtils;
import io.vavr.Tuple2;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by justin.xu on 06/2023.
 *
 * @since 1.8
 */
public class EmbedCDCServer implements CDCServer {

    private static final Logger logger = LoggerFactory.getLogger(EmbedCDCServer.class);

    private SourceDataProducer sourceDataProducer;

    private SourceDataConsumer sourceDataConsumer;

    private ScheduledExecutorService executor = null;

    private volatile boolean isStopped = false;

    private volatile boolean ready = false;

    private int loopsDurationInMillis = 3;
    private int stopWaitInMillis = 1000;

    public EmbedCDCServer(CDCPropertyPackage cdcPropertyPackage, DataProcessor dataProcessor) {

        this.executor = Executors.newScheduledThreadPool(cdcPropertyPackage.readers().size(),
            new NamedThreadFactory("cdc-server-runner"));


        sourceDataProducer = new EmbedDataProducer(cdcPropertyPackage);
        sourceDataConsumer = new EmbedDataConsumer(cdcPropertyPackage, dataProcessor);

    }

    @Override
    public void init() {
        //  初始化生产者.
        sourceDataProducer.init();

        //  初始化消费者.
        sourceDataConsumer.init();

        //  注册单一的消费者.
        for (ClientIdentity clientIdentity : sourceDataConsumer.clients()) {
            if (!sourceDataProducer.singletonRegistry(clientIdentity)) {
                throw new RuntimeException("CDCServer start error, reason : register consumer failed.");
            }
        }

        ready = true;

        logger.info("cdc-server is start.");
    }

    @Override
    public void destroy() {

        isStopped = true;

        TimeWaitUtils.wakeupAfter(stopWaitInMillis, TimeUnit.MILLISECONDS);

        //  销毁消费者
        sourceDataConsumer.destroy();

        //  销毁生产者.
        sourceDataProducer.destroy();

        if (null != executor) {
            executor.shutdown();
        }

        ready = false;
    }

    @Override
    public void execute() {
        if (null != executor && !isStopped) {
            for (Tuple2<ClientIdentity, CanalPropertiesReader> value : sourceDataProducer.clientIdentityWithReader()) {
                //  启动一个轮训任务.
                executor.execute(() -> {
                    //  服务没有关闭将继续轮询.
                    while (!isStopped) {
                        Message message = null;
                        boolean result = false;

                        CanalPropertiesReader reader = value._2();
                        int batchSize = null != reader ? reader.getBatchSize() : DEFAULT_BATCH_SIZE;
                        ClientIdentity clientIdentity = value._1();
                        if (null == clientIdentity) {
                            metrics.put(clientIdentity.getDestination(), CDCStatus.DIS_CONNECTED);
                            logger.warn("client is null, cdc status is disconnect, task will stopped.");
                            return;
                        }

                        try {
                            metrics.put(clientIdentity.getDestination(), CDCStatus.CONNECTED);
                            message = sourceDataProducer.onMessage(clientIdentity, batchSize);
                            if (null != message) {
                                sourceDataConsumer.onConsume(message);

                                //  确认当前批次
                                sourceDataProducer.ack(value._1(), message.getId());
                            }
                            result = true;
                        } catch (Exception e) {
                            if (null != message) {
                                sourceDataProducer.rollback(clientIdentity, message.getId());
                            }

                            metrics.put(clientIdentity.getDestination(), CDCStatus.CONSUME_FAILED);
                        }

                        if (!result) {
                            TimeWaitUtils.wakeupAfter(loopsDurationInMillis, TimeUnit.MILLISECONDS);
                        }

                    }

                    logger.info("cdc-consumer executor is going to stop...");
                });
            }
        }
    }

    @Override
    public boolean allReady() {
        return ready;
    }

    @Override
    public Map<String, CDCStatus> metrics() {
        return metrics;
    }
}
