package com.xforceplus.ultraman.cdc.reader;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

/**
 * Created by justin.xu on 06/2023.
 *
 * @since 1.8
 */
public class CDCResolver implements CDCConfigResolver {

    private Logger logger = LoggerFactory.getLogger(CDCResolver.class);

    private static short DEFAULT_CLIENT_ID = 999;
    private static short DEFAULT_SLAVE_ID = 899;

    private static final String CDC_MODEL_PATH = CDCS + ".model";
    
    private static final String CDC_FORCE_LOCAL = CDCS + ".forceLocal";
    private static final String CDC_PROPERTIES_PATH = CDCS + ".properties";

    private short clientIdStart = DEFAULT_CLIENT_ID;
    private short slaveIdStart = DEFAULT_SLAVE_ID;

    @Override
    public CDCPropertyPackageInternal resolve(Config config) {
        ConfigObject configObject = config.getObject(CDCS);
        if (configObject.isEmpty()) {
            throw new RuntimeException("至少配置一个数据源");
        }

        CDCPropertyPackageInternal.Model model = null;
        if (config.hasPath(CDC_MODEL_PATH)) {
            model = buildModel(config, CDC_MODEL_PATH);
        }

        boolean forceLocal = false;
//        if (config.hasPath(CDC_FORCE_LOCAL)) {
//            try {
//                forceLocal = config.
//            } catch (Exception exception) {
//                
//            }
//        }

        logger.info("cdc model : {}", model);

        if (null == model) {
            throw new RuntimeException("必须制定CDC-SERVER启动模式LOCAL/REMOTE");
        }

        if (config.hasPath(CDC_PROPERTIES_PATH)) {
            return buildCDCProperty(model, (List<Config>) config.getConfigList(CDC_PROPERTIES_PATH));
        }

        throw new RuntimeException("No Master datasource config");
    }

    private CDCPropertyPackageInternal.Model buildModel(Config config, String path) {
        String modelString = config.getString(path);

        if (null != modelString && !modelString.isEmpty()) {
            return CDCPropertyPackageInternal.Model.instance(modelString);
        }

        return null;
    }

    private CDCPropertyPackageInternal buildCDCProperty(CDCPropertyPackageInternal.Model model, List<Config> configs) {

        CDCPropertyPackageInternal cdcPropertyPackageInternal = new CDCPropertyPackageInternal();
        cdcPropertyPackageInternal.setModel(model);

        for (Config config : configs) {
            String destination = config.getString("destination");
            if (null == destination || destination.isEmpty()) {
                throw new IllegalArgumentException("destination could not be null.");
            }

            String filter = config.getString("filter");
            if (null == filter || filter.isEmpty()) {
                throw new IllegalArgumentException("filter could not be null.");
            }

            boolean hasBlackFilter = config.hasPath("blackFilter");
            String blackFilter = null;
            if(hasBlackFilter) {
                blackFilter = config.getString("blackFilter");
            }

            String masterHost = config.getString("masterHost");
            if (null == masterHost || masterHost.isEmpty()) {
                throw new IllegalArgumentException("hostName could not be null.");
            }

            int masterPort = config.getInt("masterPort");
            String masterUser = config.getString("masterUser");
            String masterPassword = config.getString("masterPassword");

            CanalPropertiesReader canalPropertiesReader =
                    new CanalPropertiesReader(masterHost, masterPort, masterUser, masterPassword, destination, filter, blackFilter);

            canalPropertiesReader.setClientId(clientIdStart--);

            if (config.hasPath("batchSize")) {
                int batchSize = config.getInt("batchSize");
                if (batchSize > 0) {
                    canalPropertiesReader.setBatchSize(batchSize);
                    canalPropertiesReader.setMemoryStorageBufferSize(batchSize);
                }
            }
            if (config.hasPath("threadBatchSize")) {
                int threadBatchSize = config.getInt("threadBatchSize");
                if (threadBatchSize > 0) {
                    canalPropertiesReader.setThreadBatchSize(threadBatchSize);
                }
            }

            if (config.hasPath("zkServers")) {
                String zkServers = config.getString("zkServers");
                if (null != zkServers && !zkServers.isEmpty()) {
                    canalPropertiesReader.setZkServers(zkServers);

                    logger.info("zkServers : {}", zkServers);
                }
            }

            if (model.equals(CDCPropertyPackageInternal.Model.LOCAL)) {
                if (config.hasPath("dataDir")) {
                    String dataDir = config.getString("dataDir");
                    if (null != dataDir && !dataDir.isEmpty()) {
                        canalPropertiesReader.setDataDir(dataDir);
                    }
                }

                if (config.hasPath("logFile")) {
                    String logFile = config.getString("logFile");
                    if (null != logFile && !logFile.isEmpty()) {
                        canalPropertiesReader.setMasterLogFile(logFile);
                    }
                }

                canalPropertiesReader.setSlaveId(slaveIdStart--);

                //  this open for test.
                if (config.hasPath("resetMeta")) {
                    boolean resetMeta = config.getBoolean("resetMeta");
                    canalPropertiesReader.setResetMeta(resetMeta);
                }
            }

            cdcPropertyPackageInternal.addWithDuplicateIgnore(canalPropertiesReader);
        }

        return cdcPropertyPackageInternal;
    }

}
