package com.xforceplus.ultraman.cdc.processor.impl;

import static com.alibaba.otter.canal.protocol.CanalEntry.EventType.DELETE;
import static com.alibaba.otter.canal.protocol.CanalEntry.EventType.INSERT;
import static com.alibaba.otter.canal.protocol.CanalEntry.EventType.UPDATE;
import static com.xforceplus.ultraman.metadata.entity.FieldType.DATETIME;
import static com.xforceplus.ultraman.metadata.entity.FieldType.LONG;

import com.alibaba.otter.canal.protocol.CanalEntry;
import com.alibaba.otter.canal.protocol.CanalEntry.EventType;
import com.alibaba.otter.canal.protocol.CanalEntry.RowData;
import com.alibaba.otter.canal.protocol.Message;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.cdc.adapter.CDCBeforeCallback;
import com.xforceplus.ultraman.cdc.adapter.EngineAdapterService;
import com.xforceplus.ultraman.cdc.context.ParserContext;
import com.xforceplus.ultraman.cdc.utils.BinLogParseUtils;
import com.xforceplus.ultraman.cdc.utils.ThreadPoolExecutorUtils;
import com.xforceplus.ultraman.cdc.utils.TimeWaitUtils;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import java.math.BigDecimal;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

/**
 * @ClassName BinlogToES
 * @description:
 * @author: WanYi
 * @create: 2023-09-01 10:53
 * @Version 1.0
 **/
@Slf4j
public class BinlogToElasticProcess {

  private static ThreadPoolExecutor executor = ThreadPoolExecutorUtils.executor;
  private final static int maxThreadNums = ThreadPoolExecutorUtils.MAX_POOL_SIZE;
  private static List<CDCBeforeCallback> cdcBeforeCallbacks = new ArrayList<>();


  private static int dynamicThread(int entries, int threadBatchSize) {
    if (entries <= threadBatchSize) {
      return 1;
    }
    int threadNum = entries % threadBatchSize == 0 ? entries / threadBatchSize : entries / threadBatchSize + 1;
    if (threadNum > maxThreadNums) {
      threadNum = maxThreadNums;
    }
    return threadNum;
  }

  private static long time_total = 0;
  public static final int RECONNECT_WAIT_IN_SECONDS = 30;

  /**
   * 解析binlog message写入至elastic
   *
   * @param message
   * @param engine
   * @param engineAdapterService
   * @param threadBatchSize
   * @param cdcBeforeCallbacks
   * @return
   **/
  public static int parseBinlogMessage(
      Message message,
      EntityClassEngine engine,
      EngineAdapterService engineAdapterService,
      List<CDCBeforeCallback> cdcBeforeCallbacks,
      int threadBatchSize) {
    try {
      /**等待元数据初始化**/
      while (!engineAdapterService.initMetedataInitStatus()) {
        TimeWaitUtils.wakeupAfter(RECONNECT_WAIT_IN_SECONDS, TimeUnit.SECONDS);
      }
      long start = System.currentTimeMillis();
      ParserContext parserContext = new ParserContext(message.getId());
      AtomicInteger oqsEngineEntityTotals = new AtomicInteger();
      ParseMessageToRowData messageToRowData = new ParseMessageToRowData(engine);
      Map<EventType, Map<String, Map<Long, OqsEngineEntity>>> binlogRowDataMap;
      binlogRowDataMap = messageToRowData.parseCanalEntries(message.getEntries(), parserContext);
      long endTime = System.currentTimeMillis() - start;
      time_total += endTime;
      log.info("计算单位(毫秒) Message当前拉取批次大小:{},CDC处理总消耗时间:{},CDC当时处理批次消耗时间:{}", message.getEntries().size(), time_total, endTime);
      Map<String, Map<Long, OqsEngineEntity>> oqsEngineEntityMap = new HashMap<>();
      mergeEventTypeMaps(oqsEngineEntityMap, binlogRowDataMap);
      /**反查mysql数据库，带出关联数据**/
  /*    oqsEngineEntityMap.entrySet().forEach(entry ->
          Optional.ofNullable(cdcBeforeCallbacks)
              .orElseGet(Collections::emptyList)
              .forEach(x -> {
                try {
                  x.mutate(toMutateEntities(entry.getValue()));
                } catch (Throwable throwable) {
                  log.error("CDC callback ERROR name:{} , ex:{}", x.name(), throwable);
                }
              }));*/
      List<Future> futures = new ArrayList<>();
      for (Map.Entry<String, Map<Long, OqsEngineEntity>> oqsEngineEntities : oqsEngineEntityMap.entrySet()) {
        List<OqsEngineEntity> collect = oqsEngineEntities.getValue().values().stream().collect(Collectors.toList());
        //int thread = dynamicThread(collect.size(), threadBatchSize);
    /*    //  不切换线程
        if (thread == 1) {
          threadWait(futures);
          Future<Boolean> future = executor.submit(() -> engineAdapterService.batchUpsertOperation(collect));
          futures.add(future);
        } else {
          //  使用多线程处理
          int lastPos = 0;
          for (int i = 0; i < thread; i++) {*/
        //threadWait(futures);
         /*   int tempEnd = lastPos + threadBatchSize;
            lastPos = Math.min(tempEnd, collect.size());
            List<OqsEngineEntity> subOqsEntities = collect.subList(i * threadBatchSize, lastPos);
         */
        Future<Boolean> future = executor.submit(() -> engineAdapterService.batchUpsertOperation(collect));
        futures.add(future);
        oqsEngineEntityTotals.addAndGet(collect.size());
         /*   threadWait(futures);
          }
*/
        threadWait(futures);
      }
      return oqsEngineEntityTotals.get();
    } catch (Throwable e) {
      e.printStackTrace();
      throw new RuntimeException("Adapter Service return False");
    }

  }


  private static List<OqsEngineEntity> toMutateEntities(Map<Long, OqsEngineEntity> entityMap) {
    Map<Long, OqsEngineEntity> mutates = new HashMap<>();

    entityMap.forEach(
        (k, v) -> {
          OqsEngineEntity engineEntity = mutates.get(v.getId());
          if (null != engineEntity) {
            engineEntity.getAttributes().putAll(v.getAttributes());
            if (v.isDeleted()) {
              engineEntity.setDeleted(true);
              if (v.getFather() > 0) {
                engineEntity.setFather(v.getFather());
              }
            }

            engineEntity.setUpdateTime(v.getUpdateTime());
          } else {
            mutates.put(v.getId(), v);
          }
        }
    );

    return new ArrayList<>(mutates.values());
  }


  /**
   * 线程等待
   *
   * @param futures
   **/
  private static void threadWait(List<Future> futures) throws Throwable {
    if (futures.size() >= maxThreadNums) {
      for (Future future : futures) {
        Boolean flag = (Boolean) future.get();
        if (!flag) {
          throw new RuntimeException("Adapter Service return False");
        }
      }
    }
  }


  /**
   * @param oqsEngineEntityMap
   * @param binlogRowDataMap
   **/
  private static void mergeEventTypeMaps(Map<String, Map<Long, OqsEngineEntity>> oqsEngineEntityMap,
      Map<EventType, Map<String, Map<Long, OqsEngineEntity>>> binlogRowDataMap) {
    EventType[] eventTypes = {INSERT, UPDATE, UPDATE};
    for (EventType eventType : eventTypes) {
      Map<String, Map<Long, OqsEngineEntity>> eventTypeMap = binlogRowDataMap.get(eventType);
      if (eventTypeMap != null) {
        eventTypeMap.entrySet().forEach(tableOqsEngine -> {
          Map<Long, OqsEngineEntity> oqsEngineEntities = oqsEngineEntityMap.get(tableOqsEngine.getKey());
          if (oqsEngineEntities != null) {
            oqsEngineEntities.putAll(tableOqsEngine.getValue());
          } else {
            oqsEngineEntityMap.put(tableOqsEngine.getKey(), tableOqsEngine.getValue());
          }
        });
      }
    }

  }
}

@Slf4j
class ParseMessageToRowData {

  private EntityClassEngine engine;

  public ParseMessageToRowData(EntityClassEngine engine) {
    this.engine = engine;
  }

  /**
   * 解析Message一个批次，需要处理多个的rowData
   *
   * @param entries
   * @param parserContext
   * @return 解析Message批次有多少条记录.
   */
  public Map<EventType, Map<String, Map<Long, OqsEngineEntity>>> parseCanalEntries(List<CanalEntry.Entry> entries, ParserContext parserContext)
      throws SQLException {
    Map binlogRowDataMap = new HashMap<>();
    for (CanalEntry.Entry entry : entries) {
      switch (entry.getEntryType()) {
        case TRANSACTIONBEGIN:
        case TRANSACTIONEND:
          eventHandler(entry.getEntryType());
          break;
        case ROWDATA:
          rowDataParse(entry, parserContext, binlogRowDataMap);
          break;
        default: {
        }
      }
    }
    return binlogRowDataMap;
  }

  private void eventHandler(CanalEntry.EntryType entryType) {

  }


  /**
   * 对rowData进行解析，rowData为对一张表的CUD操作，记录条数1～N.
   *
   * @param entry            canal对象同步实例.
   * @param parserContext    上下文.
   * @param binlogRowDataMap 存储entry对象 解析出来rowData数据集合
   */
  private void rowDataParse(CanalEntry.Entry entry, ParserContext parserContext,
      Map<EventType, Map<String, Map<Long, OqsEngineEntity>>> binlogRowDataMap) {
    String tableName = entry.getHeader().getTableName();
    if (tableName.isEmpty()) {
      log.error("batch : {}, table name could not be Null, [{}]",
          parserContext.getBatchId(), entry.getStoreValue());
      return;
    }
    Optional<IEntityClass> entityClassOp = foundEntityClassFromTableName(tableName);
    IEntityClass iEntityClass = entityClassOp.get();
    if (!entityClassOp.isPresent()) {
      log.error("batch : {}, entityClass could not be Null, [{}]",
          parserContext.getBatchId(), tableName);
      return;
    }
    CanalEntry.RowChange rowChange;
    try {
      rowChange = CanalEntry.RowChange.parseFrom(entry.getStoreValue());

      CanalEntry.EventType eventType = rowChange.getEventType();

      for (RowData rowData : rowChange.getRowDatasList()) {
        Map<String, CanalEntry.Column> beforeColumnsMap = new HashMap();
        rowData.getBeforeColumnsList().stream().forEach(column -> beforeColumnsMap.put(column.getName(), column));
        //  获取一条完整的更新后的columns
        Map<String, CanalEntry.Column> afterColumnsMap = new HashMap();

        rowData.getAfterColumnsList().stream().forEach(column -> afterColumnsMap.put(column.getName(), column));
        OqsEngineEntity entity = null;
        switch (eventType) {
          case INSERT:
          case UPDATE: {
            entity = oneRowParser(afterColumnsMap, false, iEntityClass);
            break;
          }
          case DELETE: {
            entity = oneRowParser(beforeColumnsMap, true, iEntityClass);
            break;
          }
        }
        paddingData(binlogRowDataMap, tableName, eventType, entity);
      }
    } catch (Exception e) {
      log.error("batch : {}, parse entry value failed, [{}], [{}]",
          parserContext.getBatchId(), entry.getStoreValue(), e);
      return;
    }
  }

  private void paddingData(Map<EventType, Map<String, Map<Long, OqsEngineEntity>>> binlogRowDataMap,
      String tableName, EventType eventType, OqsEngineEntity entity) {
    Map<String, Map<Long, OqsEngineEntity>> eventTypeRowDatas = new HashMap<>();
    if (binlogRowDataMap.get(eventType) == null) {
      binlogRowDataMap.put(eventType, eventTypeRowDatas);
    } else {
      eventTypeRowDatas = binlogRowDataMap.get(eventType);
    }
    Map<Long, OqsEngineEntity> eventTypeTableRowDatas = new HashMap<>();
    if (eventTypeRowDatas.get(tableName) == null) {
      eventTypeRowDatas.put(tableName, eventTypeTableRowDatas);
    } else {
      eventTypeTableRowDatas = eventTypeRowDatas.get(tableName);
    }
    eventTypeTableRowDatas.put(entity.getId(), entity);
  }


  private OqsEngineEntity oneRowParser(Map<String, CanalEntry.Column> columns, boolean isDelete,
      IEntityClass tableEntityClass)
      throws SQLException, JsonProcessingException {
    try {
      long entityClass = BinLogParseUtils.getLongFromColumn(columns, SystemColumn.SYS_ENTITY_CLASS);
      String profile = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.SYS_PROFILE);
      long id = BinLogParseUtils.getLongFromColumn(columns, SystemColumn.ID);

      IEntityClass iEntityClass = tableEntityClass;

      OqsEngineEntity.Builder builder = new OqsEngineEntity.Builder();
      builder.withDeleted(isDelete);
      builder.withEntityClassRef(new EntityClassRef(entityClass, "", "", profile));
      builder.withId(id);

      if (entityClass != tableEntityClass.id()) {
        builder.withFather(tableEntityClass.id());
      } else {
        IEntityClass ec = tableEntityClass.extendEntityClass();
        if (null != ec) {
          builder.withFather(ec.id());
        } else {
          builder.withFather(0);
        }
      }

      builder.withAttribute("id", id);
      /**取出元数据**/
      Map<String, IEntityField> iEntityFieldMap = new HashMap<>();
      engine.describe(iEntityClass, profile).getAllFields().stream().forEach(p -> iEntityFieldMap.put(p.name().replace(".", "_"), p));
      for (CanalEntry.Column column : columns.values()) {
        String name = column.getName();
        String value = column.getValue();
        IEntityField field = iEntityFieldMap.get(name);
        if (name.equals(SystemColumn.SYS_OPERATE_TIME)) {
          //  update time.
          long updateTime = Long.parseLong(column.getValue());
          builder.withUpdateTime(updateTime);
        } else if (name.equals(SystemColumn.SYS_VERSION)) {
          //  update version.
          if (!value.isEmpty()) {
            int version = Integer.parseInt(column.getValue());
            builder.withVersion(version);
          }
        } else if (name.equals(SystemColumn.ID) ||
            name.equals(SystemColumn.SYS_ENTITY_CLASS) ||
            name.equals(SystemColumn.SYS_PROFILE) ||
            name.equals(SystemColumn.SYS_DELETED)) {
          //  do nothing
        } else if (name.equals(SystemColumn.DYNAMIC_FIELD)) {
          builder.withAttributes(attrCollection(iEntityClass, columns));
        } else {
          if (null == field) {
            log.warn("entityField can not be null, column name {}", name);
            continue;
          }
          if (StringUtils.isEmpty(value) || StringUtils.equalsIgnoreCase("null", value)) {
            builder.withAttribute(name, null);
            continue;
          }
          formatFiledValues(builder, field, name, value);
        }
      }
      return builder.build();
    } catch (Exception e) {
      throw e;
    }
  }

  /**
   * 根据字段类型格式化oqsEngineEntity字段值
   *
   * @param builder
   * @param field
   * @param name
   * @param value
   **/
  private void formatFiledValues(OqsEngineEntity.Builder builder, IEntityField field, String name, String value) {
    switch (field.type()) {
      case LONG:
      case DATETIME:
        builder.withAttribute(name, Long.parseLong(value));
        break;
      case BOOLEAN:
        long result = Long.parseLong(value);
        builder.withAttribute(name, result != 0);
        break;
      case DECIMAL:
        builder.withAttribute(name, new BigDecimal(value));
        break;
      case STRINGS:
        try {
          List<String> multiValues = JacksonDefaultMapper.OBJECT_MAPPER.readValue(value, JacksonDefaultMapper.LIST_TYPE_REFERENCE);
          builder.withAttribute(name, multiValues);
        } catch (Throwable throwable) {
          log.error("{}", throwable);
          builder.withAttribute(name, value);
        }
        break;
      default:
        builder.withAttribute(name, value);
        break;
    }
  }

  /**
   * 转换attribute.
   *
   * @param columns 原始数据集.
   * @return 对象键值对.
   */
  private Map<String, Object> attrCollection(IEntityClass entityClass,
      Map<String, CanalEntry.Column> columns)
      throws JsonProcessingException {

    Map<String, Object> result;
    String attrStr = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.DYNAMIC_FIELD);
    if (attrStr.isEmpty()) {
      result = new HashMap<>();
    } else {
      result = attributesToMap(attrStr);
    }
    entityClass.selfFields().stream().filter(IEntityField::isDynamic).forEach(
        f -> {
          if (!result.containsKey(f.name())) {
            result.put(f.name(), null);
          } else if (f.type().equals(FieldType.BOOLEAN)) {
            result.computeIfPresent(
                f.name(),
                (k, s) -> ((int) s) != 0
            );

          }
        }
    );
    return result;
  }

  /**
   * 属性字符串表示解析为实际对象列表.
   *
   * @param attrStr 属性的字符串表示.
   * @return 解析结果.
   * @throws JsonProcessingException JSON解析失败.
   */
  public static Map<String, Object> attributesToMap(String attrStr) throws JsonProcessingException {
    return JacksonDefaultMapper.OBJECT_MAPPER.readValue(attrStr, Map.class);
  }

  private Optional<IEntityClass> foundEntityClassFromTableName(String tableName) {
    String[] tableSplit = tableName.split("_");
    if (tableSplit.length < 3) {
      return Optional.empty();
    }
    String code = tableSplit[2];
    String profile = null;
    if (tableSplit.length > 3) {
      profile = tableSplit[3];
    }
    return engine.loadByCode(code, profile);
  }
}
