package com.xforceplus.ultraman.cdc.processor.impl;


import static com.xforceplus.ultraman.cdc.utils.ThreadPoolExecutorUtils.CORE_POOL_SIZE;
import static com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment.DEL_UID_KEY;
import static com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment.DEL_UNAME_KEY;
import static com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment.GROUP_KEY;
import static com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment.ROOT;

import com.alibaba.otter.canal.protocol.CanalEntry;
import com.alibaba.otter.canal.protocol.CanalEntry.Entry;
import com.alibaba.otter.canal.protocol.CanalEntry.EventType;
import com.alibaba.otter.canal.protocol.CanalEntry.RowData;
import com.alibaba.otter.canal.protocol.Message;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.cdc.adapter.CDCBeforeCallback;
import com.xforceplus.ultraman.cdc.adapter.EngineAdapterService;
import com.xforceplus.ultraman.cdc.context.ParserContext;
import com.xforceplus.ultraman.cdc.dto.ParseResult;
import com.xforceplus.ultraman.cdc.dto.constant.CDCConstant;
import com.xforceplus.ultraman.cdc.processor.DataProcessor;
import com.xforceplus.ultraman.cdc.processor.EventQueue;
import com.xforceplus.ultraman.cdc.utils.BinLogParseUtils;
import com.xforceplus.ultraman.cdc.utils.ThreadPoolExecutorUtils;
import com.xforceplus.ultraman.extensions.cdc.status.StatusService;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.sdk.core.event.EntityCreated;
import com.xforceplus.ultraman.sdk.core.event.EntityDeleted;
import com.xforceplus.ultraman.sdk.core.event.EntityUpdated;
import com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment;
import com.xforceplus.ultraman.sdk.infra.event.EventPublisher;
import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import io.micrometer.core.annotation.Timed;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import scala.Tuple3;

/**
 * Created by justin.xu on 08/2022.
 *
 * @since 1.8
 */

public class DefaultDataProcessor implements DataProcessor {

  final Logger logger = LoggerFactory.getLogger(DefaultDataProcessor.class);

  @Resource
  private EntityClassEngine engine;

  @Autowired(required = false)
  private StatusService statusService;

  @Autowired(required = false)
  private List<CDCBeforeCallback> cdcBeforeCallbacks = new ArrayList<>();

  @Resource
  private EventPublisher publisher;

  @Resource
  private EngineAdapterService engineAdapterService;

  @Resource
  private EventQueue eventQueue;


  @Resource
  private ExecutorService eventThreadPool;
  private ThreadPoolExecutor executor = ThreadPoolExecutorUtils.executor;
  private long writeTimeOut = 600;


  /**
   * 属性字符串表示解析为实际对象列表.
   *
   * @param attrStr 属性的字符串表示.
   * @return 解析结果.
   * @throws JsonProcessingException JSON解析失败.
   */
  public static Map<String, Object> attributesToMap(String attrStr) throws JsonProcessingException {
    return JacksonDefaultMapper.OBJECT_MAPPER.readValue(attrStr, Map.class);
  }

  public ExecutorService getEventThreadPool() {
    return eventThreadPool;
  }

  public void setEventThreadPool(ExecutorService eventThreadPool) {
    this.eventThreadPool = eventThreadPool;
  }

  public void setEngine(EntityClassEngine engine) {
    this.engine = engine;
  }

  public void setStatusService(StatusService statusService) {
    this.statusService = statusService;
  }

  public void setCdcBeforeCallbacks(List<CDCBeforeCallback> cdcBeforeCallbacks) {
    this.cdcBeforeCallbacks = cdcBeforeCallbacks;
  }

  public void setPublisher(EventPublisher publisher) {
    this.publisher = publisher;
  }

  public void setEngineAdapterService(EngineAdapterService engineAdapterService) {
    this.engineAdapterService = engineAdapterService;
  }

  public void setEventQueue(EventQueue eventQueue) {
    this.eventQueue = eventQueue;
  }

  @Timed(
      value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
      percentiles = {0.5, 0.9, 0.99},
      extraTags = {"initiator", "cdc", "action", "consume"}
  )
  @Override
  public boolean onProcess(Message message, int threadBatchSize) throws Exception {
    return messageProcess(message, threadBatchSize) > 0;
  }

  private int dynamicThread(int entries, int threadBatchSize) {
    if (entries <= threadBatchSize) {
      return 1;
    }

    return entries % threadBatchSize == 0 ? entries / threadBatchSize : entries / threadBatchSize + 1;
  }


  private int messageProcess(Message message, int threadBatchSize) {
    try {
      if (message.getEntries().size() == 0) {
        return 0;
      }
      return BinlogToElasticProcess.parseBinlogMessage(message, engine, engineAdapterService, cdcBeforeCallbacks, threadBatchSize);

      /*Map<Integer, Future> futures = new ConcurrentHashMap<>();
      ParserContext parserContext = new ParserContext(message.getId());

      int thread = dynamicThread(message.getEntries().size(), threadBatchSize);

      int lastPos = 0;

      Map<String, OqsEngineEntity> distinctOqsEngineEntities;
      //  不切换线程
      if (thread == 1) {
        distinctOqsEngineEntities = parseCanalEntries(message.getEntries(), parserContext);
      } else {
        threadBatchSize = getThreadBatchSize(threadBatchSize, message.getEntries().size());
        //  使用多线程处理
        for (int i = 0; i < thread; i++) {
          int tempEnd = lastPos + threadBatchSize;
          lastPos = Math.min(tempEnd, message.getEntries().size());
          List<Entry> entries = message.getEntries().subList(i * threadBatchSize, lastPos);

          //  拆分线程并发执行进行binlog转换成oqsEntities处理
          submitThreadHandler(entries, futures, parserContext, i);
        }

        distinctOqsEngineEntities = new HashMap<>();
        List<Integer> threadIds = futures.keySet().stream().collect(Collectors.toList());
        Collections.sort(threadIds);
        for (int id : threadIds) {
          Map<String, OqsEngineEntity> result = (Map<String, OqsEngineEntity>) futures.get(id).get(writeTimeOut, TimeUnit.SECONDS);
          //distinctOqsEngineEntities.putAll(result);
          //  提交elastic执行保存操作
          boolean flag = engineAdapterService.batchUpsertOperation(result.values());
          if (!flag) {
            throw new RuntimeException("Adapter Service return False");
          }
        }
      }

      return distinctOqsEngineEntities.size();
*/
    } catch (Exception e) {
      logger.error("batchId : {}, consume message failed, message : {}", message.getId(), e.getMessage());
      throw e;
    }
  }


  /**
   * 对数据按批次大小进行切分，提交至线程处理
   *
   * @param batchHandlerEntities
   * @param futures
   * @param parserContext
   * @param i
   **/
  private void submitThreadHandler(List<Entry> batchHandlerEntities, Map<Integer, Future> futures, ParserContext parserContext, int i) {
    Future<Map<String, OqsEngineEntity>> future = executor.submit(() -> {
      try {
        return parseCanalEntries(batchHandlerEntities, parserContext);
      } catch (Exception e) {
        throw e;
      }
    });
    futures.put(i, future);
  }


  /**
   * 解析入口函数,对一个批次进行解析.
   *
   * @param entries 完整的批次信息.
   * @return 成功条数.
   */
  private Map<String, OqsEngineEntity> parseCanalEntries(List<CanalEntry.Entry> entries, ParserContext parserContext) {
    ParseResult parseResult = new ParseResult();
    for (CanalEntry.Entry entry : entries) {
      //  不是TransactionEnd/RowData类型数据, 将被过滤
      switch (entry.getEntryType()) {
        case TRANSACTIONBEGIN:
        case TRANSACTIONEND:
          eventHandler(entry.getEntryType());
          break;
        case ROWDATA:
          rowDataParse(entry, parserContext, parseResult);
          break;
        default: {
        }
      }
    }
    //  批次数据整理完毕，开始执行index写操作。
    if (!parseResult.getFinishEntries().isEmpty()) {
      Optional.ofNullable(cdcBeforeCallbacks)
          .orElseGet(Collections::emptyList)
          .forEach(x -> {
            try {
              x.mutate(toMutateEntities(parseResult.getFinishEntries()));
            } catch (Throwable throwable) {
              logger.error("CDC callback ERROR name:{} , ex:{}", x.name(), throwable);
            }
          });
    }
    return parseResult.getFinishEntries();
  }

  private OqsEngineEntity cloneEntity(OqsEngineEntity src) {
    OqsEngineEntity target = new OqsEngineEntity();
    target.setEntityClassRef(src.getEntityClassRef());
    target.setId(src.getId());
    target.setVersion(src.getVersion());

    Map<String, Object> attributes = src.getAttributes();
    try {
      String attr = JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(attributes);
      Map<String, Object> newAttr = JacksonDefaultMapper.OBJECT_MAPPER.readValue(attr, Map.class);
      target.setAttributes(newAttr);
    } catch (Throwable throwable) {
      logger.error("{}", throwable);
    }

    target.setUpdateTime(src.getUpdateTime());
    target.setFather(src.getFather());
    target.setDeleted(src.isDeleted());
    return target;
  }

  /**
   * 对rowData进行解析，rowData为对一张表的CUD操作，记录条数1～N.
   *
   * @param entry         canal对象同步实例.
   * @param parserContext 上下文.
   */
  private void rowDataParse(CanalEntry.Entry entry, ParserContext parserContext,
      ParseResult parseResult) {

    String tableName = entry.getHeader().getTableName();
    if (tableName.isEmpty()) {
      logger.error("batch : {}, table name could not be Null, [{}]",
          parserContext.getBatchId(), entry.getStoreValue());
      return;
    }
    Optional<IEntityClass> entityClassOp = foundEntityClassFromTableName(tableName);
    if (!entityClassOp.isPresent()) {
      logger.error("batch : {}, entityClass could not be Null, [{}]",
          parserContext.getBatchId(), entry.getStoreValue());
      return;
    }

    CanalEntry.RowChange rowChange = null;
    try {
      rowChange = CanalEntry.RowChange.parseFrom(entry.getStoreValue());
    } catch (Exception e) {
      logger.error("batch : {}, parse entry value failed, [{}], [{}]",
          parserContext.getBatchId(), entry.getStoreValue(), e.getMessage());
      return;
    }

    CanalEntry.EventType eventType = rowChange.getEventType();
    //  遍历RowData
    for (CanalEntry.RowData rowData : rowChange.getRowDatasList()) {

      //  获取一条完整的更新前的columns
      Map<String, CanalEntry.Column> beforeColumnsMap = new HashMap();
      rowData.getBeforeColumnsList().forEach(column -> beforeColumnsMap.put(column.getName(), column));
      //  获取一条完整的更新后的columns
      Map<String, CanalEntry.Column> afterColumnsMap = new HashMap();
      rowData.getAfterColumnsList().forEach(column -> afterColumnsMap.put(column.getName(), column));

      OqsEngineEntity beforeEntity = null;
      OqsEngineEntity afterEntity = null;

      SystemAttachment systemAttachment = null;
      try {
        switch (eventType) {
          case INSERT: {
            afterEntity = oneRowParser(afterColumnsMap, false, entityClassOp.get(), parseResult);
            systemAttachment = getAttachment(afterColumnsMap);
            mergeEntityToResult(parseResult, afterEntity);
            break;
          }
          case UPDATE: {
            try {
              beforeEntity = oneRowParser(beforeColumnsMap, false, entityClassOp.get(), parseResult);
            } catch (Exception e) {
              logger.warn(
                  "beforeEntity in update is null, will be ignore, but it will influence event");
            }
            afterEntity = oneRowParser(afterColumnsMap, false, entityClassOp.get(), parseResult);
            systemAttachment = getAttachment(afterColumnsMap);
            mergeEntityToResult(parseResult, afterEntity);
            break;
          }
          case DELETE: {
            beforeEntity = oneRowParser(beforeColumnsMap, true, entityClassOp.get(), parseResult);
            systemAttachment = getAttachment(beforeColumnsMap);
            mergeEntityToResult(parseResult, beforeEntity);
            break;
          }
        }

        OqsEngineEntity beforeClone = null;
        OqsEngineEntity afterClone = null;

        if (beforeEntity != null) {
          beforeClone = cloneEntity(beforeEntity);
        }

        if (afterEntity != null) {
          afterClone = cloneEntity(afterEntity);
        }

        if (systemAttachment == null) {
          systemAttachment = new SystemAttachment();
        }
        eventHandler(eventType, systemAttachment, entityClassOp.get(), beforeClone, afterClone);
      } catch (Exception e) {
        //  对于数据层面的错误，只记录，不回滚批次.
        logger.warn("parse entity error, message : {}", e.getMessage());
      }
    }
  }

  private SystemAttachment getAttachment(Map<String, CanalEntry.Column> columns) {
    String attrStr = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.DYNAMIC_FIELD);
    SystemAttachment attachment = new SystemAttachment();
    try {
      if (attrStr.isEmpty()) {
        return attachment;
      } else {
        Map<String, Object> map = attributesToMap(attrStr);
        Object o = map.get(ROOT);
        if (o != null) {
          Map<String, Object> attachmentMap = (Map<String, Object>) o;
          Object grouped = attachmentMap.get(GROUP_KEY);
          if (grouped != null) {
            attachment.setGrouped(Integer.parseInt(grouped.toString()));
          }

          Object delUid = attachmentMap.get(DEL_UID_KEY);
          if (delUid != null) {
            attachment.setDelUId(Long.parseLong(delUid.toString()));
          }

          Object delUname = attachmentMap.get(DEL_UNAME_KEY);
          if (delUname != null) {
            attachment.setDelUname(delUname.toString());
          }
        }
        return attachment;
      }
    } catch (Throwable throwable) {
      return attachment;
    }
  }

  private Optional<IEntityClass> foundEntityClassFromTableName(String tableName) {
    String[] tableSplit = tableName.split("_");
    if (tableSplit.length < 3) {
      return Optional.empty();
    }

    String code = tableSplit[2];
    String profile = null;
    if (tableSplit.length > 3) {
      profile = tableSplit[3];
    }

    return engine.loadByCode(code, profile);
  }

  private List<OqsEngineEntity> toMutateEntities(Map<String, OqsEngineEntity> entityMap) {
    Map<Long, OqsEngineEntity> mutates = new HashMap<>();

    entityMap.forEach(
        (k, v) -> {
          OqsEngineEntity engineEntity = mutates.get(v.getId());
          if (null != engineEntity) {
            engineEntity.getAttributes().putAll(v.getAttributes());
            if (v.isDeleted()) {
              engineEntity.setDeleted(true);
              if (v.getFather() > 0) {
                engineEntity.setFather(v.getFather());
              }
            }

            engineEntity.setUpdateTime(v.getUpdateTime());
          } else {
            mutates.put(v.getId(), v);
          }
        }
    );

    return new ArrayList<>(mutates.values());
  }

  //  将合并值到attributes中
  private void mergeEntityToResult(ParseResult parseResult, OqsEngineEntity entity) {

    String key = entity.getId() + "@@" + entity.getTable();

    OqsEngineEntity target = parseResult.getFinishEntries().get(key);
    if (null == target) {
      parseResult.getFinishEntries().put(key, entity);
      return;
    }
    target.getAttributes().putAll(entity.getAttributes());

    if (entity.isDeleted()) {
      target.setDeleted(true);
    }

    if (entity.getFather() > 0 && target.getFather() == 0) {
      target.setFather(entity.getFather());
    }

    target.setUpdateTime(entity.getUpdateTime());
  }

  private OqsEngineEntity oneRowParser(Map<String, CanalEntry.Column> columns, boolean isDelete,
      IEntityClass tableEntityClass,
      ParseResult parseResult)
      throws SQLException, JsonProcessingException {
    try {
      long entityClass = BinLogParseUtils.getLongFromColumn(columns, SystemColumn.SYS_ENTITY_CLASS);
      String profile = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.SYS_PROFILE);
      long id = BinLogParseUtils.getLongFromColumn(columns, SystemColumn.ID);

      IEntityClass iEntityClass = tableEntityClass;

      OqsEngineEntity.Builder builder = new OqsEngineEntity.Builder();
      builder.withDeleted(isDelete);
      builder.withEntityClassRef(new EntityClassRef(entityClass, "", "", profile));
      builder.withId(id);

      if (entityClass != tableEntityClass.id()) {
        builder.withFather(tableEntityClass.id());
      } else {
        IEntityClass ec = tableEntityClass.extendEntityClass();
        if (null != ec) {
          builder.withFather(ec.id());
        } else {
          builder.withFather(0);
        }
      }

      builder.withAttribute("id", id);
      /**取出元数据**/
      Map<String, IEntityField> iEntityFieldMap = new HashMap<>();
      engine.describe(iEntityClass, profile).getAllFields().stream().forEach(p -> iEntityFieldMap.put(p.name().replace(".", "_"), p));

      for (CanalEntry.Column column : columns.values()) {
        if (column.getName().equals(SystemColumn.SYS_OPERATE_TIME)) {
          //  update time.
          long updateTime = Long.parseLong(column.getValue());
          builder.withUpdateTime(updateTime);
        } else if (column.getName().equals(SystemColumn.SYS_VERSION)) {
          //  update version.
          if (!column.getValue().isEmpty()) {
            int version = Integer.parseInt(column.getValue());
            builder.withVersion(version);
          }
        } else if (column.getName().equals(SystemColumn.ID) ||
            column.getName().equals(SystemColumn.SYS_ENTITY_CLASS) ||
            column.getName().equals(SystemColumn.SYS_PROFILE) ||
            column.getName().equals(SystemColumn.SYS_DELETED)) {
          //  do nothing
        } else if (column.getName().equals(SystemColumn.DYNAMIC_FIELD)) {
          builder.withAttributes(attrCollection(iEntityClass, columns));
        } else {
          String name = column.getName();
          String value = column.getValue();
         /* IEntityField field = engine.describe(iEntityClass, profile).getAllFields().stream()
              .filter(p -> p.name().replace(".", "_").equalsIgnoreCase(name))
              .findFirst().orElse(null);*/
          IEntityField field = iEntityFieldMap.get(name);
          if (null == field) {
            logger.warn("entityField can not be null, column name {}", name);
            continue;
          }
          builder.withAttribute(name, value);
       /* if (field.type() == FieldType.LONG || field.type() == FieldType.DATETIME) {
              builder.withAttribute(name, Long.parseLong(value));
            } else if (field.type() == FieldType.BOOLEAN) {
              long result = Long.parseLong(value);
              builder.withAttribute(name, result != 0);
            } else if (field.type() == FieldType.DECIMAL) {
              builder.withAttribute(name, new BigDecimal(value));
            } else
            if (field.type() == FieldType.STRINGS) {
              try {
                List<String> multiValues = JacksonDefaultMapper.OBJECT_MAPPER.readValue(value, JacksonDefaultMapper.LIST_TYPE_REFERENCE);
                builder.withAttribute(name, multiValues);
              } catch (Throwable throwable) {
                logger.error("{}", throwable);
                builder.withAttribute(name, value);
              }
            } else {
              builder.withAttribute(name, value);
            }
        }*/
        }
      }

      if (parseResult.getStartId() == CDCConstant.NOT_INIT_START_ID) {
        parseResult.setStartId(id);
      }

      return builder.build();
    } catch (Exception e) {
      throw e;
    }
  }

  private void eventHandler(CanalEntry.EntryType entryType) {

  }

  private void eventHandler(CanalEntry.EventType eventType, SystemAttachment attachment, IEntityClass entityClass, OqsEngineEntity before,
      OqsEngineEntity after) {
    //TO publish event
    switch (eventType) {
      case INSERT: {
        publishCreatedEvent(entityClass, attachment, after);
        break;
      }
      case UPDATE: {
        if (attachment.getDelUname() != null && attachment.getDelUId() > 0) {
          //skip del event
          break;
        }
        publishUpdatedEvent(entityClass, attachment, before, after);
        break;
      }
      case DELETE: {
        publishDeletedEvent(entityClass, attachment, before);
        break;
      }
      default:
        //do nothing
    }
  }

  /**
   * 转换attribute.
   *
   * @param columns 原始数据集.
   * @return 对象键值对.
   */
  private Map<String, Object> attrCollection(IEntityClass entityClass, Map<String, CanalEntry.Column> columns) throws JsonProcessingException {

    Map<String, Object> result;
    String attrStr = BinLogParseUtils.getStringFromColumn(columns, SystemColumn.DYNAMIC_FIELD);
    if (attrStr.isEmpty()) {
      result = new HashMap<>();
    } else {
      result = attributesToMap(attrStr);
    }
    entityClass.selfFields().stream().filter(IEntityField::isDynamic).forEach(
        f -> {
          if (!result.containsKey(f.name())) {
            result.put(f.name(), null);
          } else if (f.type().equals(FieldType.BOOLEAN)) {
            result.computeIfPresent(
                f.name(),
                (k, s) -> ((int) s) != 0
            );

          }
        }
    );
    return result;
  }

  private void fillAttachment(SystemAttachment systemAttachment, Map<String, Object> context) {
    context.put("attachment", systemAttachment);
  }

  private void publishDeletedEvent(IEntityClass entityClass, SystemAttachment attachment, OqsEngineEntity b) {
    eventQueue.feedDelete(attachment, b)
        .thenAcceptOnce(x -> {
          if (publisher != null) {
            Map<String, Object> context = new HashMap<>();
            fillAttachment(attachment, context);
            publisher.publishTransactionEvent(new EntityDeleted(entityClass.code()
                , x.getId(), x.getAttributes(), false, context));
          }
        }, eventThreadPool);
  }

  private void publishUpdatedEvent(IEntityClass entityClass, SystemAttachment attachment, OqsEngineEntity b, OqsEngineEntity a) {
    eventQueue.feedUpdate(attachment, b, a)
        .thenAcceptOnce(x -> {
          if (publisher != null) {
            OqsEngineEntity before = x._1;
            OqsEngineEntity after = x._2;
            Map<String, Object> context = new HashMap<>();
            fillAttachment(attachment, context);
            EntityUpdated entityUpdated = new EntityUpdated(entityClass.code()
                , before.getId(), before.getAttributes(), after.getAttributes(), false, context);
            int version = after.getVersion();
            entityUpdated.setVer(version);
            publisher.publishTransactionEvent(entityUpdated);
          }
        }, eventThreadPool);
  }

  private void publishCreatedEvent(IEntityClass entityClass, SystemAttachment attachment, OqsEngineEntity after) {
    eventQueue.feedCreate(attachment, after)
        .thenAcceptOnce(x -> {
          if (publisher != null) {
            Map<String, Object> context = new HashMap<>();
            fillAttachment(attachment, context);
            publisher.publishTransactionEvent(new EntityCreated(entityClass.code()
                , x.getId(), x.getAttributes(), false, context));
          }
        }, eventThreadPool);
  }

}
