package com.xforceplus.ultraman.cdc.core.local.embed;

import com.alibaba.otter.canal.instance.manager.CanalInstanceWithManager;
import com.alibaba.otter.canal.instance.manager.model.Canal;
import com.alibaba.otter.canal.instance.manager.model.CanalParameter;
import com.alibaba.otter.canal.protocol.CanalEntry;
import com.alibaba.otter.canal.protocol.ClientIdentity;
import com.alibaba.otter.canal.protocol.Message;
import com.alibaba.otter.canal.server.embedded.CanalServerWithEmbedded;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.xforceplus.ultraman.cdc.reader.CDCPropertyPackage;
import io.vavr.Tuple2;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.xforceplus.ultraman.cdc.reader.CanalPropertiesReader;

/**
 * Created by justin.xu on 08/2022.
 *
 * @since 1.8
 */
public class EmbedDataProducer implements SourceDataProducer {

    final Logger logger = LoggerFactory.getLogger(EmbedDataProducer.class);

    private CanalServerWithEmbedded canalServerWithEmbedded;

    private CDCPropertyPackage cdcPropertyPackage;

    private Map<String, Tuple2<ClientIdentity, CanalPropertiesReader>> clientIdentities;

    public EmbedDataProducer(CDCPropertyPackage cdcPropertyPackage) {
        this.cdcPropertyPackage = cdcPropertyPackage;
        canalServerWithEmbedded = CanalServerWithEmbedded.instance();
        this.clientIdentities = new HashMap<>();
    }

    @Override
    public void init() {
        //  降级->非降级(需要删除当前的meta.data文件）.
        for (CanalPropertiesReader reader : cdcPropertyPackage.readers()) {
            if (reader.isResetMeta()) {
                try {
                    Files.deleteIfExists(Paths.get(reader.fullLogFile()));
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }

            canalServerWithEmbedded.setCanalInstanceGenerator(destination -> {
                Canal canal = buildCanal(reader);
                return new CanalInstanceWithManager(canal, reader.getFilter());
            });
        }

        canalServerWithEmbedded.start();
        for (CanalPropertiesReader reader : cdcPropertyPackage.readers()) {
            canalServerWithEmbedded.start(reader.getDestination());
        }
    }

    @Override
    public void destroy() {
        for (Map.Entry<String, Tuple2<ClientIdentity, CanalPropertiesReader>> entry : clientIdentities.entrySet()) {
            canalServerWithEmbedded.unsubscribe(entry.getValue()._1());

            logger.info("cdc-instance destination {} is shutdown.", entry.getValue()._1().getDestination());
        }

        canalServerWithEmbedded.stop();
    }

    @Override
    public boolean singletonRegistry(ClientIdentity client) {

        Optional<CanalPropertiesReader> canalPropertiesReader =
                cdcPropertyPackage.find(client.getDestination());

        if (clientIdentities.containsKey(client.getDestination()) || !canalPropertiesReader.isPresent()) {
            return false;
        }

        clientIdentities.put(client.getDestination(), new Tuple2<>(client, canalPropertiesReader.get()));

        canalServerWithEmbedded.subscribe(client);
        return true;
    }

    @Override
    public Collection<Tuple2<ClientIdentity, CanalPropertiesReader>> clientIdentityWithReader() {
        return clientIdentities.values();
    }

    @Override
    public Message onMessage(ClientIdentity clientIdentity, int batchSize) {
        //  当处于未被订阅状态时，将不会返回message.
        Message message = null;
        try {
            message = canalServerWithEmbedded.getWithoutAck(clientIdentity, batchSize);

            return toConsumeMessage(message);

        } catch (Exception e) {
            logger.error("producer onMessage error, reason : {}", e.getMessage());
            //  解析到一条不可被消费的数据、将直接跳过该记录.
            //  所有的rollback必须来自consumer.
            if (null != message && message.getId() > 0) {
                ack(clientIdentity, message.getId());
            }
        }
        return null;
    }

    @Override
    public void ack(ClientIdentity clientIdentity, long batchId) {
        canalServerWithEmbedded.ack(clientIdentity, batchId);
    }

    @Override
    public void rollback(ClientIdentity clientIdentity, long batchId) {
        canalServerWithEmbedded.rollback(clientIdentity, batchId);
    }

    private Message toConsumeMessage(Message message) throws InvalidProtocolBufferException {
        Message result = null;
        if (message.isRaw() && !message.getRawEntries().isEmpty()) {
            result = new Message(message.getId());
            for (ByteString byteString : message.getRawEntries()) {
                result.addEntry(CanalEntry.Entry.parseFrom(byteString));
            }
            result.setRaw(false);

        } else if (!message.getEntries().isEmpty()) {
            result = message;
        }

        return result;
    }


    private Canal buildCanal(CanalPropertiesReader propertiesReader) {
        Canal canal = new Canal();
        canal.setId(1L);
        canal.setName(propertiesReader.getDestination());
        canal.setDesc(propertiesReader.getDesc());

        CanalParameter parameter = new CanalParameter();

        parameter.setMetaMode(CanalParameter.MetaMode.LOCAL_FILE);
        parameter.setDataDir(propertiesReader.getDataDir());
        parameter.setIndexMode(CanalParameter.IndexMode.MEMORY_META_FAILBACK);

        parameter.setMemoryStorageBufferSize(propertiesReader.getMemoryStorageBufferSize());
        parameter.setStorageBatchMode(CanalParameter.BatchMode.ITEMSIZE);
        parameter.setMemoryStorageBufferSize(propertiesReader.getMemoryStorageBufferSize());
        parameter.setReceiveBufferSize(propertiesReader.getMemoryStorageBufferSize());
        parameter.setSourcingType(CanalParameter.SourcingType.MYSQL);
        parameter.setDbAddresses(Arrays.asList(new InetSocketAddress(propertiesReader.getMasterHost(), propertiesReader.getMasterPort()),
                new InetSocketAddress(propertiesReader.getMasterHost(), propertiesReader.getMasterPort())));
        parameter.setDbUsername(propertiesReader.getMasterUser());
        parameter.setDbPassword(propertiesReader.getMasterPasswd());

        parameter.setSlaveId(propertiesReader.getSlaveId());

        canal.setCanalParameter(parameter);
        return canal;
    }
}
