package com.xforceplus.ultraman.cdc.core.remote.runner;

import static com.xforceplus.ultraman.cdc.core.remote.context.RunningStatus.RUN;
import static com.xforceplus.ultraman.cdc.core.remote.context.RunningStatus.TRY_STOP;

import com.alibaba.otter.canal.protocol.Message;
import com.xforceplus.ultraman.cdc.core.remote.connect.CDCConnector;
import com.xforceplus.ultraman.cdc.core.remote.context.RunnerContext;
import com.xforceplus.ultraman.cdc.core.remote.context.RunningStatus;
import com.xforceplus.ultraman.cdc.dto.enums.CDCStatus;
import com.xforceplus.ultraman.cdc.processor.DataProcessor;
import com.xforceplus.ultraman.cdc.reader.CanalPropertiesReader;
import com.xforceplus.ultraman.cdc.utils.ThreadPoolExecutorUtils;
import com.xforceplus.ultraman.cdc.utils.TimeWaitUtils;
import com.xforceplus.ultraman.cdc.utils.VerifyThreadParamUtils;
import java.sql.SQLException;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import io.vavr.Tuple2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by justin.xu on 06/2023.
 *
 * @since 1.8
 */
public class DefaultCDCConsumer implements CDCConsumer {

  final Logger logger = LoggerFactory.getLogger(DefaultCDCConsumer.class);
  public static final int MESSAGE_GET_WARM_INTERVAL = 100;

  //  当前CDC 中没有同步message时的休眠间隔 (默认5毫秒)
  public static final int FREE_MESSAGE_WAIT_IN_MS = 5;


  private RunnerContext context;

  private DataProcessor dataProcessor;

  private CDCConnector connector;

  private Map<String, CDCStatus> cdcStatusMap;
  private CanalPropertiesReader canalPropertiesReader;


  public DefaultCDCConsumer(CDCConnector connector, DataProcessor dataProcessor, Map<String, CDCStatus> cdcStatusMap,
      CanalPropertiesReader canalPropertiesReader) {
    this.connector = connector;
    this.dataProcessor = dataProcessor;
    this.context = new RunnerContext();
    this.cdcStatusMap = cdcStatusMap;
    this.canalPropertiesReader = canalPropertiesReader;
  }

  @Override
  public void init() {
    context.setRunningStatus(RUN);
    connector.init();
  }

  @Override
  public void destroy() {
    context.setRunningStatus(TRY_STOP);

    int loopTime = 10;
    while (loopTime > 0 && !context.getRunningStatus().equals(RunningStatus.STOP_SUCCESS)) {
      loopTime--;
      TimeWaitUtils.wakeupAfter(1, TimeUnit.SECONDS);
    }
    connector.destroy();
  }


  @Override
  public void execute() {

    logger.info("connector {} is start executing...", connector.name());

    while (true) {
      if (needTerminate()) {
        break;
      }

      try {
        //  连接
        connector.open();

        if (context.getBatchId() > 0) {
          connector.rollback();
        }
        cdcStatusMap.put(connector.name(), CDCStatus.CONNECTED);
      } catch (Exception e) {
        context.incrementContinuesConnectFails();

        if (connector.isMaxRetry(context.getContinuesConnectFails())) {
          connector.close();

          connector.init();

          context.resetContinuesConnectFails();
        } else {
          cdcStatusMap.put(connector.name(), CDCStatus.DIS_CONNECTED);
          closeAndCallBackError(CDCStatus.DIS_CONNECTED,
              String.format("[cdc-runner] canal-server connection error, %s", e.getMessage()));
        }

        continue;
      }

      context.resetContinuesConnectFails();

      try {
        //  消费
        consume();
      } catch (Exception e) {
        cdcStatusMap.put(connector.name(), CDCStatus.CONSUME_FAILED);
        closeAndCallBackError(CDCStatus.CONSUME_FAILED,
            String.format("[cdc-runner] canal-client consume error, %s", e.getMessage()));
      }
    }

    logger.info("connector {} is finish executing...", connector.name());
  }

  @Override
  public Tuple2<String, CDCStatus> namedCdcStatus() {
    return new Tuple2<>(connector.name(), context.getCdcStatus());
  }

  /**
   * 消费.
   */
  private void consume() throws SQLException {
    while (true) {
      //  服务被终止
      if (context.getRunningStatus().shouldStop()) {
        context.setRunningStatus(RunningStatus.STOP_SUCCESS);
        break;
      }
      executeOneBatch(connector);
    }
  }


  private void executeOneBatch(CDCConnector connector) throws SQLException {
    Message message = null;
    long batchId;
    long start = System.currentTimeMillis();
    try {
      //获取指定数量的数据
      message = connector.getMessageWithoutAck();

      long duration = System.currentTimeMillis() - start;

      batchId = message.getId();

      if (duration > MESSAGE_GET_WARM_INTERVAL) {
        logger.debug(
            "[batchProcess] read message from canal server use too much times, use timeMs : {}, batchId : {}",
            duration, message.getId());
      }
    } catch (Exception e) {
      //  未获取到数据,回滚
      connector.rollback();
      String error = String.format("read message from canal server error, %s", e.getMessage());
      logger.error("[batchProcess] {}", error);
      throw new SQLException(error);
    }

    //  当synced标志位设置为True时，表示后续的操作必须通过最终一致性操作保持成功
    try {
      boolean isNotEmptyBatch = false;
      long consumerStart = System.currentTimeMillis();
      if (batchId != 0 || message.getEntries().size() > 0) {
        //logger.info("远程模式CDC读取数据量：{}", (message.getEntries().size()));

        isNotEmptyBatch = dataProcessor.onProcess(message);
      }

      connector.ack(batchId);

      if (!isNotEmptyBatch) {
        //  没有新的同步信息，睡眠5ms进入下次轮训
        TimeWaitUtils.wakeupAfter(FREE_MESSAGE_WAIT_IN_MS, TimeUnit.MILLISECONDS);
      } else {
        long consumerEnd = System.currentTimeMillis();

        logger.info("[batchProcess] metrics total use time {}, consumer use time {},rows{}", consumerEnd - start,
            consumerEnd - consumerStart, message.getEntries().size());
      }
    } catch (Exception e) {
      String error = "consume message error";

      //  当未执行到最终必须成功时,需进行rollback
      connector.rollback();

      logger.error("[batchProcess] consume batch error, connection will reset..., message : {}, {}", error,
          e.getMessage());
      throw new SQLException(error);
    }
  }

  private void closeAndCallBackError(CDCStatus cdcStatus, String errorMessage) {

    connector.close();
    logger.error(errorMessage);

    //  这里将进行睡眠->同步错误信息->进入下次循环
    TimeWaitUtils.wakeupAfter(TimeWaitUtils.RECONNECT_WAIT_IN_SECONDS, TimeUnit.SECONDS);

  }

  private boolean needTerminate() {
    if (context.getRunningStatus().shouldStop()) {
      context.setRunningStatus(RunningStatus.STOP_SUCCESS);
      return true;
    }
    return false;
  }
}
