package com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl;

import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityValue;
import com.xforceplus.ultraman.oqsengine.pojo.dto.values.IValue;

import java.io.Serializable;
import java.util.*;
import java.util.function.Predicate;

/**
 * Entity实体值对象.
 *
 * @author wangzheng
 * @version 1.0 2020/3/26 15:10
 */
public class EntityValue implements IEntityValue, Cloneable, Serializable {
    /**
     * 数据id
     */
    private long id;

    /**
     * Entity的值集合
     */
    private Map<Long, IValue> values;


    public EntityValue(long id) {
        this.id = id;
    }

    @Override
    public long id() {
        return id;
    }

    @Override
    public void restId(long id) {
        this.id = id;
    }

    @Override
    public Optional<IValue> getValue(String fieldName) {
        if (values == null) {
            return Optional.empty();

        }

        for (IValue v : values.values()) {
            if (v.getField().name().equals(fieldName)) {
                return Optional.of(v);
            }
        }

        return Optional.empty();
    }

    @Override
    public Optional<IValue> getValue(long fieldId) {
        return values.entrySet().stream().filter(x -> x.getKey() == fieldId)
                .map(Map.Entry::getValue).findFirst();
    }

    @Override
    public IEntityValue addValue(IValue value) {
        lazyInit();

        values.put(value.getField().id(), value);
        return this;
    }

    @Override
    public Collection<IValue> values() {
        if (values == null) {
            return Collections.emptyList();
        }
        return values.values();
    }

    @Override
    public IEntityValue addValues(Collection<IValue> values) {
        lazyInit();
        values.stream().forEach(v -> {
            this.values.put(v.getField().id(), v);

        });
        return this;
    }

    @Override
    public IValue remove(IEntityField field) {
        lazyInit();

        return values.remove(field.id());
    }

    @Override
    public void filter(Predicate<? super IValue> predicate) {
        if (values != null) {
            values.entrySet().removeIf(entry -> !predicate.test(entry.getValue()));
        }
    }

    @Override
    public IEntityValue clear() {
        if (values != null) {
            values.clear();
        }
        return this;
    }

    @Override
    public Object clone() throws CloneNotSupportedException {
        EntityValue cloneValue = new EntityValue(id);
        cloneValue.addValues(values());
        return cloneValue;
    }

    private void lazyInit() {
        if (this.values == null) {
            // 这里为了保存顺序为加入顺序.
            this.values = new LinkedHashMap<>();
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof EntityValue)) {
            return false;
        }
        EntityValue that = (EntityValue) o;
        return id == that.id && equalsValues(that);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, values);
    }

    @Override
    public String toString() {
        return "EntityValue{" +
                "id=" + id +
                ", values=" + (values != null ? values.toString() : "NULL") +
                '}';
    }

    // 比较两个 map.
    private boolean equalsValues(EntityValue that) {
        Map<Long, IValue> thatValues = that.values;
        if (this.values == thatValues) {
            return true;
        }

        if (thatValues.size() != this.values.size()) {
            return false;
        }

        IValue thisValue;
        IValue thatValue;
        for (Long id : this.values.keySet()) {
            thisValue = this.values.get(id);
            thatValue = thatValues.get(id);

            if (!thisValue.equals(thatValue)) {
                return false;
            }
        }

        return true;
    }
}
