package com.xforceplus.ultraman.oqsengine.pojo.reader.record;

import com.xforceplus.ultraman.bocp.gen.typed.TypedField;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.pojo.dto.values.IValue;
import com.xforceplus.ultraman.oqsengine.pojo.reader.converter.LocalDateTimeConverter;
import com.xforceplus.ultraman.oqsengine.pojo.utils.ConvertHelper;
import com.xforceplus.ultraman.oqsengine.pojo.utils.PrettyPrinter;
import com.xforceplus.ultraman.oqsengine.pojo.utils.PropertyHelper;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.apache.commons.beanutils.BeanUtilsBean;
import org.apache.commons.beanutils.FluentPropertyBeanIntrospector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static com.xforceplus.ultraman.oqsengine.pojo.utils.PropertyHelper.REL_DELIMITER;
import static com.xforceplus.ultraman.oqsengine.pojo.utils.PropertyHelper.REL_PREFIX;

/**
 * Record
 * <p>
 * field and Row
 *
 * @author admin
 */
public class GeneralRecord implements Record {

    private Logger log = LoggerFactory.getLogger(GeneralRecord.class);

    private final Object[] values;

    private final IEntityField[] fields;

    private final Boolean isEmpty;

    private Long id;

    private Long parentId;

    private Long typeId;

    private Integer version;

    /**
     * cross all context
     */
    public static Record empty() {
        return new GeneralRecord();
    }

    private GeneralRecord() {
        isEmpty = true;
         values = null;
        fields = null;
    };

    @Deprecated
    public GeneralRecord(Collection<? extends IEntityField> fields) {
        this(fields, null);
    }

    public GeneralRecord(Collection<? extends IEntityField> fields, Integer version) {

        int size = fields.size();

        this.fields = new IEntityField[fields.size()];
        this.values = new Object[size];

        int i = 0;
        for (IEntityField field : fields) {
            this.fields[i] = field;
            i++;
        }

        isEmpty = false;
        this.version = version;
    }

    @Override
    public Long getId() {
        return id;
    }

    @Override
    public Long getTypeId() {
        return typeId;
    }

    @Override
    public void setTypeId(Long typeId) {
        this.typeId = typeId;
    }

    @Override
    public void setId(Long id) {
        this.id = id;
    }

    @Override
    public Long getParentId() {
        return parentId;
    }

    @Override
    public void setParentId(Long parentId) {
        this.parentId = parentId;
    }

    /**
     * object cannot be null
     * toMap
     *
     * @param fieldName
     * @return
     */
    @Override
    public Optional<Object> get(String fieldName) {
        return field(fieldName).flatMap(this::get);
    }

    @Override
    public Object get(int index) {
        if (index >= values.length) {
            log.error("OutOfBound exception for visit index:{} and return null!", index);
            return null;
        }
        return values[index];
    }

    private Optional<IEntityField> field(String fieldName) {
        if (fieldName == null) {
            return null;
        }

        IEntityField fieldMatch = null;

        for (IEntityField f : fields) {
            if (fieldName.equals(f.name())) {
                if (fieldMatch == null) {
                    fieldMatch = f;
                } else {
                    log.info("Ambiguous match found for "
                            + fieldName + ". Both " + fieldMatch + " and " + f + " match.");
                }
            }
        }

        return Optional.ofNullable(fieldMatch);
    }

    /**
     * find
     *
     * @param field
     * @return
     */
    private final int indexOf(IEntityField field) {

        if (field != null) {
            int size = fields.length;

            for (int i = 0; i < size; i++) {
                if (fields[i] == field) {
                    return i;
                }
            }

            for (int i = 0; i < size; i++) {
                if (fields[i].equals(field)) {
                    return i;
                }
            }
        }

        return -1;
    }

    @Override
    public Optional<Object> get(IEntityField field) {

        int index = indexOf(field);

        if (index > -1 && index < values.length) {
            return Optional.ofNullable(values[index]);
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<IValue> getTypedValue(String fieldName) {
        return field(fieldName).flatMap(this::getTypedValue);
    }

    @Override
    public Optional<IValue> getTypedValue(IEntityField field) {
        return get(field)
                .flatMap(x -> field
                        .type()
                        .toTypedValue(field, x.toString()));
    }

    @Override
    public <T> Optional<T> get(String fieldName, Class<? extends T> type) {
        return field(fieldName)
                .flatMap(this::get)
                .map(x -> (T) x);
    }

    @Override
    public <T> Optional<T> get(IEntityField field, Class<? extends T> type) {
        return get(field)
                .map(x -> (T) x);
    }

    @Override
    public <T> Optional<T> get(TypedField<T> typedField) {

        Optional<Object> o = this.get(typedField.code());

        if(o.isPresent()){
            if(o.get().getClass().isAssignableFrom(typedField.getClass())){
                return o.map(x -> (T)x);
            }else{
                return o.map(x -> (T) ConvertHelper.bean.getConvertUtils().convert(x.toString(), typedField.type()));
            }
        }

        return Optional.empty();
    }

    @Override
    public <T> void set(TypedField<T> typedField, T t) {
        this.set(typedField.code(), t);
    }

    @Override
    public void set(String fieldName, Object t) {
        field(fieldName).ifPresent(x -> set(x, t));
    }

    @Override
    public void set(IEntityField field, Object t) {
        int index = indexOf(field);

        if (index > -1 && index < values.length) {
            values[index] = t;
        } else {
            log.warn("{} is not present", field.name());
        }
    }

    @Override
    public void fromMap(Map<String, Object> map) {
        map.forEach(this::set);
    }

    @Override
    public void setTypedValue(IValue iValue) {

        if (iValue != null
                && iValue.getValue() != null
                && iValue.getField() != null) {
            set(iValue.getField(), iValue.getValue());
        }
    }

    /**
     * change to targetClass
     * @param targetClass
     * @param <T>
     * @return
     */
    @Override
    public <T> T into(Class<T> targetClass) {
        try {
            Object o = targetClass.newInstance();

            Map<String, Object> map = this.toMap(Collections.emptySet());

            //TODO
            map.forEach((k,v) -> {
                if(v != null) {
                    String s = PropertyHelper.convertToCamel(k);
                    try {
                        ConvertHelper.bean.setProperty(o, s, v);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });

            return (T)o;
        } catch (InstantiationException | IllegalAccessException e) {
            e.printStackTrace();
            return null;
        }
    }

    @Override
    public Stream<Tuple2<IEntityField, Object>> stream() {
        return IntStream.range(0, fields.length).mapToObj(i -> Tuple.of(fields[i], values[i]));
    }

    @Override
    public Stream<Tuple2<IEntityField, Object>> stream(Set<String> filterName) {
        return IntStream.range(0, fields.length)
                .mapToObj(i -> {
                    String name = fields[i].name();
                    if (filterName != null && !filterName.isEmpty()) {
                        if (filterName.contains(name)) {
                            return Tuple.of(fields[i], values[i]);
                        }
                        return null;
                    } else {
                        return Tuple.of(fields[i], values[i]);
                    }
                }).filter(Objects::nonNull);
    }

    @Override
    public Stream<Tuple2<IEntityField, Object>> stream(List<String> orderedKeys) {
        if(orderedKeys == null || orderedKeys.isEmpty()){
            return this.stream();
        }

        return orderedKeys.stream().map(x -> {
            Optional<IEntityField> fieldOp = field(x);
            return fieldOp.map(entityField -> Tuple.of(entityField, get(entityField).orElse(null))).orElse(null);
        }).filter(Objects::nonNull);
    }

    /**
     * filter may contains "_" but return map should combine values;
     *
     * @param filterName
     * @return
     */
    @Override
    public Map<String, Object> toMap(Set<String> filterName) {

        Map<String, Object> map = new HashMap<>(values.length + 1);

        IntStream.range(0, values.length)
                .forEach(i -> {

                    String name = fields[i].name();
                    if (filterName != null && !filterName.isEmpty()) {
                        if (isInSet(filterName, name)) {
                            map.putIfAbsent(normalizeKey(name), values[i]);
                        }
                    } else {
                        map.putIfAbsent(normalizeKey(name), values[i]);
                    }
                });

        //make up map id and make sure id is String
        if (id != null) {
            //just override it
            map.put("id", id.toString());
        }

        return map;
    }

    private boolean isInSet(Set<String> filterName, String key) {
        List<String> starSet = filterName.stream()
                .filter(x -> x.endsWith(".*"))
                .map(x -> x.substring(0, x.length() - 2))
                .collect(Collectors.toList());

        if (key.contains(".")) {
            String[] split = key.split("\\.");
            String prefixKey = split[0];
            if (starSet.contains(prefixKey)) {
                return true;
            }
        }

        return filterName.contains(key);
    }

    private String normalizeKey(String key) {
        return key.startsWith(REL_PREFIX) ? key.substring(1) : key;
    }

    @Override
    public Map<String, Object> toNestedMap(Set<String> filterName) {

        Map<String, Object> map = new HashMap<>(values.length);

        IntStream.range(0, values.length)
                .forEach(i -> {
                    String name = normalizeKey(fields[i].name());
                    Boolean insert = false;
                    if (filterName != null && !filterName.isEmpty()) {
                        if (filterName.contains(name)) {
                            insert = true;
                        }
                    } else {
                        insert = true;
                    }

                    if (insert) {
                        //insert
                        if (name.contains(REL_DELIMITER)) {
                            String[] names = name.split("\\" + REL_DELIMITER);

                            Map currentMap = map;

                            for (int index = 0; index < names.length - 1; index++) {
                                String seg = names[index];
                                Object segMap = currentMap.get(seg);
                                if (segMap == null) {
                                    Map<String, Object> innerMap = new HashMap<>();
                                    currentMap.put(seg, innerMap);
                                    currentMap = innerMap;
                                } else if (segMap instanceof Map) {
                                    currentMap = (Map) segMap;
                                }
                            }

                            currentMap.put(names[names.length - 1], values[i]);

                        } else {
                            map.put(name, values[i]);
                        }
                    }
                });

        return map;

    }

    @Override
    public Boolean isEmpty() {
        return isEmpty;
    }

    @Override
    public Boolean nonEmpty() {
        return !isEmpty;
    }

    @Override
    public Integer version() {
        return version;
    }

    /**
     * TODO equals
     *
     * @param record
     * @return
     */
    @Override
    public int compareTo(Record record) {
        return 0;
    }

    @Override
    public String toString() {
        return PrettyPrinter.print(this);
    }
}
