package com.xforceplus.ultraman.oqsengine.pojo.converter;

import com.xforceplus.ultraman.oqsengine.pojo.dto.conditions.Condition;
import com.xforceplus.ultraman.oqsengine.pojo.dto.conditions.ConditionOperator;
import com.xforceplus.ultraman.oqsengine.pojo.dto.conditions.Conditions;
import com.xforceplus.ultraman.oqsengine.pojo.dto.conditions.ValueConditionNode;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.*;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.*;
import com.xforceplus.ultraman.oqsengine.pojo.dto.values.DateTimeValue;
import com.xforceplus.ultraman.oqsengine.pojo.dto.values.EmptyTypedValue;
import com.xforceplus.ultraman.oqsengine.pojo.dto.values.IValue;
import com.xforceplus.ultraman.oqsengine.pojo.dto.values.LongValue;
import com.xforceplus.ultraman.oqsengine.pojo.reader.IEntityClassReader;
import com.xforceplus.ultraman.oqsengine.pojo.reader.record.EmptyValue;
import com.xforceplus.ultraman.oqsengine.pojo.reader.record.Record;
import com.xforceplus.ultraman.oqsengine.pojo.utils.ConvertHelper;
import com.xforceplus.ultraman.oqsengine.sdk.*;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.ultraman.oqsengine.pojo.utils.OptionalHelper.ofEmptyStr;
import static com.xforceplus.ultraman.oqsengine.sdk.FieldConditionUp.Op.*;
import static io.vavr.API.*;
import static io.vavr.Predicates.instanceOf;

/**
 * EntityClass helper to convert things to entityClass
 */
public class IEntityClassHelper {

    private static Logger logger = LoggerFactory.getLogger(IEntityClassHelper.class);

    /**
     * TODO append more logic here
     * fieldUp to ientity field
     *
     * @param fieldUp
     * @return
     */
    public static IEntityField toEntityField(FieldUp fieldUp) {
        return new EntityField(
                fieldUp.getId()
                , fieldUp.getCode()
                , fieldUp.getName()
                , FieldType.valueOf(fieldUp.getFieldType())
                , FieldConfig.build()
                .searchable(ofEmptyStr(fieldUp.getSearchable())
                        .map(Boolean::valueOf).orElse(false))
                .max(ofEmptyStr(fieldUp.getMaxLength())
                        .map(String::valueOf)
                        .map(Long::parseLong).orElse(-1L))
                .min(ofEmptyStr(fieldUp.getMinLength()).map(String::valueOf)
                        .map(Long::parseLong).orElse(-1L))
                .precision(fieldUp.getPrecision())
                .identifie(fieldUp.getIdentifier())
                .isSystem(fieldUp.getIsSystem())
                .required("true".equals(fieldUp.getRequired()))
                , fieldUp.getDictId()
                , fieldUp.hasField(FieldUp.getDescriptor().findFieldByNumber(FieldUp.DEFAULTVALUE_FIELD_NUMBER)) ? fieldUp.getDefaultValue() : null
        );
    }

    public static Relation toEntityRelation(RelationUp relationUp) {
        return new Relation(relationUp.getName()
                , relationUp.getRelatedEntityClassId()
                , relationUp.getRelationType()
                , relationUp.getIdentity()
                , relationUp.hasEntityField() ? toEntityField(relationUp.getEntityField()) : null);
    }

    public static IEntityClass getSubEntityClass(EntityUp entityUp) {
        boolean hasSubClass = entityUp.hasField(EntityUp.getDescriptor().findFieldByNumber(EntityUp.SUBENTITYCLASS_FIELD_NUMBER));

        if (hasSubClass) {
            return toEntityClass(entityUp.getSubEntityClass());
        }

        return null;
    }

    public static IEntityClass toEntityClass(EntityUp entityUp) {

        boolean hasExtendedClass = entityUp.hasField(EntityUp.getDescriptor()
                .findFieldByNumber(EntityUp.EXTENDENTITYCLASS_FIELD_NUMBER));

        //Long id, String code, String relation, List<IEntityClass> entityClasss, IEntityClass extendEntityClass, List<Field> fields
        IEntityClass entityClass = new EntityClass(
                entityUp.getId()
                , Long.valueOf(entityUp.getVersion()).intValue()
                , entityUp.getCode()
                , entityUp.getRelationList().stream()
                .map(IEntityClassHelper::toEntityRelation)
                .collect(Collectors.toList())
                , entityUp.getEntityClassesList().stream()
                .map(relatedEntityUp -> {
                    boolean hasParent = relatedEntityUp.hasField(EntityUp.getDescriptor()
                            .findFieldByNumber(EntityUp.EXTENDENTITYCLASS_FIELD_NUMBER));
                    if (hasParent) {
                        IEntityClass parent = toRawEntityClass(relatedEntityUp.getExtendEntityClass());
                        return toRawEntityClass(relatedEntityUp, parent);
                    } else {
                        return toRawEntityClass(relatedEntityUp);
                    }

                })
                .collect(Collectors.toList())
                , hasExtendedClass ? toRawEntityClass(entityUp.getExtendEntityClass()) : null
                , Collections.emptyList()
                , entityUp.getFieldsList().stream().map(IEntityClassHelper::toEntityField).collect(Collectors.toList())
        );

        List<IEntityClass> children = entityUp.getNarrowEntityClassesList()
                .stream().map(x -> IEntityClassHelper.toRawEntityClass(x, entityClass)).collect(Collectors.toList());

        entityClass.resetChildEntityClass(children);

        return entityClass;
    }


    /**
     * private IEntityClass toEntityClass(EntityUp entityUp) {
     * <p>
     * boolean hasExtendedClass = entityUp.hasField(EntityUp.getDescriptor().findFieldByNumber(EntityUp.EXTENDENTITYCLASS_FIELD_NUMBER));
     * <p>
     * //Long id, String code, String relation, List<IEntityClass> entityClasss, IEntityClass extendEntityClass, List<Field> fields
     * IEntityClass entityClass = new EntityClass(
     * entityUp.getId()
     * , entityUp.getCode()
     * , entityUp.getRelationList().stream()
     * .map(this::toEntityRelation)
     * .collect(Collectors.toList())
     * , entityUp.getEntityClassesList().stream()
     * .map(this::toRawEntityClass)
     * .collect(Collectors.toList())
     * , hasExtendedClass ? toRawEntityClass(entityUp.getExtendEntityClass()) : null
     * , Collections.emptyList()
     * , entityUp.getFieldsList().stream().map(this::toEntityField).collect(Collectors.toList())
     * );
     * <p>
     * <p>
     * List<IEntityClass> children = entityUp.getNarrowEntityClassesList()
     * .stream().map(x -> this.toRawEntityClass(x, entityClass)).collect(Collectors.toList());
     * <p>
     * entityClass.resetChildEntityClass(children);
     * <p>
     * return entityClass;
     * }
     *
     * @param entityField
     * @param value
     * @return
     */
    public static List<IValue> toTypedValue(IEntityField entityField, String value) {
        if (value != null && EmptyValue.isEmpty(value)) {
            return Collections.singletonList(new EmptyTypedValue(entityField));
        } else {
            return entityField.type().toTypedValue(entityField, value)
                    .map(Collections::singletonList).orElseGet(Collections::emptyList);
        }
    }


    public static IEntityClass toRawEntityClass(EntityUp entityUp) {
        return toRawEntityClass(entityUp, null);
    }

    public static IEntityClass toRawEntityClass(EntityUp entityUp, IEntityClass parent) {
        return new EntityClass(
                entityUp.getId()
                , Optional.ofNullable(entityUp.getVersion()).map(x -> Long.valueOf(x).intValue()).orElse(-1)
                , entityUp.getCode()
                , null
                , Collections.emptyList()
                , parent
                , Collections.emptyList()
                , entityUp.getFieldsList().stream().map(IEntityClassHelper::toEntityField).collect(Collectors.toList())
        );
    }

    /**
     * turn conditionUp to conditions
     *
     * @param mainClass
     * @param reader
     * @param conditionsUp
     * @param ids
     * @return
     */
    public static Optional<Conditions> toConditions(IEntityClass mainClass, IEntityClassReader reader
            , ConditionsUp conditionsUp, List<Long> ids) {
        Optional<Conditions> conditions = conditionsUp.getFieldsList().stream().map(x -> {
            /**
             * turn alias field to columnfield
             */
            Optional<AliasField> field = reader.field(x.getField().getId());
            return toOneConditions(field.map(AliasField::getOriginObject).map(f -> (ColumnField) f), x, mainClass);
        }).filter(Objects::nonNull).reduce((a, b) -> a.addAnd(b, true));

        //remove special behavior for ids
//        //Remove Empty ids judgment
        if (ids != null && !ids.isEmpty()) {
//            Optional<IEntityField> idField = IEntityClassHelper.findFieldByCode(entityClass, "id");
            Optional<IEntityField> idField = reader.column("id").map(ColumnField::getOriginObject);
            Optional<Conditions> conditionsIds = idField.map(field -> {
                return new Conditions(new Condition(field
                        , ConditionOperator.MULTIPLE_EQUALS
                        , ids.stream().map(x -> new LongValue(field, x)).toArray(IValue[]::new)));
            });

            if (conditions.isPresent()) {
                if (conditionsIds.isPresent()) {
                    return conditions.map(x -> x.addAnd(conditionsIds.get(), true));
                }
            } else {
                return conditionsIds;
            }
        }
        return conditions;
    }

    /**
     * turn entityUp to IEntityValue
     *
     * @param reader
     * @param entityUp
     * @return
     */
    public static IEntityValue toEntityValue(IEntityClassReader reader, EntityUp entityUp) {

        List<IValue> valueList = entityUp.getValuesList().stream()
                .flatMap(y -> {
                    Optional<? extends IEntityField> entityFieldOp = reader.field(y.getFieldId()).map(AliasField::getOrigin);
                    return entityFieldOp
                            .map(x -> toTypedValue(x, y.getValue()))
                            .orElseGet(Collections::emptyList)
                            .stream();
                }).filter(Objects::nonNull).collect(Collectors.toList());
        EntityValue entityValue = new EntityValue(entityUp.getId());
        entityValue.addValues(valueList);
        return entityValue;
    }

    /**
     * turn a fieldCondition to conditions
     *
     * @param fieldOp
     * @param fieldCondition
     * @param mainClass
     * @return
     */
    public static Conditions toOneConditions(Optional<ColumnField> fieldOp, FieldConditionUp fieldCondition, IEntityClass mainClass) {

        Conditions conditions = null;

        if (fieldOp.isPresent()) {
            FieldConditionUp.Op op = fieldCondition.getOperation();

            ColumnField columnField = fieldOp.get();
            IEntityField originField = columnField;

            //in order
            List<String> nonNullValueList = fieldCondition
                    .getValuesList()
                    .stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());

            //return if field with invalid
            if (nonNullValueList.isEmpty()) {
//                conditions = Conditions.buildEmptyConditions();
//                return conditions;
                throw new RuntimeException("Field " + columnField + " Value is Missing");
            }

            switch (op) {
                case eq:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.EQUALS
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case ne:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.NOT_EQUALS
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case ge:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.GREATER_THAN_EQUALS
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case gt:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.GREATER_THAN
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case ge_le:
                    if (nonNullValueList.size() > 1) {

                        //will get empty
                        String leftValue = nonNullValueList.get(0);
                        String rightValue = nonNullValueList.get(1);

                        Condition left = null;
                        Condition right = null;

                        if (!StringUtils.isEmpty(leftValue)) {
                            left = new Condition(
                                    isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                    , originField
                                    , ConditionOperator.GREATER_THAN_EQUALS
                                    , toTypedValue(fieldOp.get()
                                    , nonNullValueList.get(0)).toArray(new IValue[]{}));
                        }

                        if (!StringUtils.isEmpty(rightValue)) {
                            right = new Condition(
                                    isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                    , originField
                                    , ConditionOperator.LESS_THAN_EQUALS
                                    , toTypedValue(fieldOp.get()
                                    , nonNullValueList.get(1)).toArray(new IValue[]{}));
                        }

                        if (left == null && right == null) {
                            conditions = null;
                        } else if (left == null) {
                            conditions = new Conditions(right);
                        } else if (right == null) {
                            conditions = new Conditions(left);
                        } else {
                            conditions = new Conditions(left).addAnd(right);
                        }
                    } else {
                        logger.warn("required value more then 1, fallback to ge");
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    }
                    break;
                case gt_le:
                    if (nonNullValueList.size() > 1) {

                        //will get empty
                        String leftValue = nonNullValueList.get(0);
                        String rightValue = nonNullValueList.get(1);

                        Condition left = null;
                        Condition right = null;

                        if (!StringUtils.isEmpty(leftValue)) {
                            left = new Condition(
                                    isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                    , originField
                                    , ConditionOperator.GREATER_THAN
                                    , toTypedValue(fieldOp.get()
                                    , nonNullValueList.get(0)).toArray(new IValue[]{}));
                        }

                        if (!StringUtils.isEmpty(rightValue)) {
                            right = new Condition(
                                    isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                    , originField
                                    , ConditionOperator.LESS_THAN_EQUALS
                                    , toTypedValue(fieldOp.get()
                                    , nonNullValueList.get(1)).toArray(new IValue[]{}));

                            conditions = new Conditions(left).addAnd(right);
                        }

                        if (left == null && right == null) {
                            conditions = null;
                        } else if (left == null) {
                            conditions = new Conditions(right);
                        } else if (right == null) {
                            conditions = new Conditions(left);
                        } else {
                            conditions = new Conditions(left).addAnd(right);
                        }
                    } else {
                        logger.warn("required value more then 1, fallback to gt");
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    }
                    break;
                case ge_lt:
                    if (nonNullValueList.size() > 1) {
                        Condition left = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{}));

                        Condition right = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.LESS_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(1)).toArray(new IValue[]{}));


                        conditions = new Conditions(left).addAnd(right);

                    } else {
                        logger.warn("required value more then 2, fallback to ge");
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    }
                    break;
                case gt_lt:
                    if (nonNullValueList.size() > 1) {
                        Condition left = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{}));

                        Condition right = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.LESS_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(1)).toArray(new IValue[]{}));


                        conditions = new Conditions(left).addAnd(right);

                    } else {
                        logger.warn("required value more then 2, fallback to ge");
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    }
                    break;
                case le:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.LESS_THAN_EQUALS
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case lt:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.LESS_THAN
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case in:
                    conditions = new Conditions(
                            new Condition(
                                    isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                    , originField
                                    , ConditionOperator.MULTIPLE_EQUALS
                                    , nonNullValueList.stream().flatMap(x -> toTypedValue(fieldOp.get(), x).stream())
                                    .toArray(IValue[]::new)
                            )
                    );
                    break;
                case ni:
                    if (nonNullValueList.size() == 1) {
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.NOT_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    } else {
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.NOT_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));

                        Conditions finalConditions = conditions;
                        nonNullValueList.stream().skip(1).forEach(x -> {
                            finalConditions.addAnd(new Conditions(new Condition(
                                    isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                    , originField
                                    , ConditionOperator.NOT_EQUALS
                                    , toTypedValue(fieldOp.get()
                                    , x).toArray(new IValue[]{}))), false);
                        });

                        conditions = finalConditions;
                    }
                    break;
                case like:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.LIKE
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                default:
            }
        }

        if (conditions == null) {
            throw new RuntimeException("Condition is invalid " + fieldCondition);
        }

        return conditions;
    }

    /**
     * TODO
     * is the column is not belong to current entityClass
     *
     * @param columnField
     * @param mainClass
     * @return
     */
    public static boolean isRelatedField(ColumnField columnField, IEntityClass mainClass) {
        IEntityClass entityClass = columnField.originEntityClass();

        if (mainClass.extendEntityClass() != null) {
            return mainClass.id() != entityClass.id() && mainClass.extendEntityClass().id() != entityClass.id();
        } else {
            return entityClass.id() != mainClass.id();
        }
    }

    /**
     * entityUp in to IEntity
     *
     * @param entityClass
     * @param in
     * @return
     */
    public static IEntity toEntity(IEntityClass entityClass, IEntityClassReader reader, EntityUp in) {
        return new Entity(in.getObjId(), entityClass, toEntityValue(reader, in));
    }

    /**
     * entityUp in to IEntity
     *
     * @param entityClass
     * @param in
     * @return
     */
    public static IEntity toEntity(IEntityClass entityClass, IEntityClassReader reader, EntityUp in, int version) {
        return new Entity(in.getObjId(), entityClass, toEntityValue(reader, in), version);
    }

    /**
     * to entityUp
     *
     * @param entity
     * @return
     */
    public static EntityUp toEntityUp(IEntity entity) {
        EntityUp.Builder builder = EntityUp.newBuilder();
        if (entity.entityClass() != null) {
            builder.setId(entity.entityClass().id());
        }
        builder.setObjId(entity.id());
        builder.addAllValues(entity.entityValue().values().stream()
                .map(IEntityClassHelper::toValueUp)
                .collect(Collectors.toList()));
        return builder.build();
    }

    private static ValueUp toValueUp(IValue value) {
        //TODO format?
        IEntityField field = value.getField();
        return ValueUp.newBuilder()
                .setValue(toValueStr(value))
                .setName(field.name())
                .setFieldId(field.id())
                .setFieldType(field.type().name())
                .build();
    }

    private static String toValueStr(IValue value) {
        String retVal
                = Match(value)
                .of(Case($(instanceOf(DateTimeValue.class)), x -> String.valueOf(x.valueToLong())),
                        Case($(), IValue::valueToString));
        return retVal;
    }

    public static EntityUp toEntityUp(IEntityClass entityClass, IEntityClass... subEntityClass) {
        return toEntityUpBuilder(entityClass, null, subEntityClass).build();
    }

    public static EntityUp toEntityUp(IEntityClass entityClass) {
        return toEntityUpBuilder(entityClass, null).build();
    }

    public static EntityUp.Builder toEntityUpBuilder(IEntityClass entityClass, Long id) {
        return toEntityUpBuilder(entityClass, id, null);
    }

    public static EntityUp.Builder toEntityUpBuilder(IEntityClass entityClass, Long id, IEntityClass... subEntityClass) {

        EntityUp.Builder builder = EntityUp.newBuilder();

        //add parent
        if (entityClass.extendEntityClass() != null) {
            IEntityClass parent = entityClass.extendEntityClass();
            EntityUp parentUp = toRawEntityUp(parent);
            builder.setExtendEntityClass(parentUp);
        }

        if (entityClass.ver() > 0) {
            builder.setVersion(entityClass.ver());
        }

        //add obj id
        if (id != null) {
            builder.setObjId(id);
        }

        //add relation
        //relation may has no field
        builder.addAllRelation(entityClass.relations().stream().map(rel -> {
            RelationUp.Builder relation = RelationUp.newBuilder();

            if (rel.getEntityField() != null) {
                relation.setEntityField(toFieldUp(rel.getEntityField()));
            }

            relation.setName(Optional.ofNullable(rel.getName()).orElse(""));
            relation.setRelationType(rel.getRelationType());

            if (rel.isIdentity()) {
                relation.setIdentity(rel.isIdentity());
            }

            relation.setRelatedEntityClassId(rel.getEntityClassId());
            return relation.build();
        }).collect(Collectors.toList()));

        builder.setId(entityClass.id())
                .setCode(entityClass.code())
                .addAllFields(entityClass.fields().stream().map(IEntityClassHelper::toFieldUp).collect(Collectors.toList()));

        if (entityClass.entityClasss() != null && !entityClass.entityClasss().isEmpty()) {
            builder.addAllEntityClasses(entityClass.entityClasss().stream()
                    .map(IEntityClassHelper::toEntityUp).collect(Collectors.toList()));
        }

        if (subEntityClass == null) {
            if (entityClass.childEntityClasses() != null && !entityClass.childEntityClasses().isEmpty()) {
                builder.addAllNarrowEntityClasses(entityClass.childEntityClasses().stream().map(IEntityClassHelper::toRawEntityUp).collect(Collectors.toList()));
            }
        } else {
            builder.addAllNarrowEntityClasses(Stream.of(subEntityClass).map(IEntityClassHelper::toRawEntityUp).collect(Collectors.toList()));
        }

        return builder;
    }


    public static FieldUp toFieldUp(IEntityField field) {
        FieldUp.Builder builder =
                FieldUp.newBuilder()
                        .setCode(field.name())
                        .setFieldType(field.type().name())
                        .setId(field.id())
                        .setName(Optional.ofNullable(field.cnName()).orElse(""))
                        .setDictId(Optional.ofNullable(field.dictId()).orElse(""));


        if (field.config() != null) {
            builder.setSearchable(String.valueOf(field.config().isSearchable()));
            builder.setMaxLength(String.valueOf(field.config().getMax()));
            builder.setMinLength(String.valueOf(field.config().getMin()));
            builder.setPrecision(field.config().getPrecision());
            builder.setIdentifier(field.config().isIdentifie());
            builder.setUniqueName(field.config().uniqueName());
            builder.setIsSystem(field.config().isSystem());
            builder.setRequired(Boolean.valueOf(field.config().isRequired()).toString());
        }

        if (field.defaultValue() != null) {
            builder.setDefaultValue(field.defaultValue());
        }


        return builder.build();
    }

    public static EntityUp toRawEntityUp(IEntityClass entity) {
        return EntityUp.newBuilder()
                .setId(entity.id())
                .setVersion(entity.ver())
                .setCode(entity.code())
                .addAllFields(entity.fields().stream().map(IEntityClassHelper::toFieldUp).collect(Collectors.toList()))
                .build();
    }

    public static Optional<FieldConditionUp> toFieldCondition(Condition condition) {

        IEntityField field = condition.getField();
        IValue value = condition.getFirstValue();
        ConditionOperator operator = condition.getOperator();
        //check
        if (value == null || field == null || operator == null) {
            return Optional.empty();
        }

        FieldConditionUp fieldCondition = FieldConditionUp.newBuilder()
                .setCode(field.name())
                .setOperation(toConditionOp(operator))
                .addValues(value.valueToString())
                .setField(toFieldUp(field))
                .build();

        return Optional.of(fieldCondition);
    }

    private static FieldConditionUp.Op toConditionOp(ConditionOperator conditionOperator) {
        FieldConditionUp.Op op;
        switch (conditionOperator) {
            case GREATER_THAN:
                op = gt;
                break;
            case GREATER_THAN_EQUALS:
                op = ge;
                break;
            case LIKE:
                op = like;
                break;
            case NOT_EQUALS:
                op = ne;
                break;
            case LESS_THAN_EQUALS:
                op = le;
                break;
            case LESS_THAN:
                op = lt;
                break;
            case EQUALS:
            default:
                op = eq;
        }
        return op;
    }

    public static FieldSortUp toSortUp(IEntityField field, boolean isAsc) {
        return FieldSortUp.newBuilder()
                .setCode(field.name())
                .setOrder(isAsc ? FieldSortUp.Order.asc : FieldSortUp.Order.desc)
                .build();
    }

    /**
     * TODO
     * turn entityClass to EntityValueUp
     *
     * @param entityClass
     * @param id
     * @param valueTuple
     * @return
     */
    public static EntityUp toEntityUp(IEntityClass entityClass, Long id, List<Tuple2<IEntityField, Object>> valueTuple) {
        //build entityUp
        EntityUp.Builder builder = toEntityUpBuilder(entityClass, id);
        List<ValueUp> valueList = valueTuple.stream()
                .filter(x -> x._2() != null)
                .map(x -> {
                    return ValueUp.newBuilder()
                            .setFieldId(x._1.id())
                            .setFieldType(x._1.type().getType())
                            .setValue(ConvertHelper.convert(x._2()))
                            .build();
                }).collect(Collectors.toList());
        builder = builder.addAllValues(valueList);
        return builder.build();
    }

    public static EntityUp toEntityUp(IEntityClass entityClass, long id) {
        return toEntityUpBuilder(entityClass, id).build();
    }

    public static ConditionsUp toConditionsUp(Conditions conditions) {

        ConditionsUp.Builder conditionsUpBuilder = ConditionsUp.newBuilder();
        conditionsUpBuilder.addAllFields(conditions.collect().stream().filter(Conditions::isValueNode)
                .map(x -> ((ValueConditionNode) x).getCondition())
                .map(IEntityClassHelper::toFieldCondition)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList()));
        return conditionsUpBuilder.build();
    }

    public static Record toRecord(IEntityClassReader reader, EntityUp up, List<Long> ids) {
        Map<Tuple2<String, Long>, Object> retValue = up.getValuesList().stream()
                .collect(Collectors.toMap(x -> {
                            return Tuple.of(x.getName(), x.getFieldId());
                        }
                        , ValueUp::getValue));

        long id = up.getId();


        Record record = reader.toRecordNew(retValue, id);

        if (id > 0) {
            record.setTypeId(id);
        }

        long objId = up.getObjId();
        record.setId(objId);

        if (ids.size() > 1) {
            record.setParentId(ids.get(1));
        }

        return record;
    }

    public static Collection<IEntityClass> flattenEntityClass(IEntityClass entityClass) {
        List<IEntityClass> flats = new ArrayList<>();
        if (null == entityClass) {
            return flats;
        }

        //  add self
        flats.add(entityClass);
        //  add child
        if (null != entityClass.childEntityClasses()) {
            flats.addAll(entityClass.childEntityClasses());
        }
        return flats;
    }

    /**
     * 取compareEntityClassId所对应的IEntityClass
     *
     * @param flatEntityClasses
     * @param compareEntityClassId
     * @return
     */
    public static Optional<IEntityClass> entityMember(Collection<IEntityClass> flatEntityClasses, long compareEntityClassId) {
        return flatEntityClasses.stream().filter(e -> e.id() == compareEntityClassId).findFirst();
    }

    /**
     * 判断是否存在compareEntityClassId所对应的IEntityClass
     *
     * @param entityClass
     * @param compareEntityClassId
     * @return
     */
    public static boolean isEntityMember(IEntityClass entityClass, long compareEntityClassId) {
        List<IEntityClass> flattenEntityClasses = (List<IEntityClass>) flattenEntityClass(entityClass);
        return flattenEntityClasses.stream().anyMatch(e -> e.id() == compareEntityClassId);
    }
}
