package com.xforceplus.ultraman.oqsengine.pojo.converter;

import com.xforceplus.ultraman.oqsengine.pojo.dto.conditions.Condition;
import com.xforceplus.ultraman.oqsengine.pojo.dto.conditions.ConditionOperator;
import com.xforceplus.ultraman.oqsengine.pojo.dto.conditions.Conditions;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.ColumnField;
import com.xforceplus.ultraman.oqsengine.pojo.dto.values.IValue;
import com.xforceplus.ultraman.oqsengine.pojo.reader.IEntityClassReader;
import com.xforceplus.ultraman.oqsengine.sdk.FilterNode;
import com.xforceplus.ultraman.oqsengine.sdk.Filters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * a helper to convert tree node to conditions
 */
public class TreeExpHelper {

    private static Logger logger = LoggerFactory.getLogger(TreeExpHelper.class);

    public static Optional<Conditions> toConditions(IEntityClassReader reader, Filters filters) {
        if (!filters.isInitialized()) {
            return Optional.empty();
        } else {
            //and clause
            List<FilterNode> filterList = filters.getNodesList();
            return filterList.stream()
                    .map(x -> {
                        return toConditions(x, reader);
                    }).filter(Objects::nonNull)
                    .reduce((a, b) -> a.addAnd(b, true));
        }
    }

    private static boolean isRelatedField(ColumnField columnField, IEntityClass mainClass) {
        IEntityClass entityClass = columnField.originEntityClass();

        if (mainClass.extendEntityClass() != null) {
            return mainClass.id() != entityClass.id() && mainClass.extendEntityClass().id() != entityClass.id();
        } else {
            return entityClass.id() != mainClass.id();
        }
    }

    /**
     * 0 is clause
     * 1 is Field or Value
     *
     * @param node
     * @return
     */
    private static Conditions toConditions(FilterNode node, IEntityClassReader reader) {
        if (node.getNodeType() == 0) {
            //condition
            List<FilterNode> nodesList = node.getNodesList();

            if (FilterNode.Operator.or == node.getOperator()) {
                //or
                return nodesList.stream()
                        .filter(x -> x.getNodeType() == 0)
                        .map(x -> toConditions(x, reader))
                        .reduce((a, b) -> a.addOr(b, true)).orElseThrow(() -> new RuntimeException("no conditions in OR"));
            } else if (FilterNode.Operator.and == node.getOperator()) {
                //and
                return nodesList.stream()
                        .filter(x -> x.getNodeType() == 0)
                        .map(x -> toConditions(x, reader))
                        .reduce((a, b) -> a.addAnd(b, true)).orElseThrow(() -> new RuntimeException("no conditions in AND"));
            } else {

                Optional<FilterNode> filterNode = extractFieldNode(node);

                List<FilterNode> values = extractValueNode(node);
                Conditions conditions = toOneConditions(filterNode.flatMap(x -> reader.column(x.getPayload()))
                        , values.stream().map(FilterNode::getPayload).collect(Collectors.toList())
                        , node.getOperator(), reader.getEntityClass());

                if (conditions == null) {
                    throw new RuntimeException("Condition is invalid with field " + filterNode + " " + node.getOperator() + " values:" + values);
                }

                return conditions;
            }
        }

        return null;
    }

    private static Optional<FilterNode> extractFieldNode(FilterNode node) {
        return node.getNodesList()
                .stream()
                .filter(x -> x.getNodeType() == 1)
                .findFirst();
    }

    private static List<FilterNode> extractValueNode(FilterNode node) {
        return node.getNodesList()
                .stream()
                .filter(x -> x.getNodeType() == 2)
                .collect(Collectors.toList());
    }

    //TODO error handler
    private static Conditions toOneConditions(
            Optional<ColumnField> fieldOp
            , List<String> values
            , FilterNode.Operator op
            , IEntityClass mainClass) {

        Conditions conditions = null;

        if (fieldOp.isPresent()) {
            ColumnField columnField = fieldOp.get();
            IEntityField originField = columnField;

            //in order
            List<String> nonNullValueList =
                    values
                            .stream()
                            .filter(Objects::nonNull)
                            .collect(Collectors.toList());

            //return if field with invalid
            if (nonNullValueList.isEmpty()) {
                throw new RuntimeException("Field " + columnField + " Value is Missing");
            }

            switch (op) {
                case eq:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.EQUALS
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case ne:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.NOT_EQUALS
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case ge:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.GREATER_THAN_EQUALS
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case gt:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.GREATER_THAN
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case ge_le:
                    if (nonNullValueList.size() > 1) {
                        Condition left = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{}));

                        Condition right = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.LESS_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(1)).toArray(new IValue[]{}));

                        conditions = new Conditions(left).addAnd(right);

                    } else {
                        logger.warn("required value more then 2, fallback to ge");
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    }
                    break;
                case gt_le:
                    if (nonNullValueList.size() > 1) {
                        Condition left = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{}));

                        Condition right = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.LESS_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(1)).toArray(new IValue[]{}));


                        conditions = new Conditions(left).addAnd(right);

                    } else {
                        logger.warn("required value more then 2, fallback to gt");
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    }
                    break;
                case ge_lt:
                    if (nonNullValueList.size() > 1) {
                        Condition left = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{}));

                        Condition right = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.LESS_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(1)).toArray(new IValue[]{}));


                        conditions = new Conditions(left).addAnd(right);

                    } else {
                        logger.warn("required value more then 2, fallback to ge");
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    }
                    break;
                case gt_lt:
                    if (nonNullValueList.size() > 1) {
                        Condition left = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{}));

                        Condition right = new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.LESS_THAN
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(1)).toArray(new IValue[]{}));


                        conditions = new Conditions(left).addAnd(right);

                    } else {
                        logger.warn("required value more then 2, fallback to ge");
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.GREATER_THAN_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    }
                    break;
                case le:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.LESS_THAN_EQUALS
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case lt:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.LESS_THAN
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                case in:
                    conditions = new Conditions(
                            new Condition(
                                    isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                    , originField
                                    , ConditionOperator.MULTIPLE_EQUALS
                                    , nonNullValueList.stream().flatMap(x -> toTypedValue(fieldOp.get(), x).stream())
                                    .toArray(IValue[]::new)
                            )
                    );
                    break;
                case ni:
                    if (nonNullValueList.size() == 1) {
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.NOT_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    } else {
                        conditions = new Conditions(new Condition(
                                isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                , originField
                                , ConditionOperator.NOT_EQUALS
                                , toTypedValue(fieldOp.get()
                                , nonNullValueList.get(0)).toArray(new IValue[]{})));

                        Conditions finalConditions = conditions;
                        nonNullValueList.stream().skip(1).forEach(x -> {
                            finalConditions.addAnd(new Conditions(new Condition(
                                    isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                                    , originField
                                    , ConditionOperator.NOT_EQUALS
                                    , toTypedValue(fieldOp.get()
                                    , x).toArray(new IValue[]{}))), false);
                        });

                        conditions = finalConditions;
                    }
                    break;
                case like:
                    conditions = new Conditions(new Condition(
                            isRelatedField(columnField, mainClass) ? columnField.originEntityClass() : null
                            , originField
                            , ConditionOperator.LIKE
                            , toTypedValue(fieldOp.get()
                            , nonNullValueList.get(0)).toArray(new IValue[]{})));
                    break;
                default:

            }
        }

        return conditions;
    }


    //TODO
    private static List<IValue> toTypedValue(IEntityField entityField, String value) {
        return entityField.type().toTypedValue(entityField, value).map(Collections::singletonList).orElseGet(Collections::emptyList);
    }
}
