package com.xforceplus.ultraman.sdk.bulk.controller;

import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.tech.base.core.dispatcher.ServiceDispatcher;
import com.xforceplus.tech.common.utils.JsonHelper;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionQueryRequest;
import com.xforceplus.ultraman.metadata.domain.vo.dto.NameMapping;
import com.xforceplus.ultraman.metadata.domain.vo.dto.RelationQuery;
import com.xforceplus.ultraman.metadata.domain.vo.dto.Response;
import com.xforceplus.ultraman.metadata.helper.ConditionQueryRequestHelper;
import com.xforceplus.ultraman.sdk.bulk.controller.dto.ImportRequest;
import com.xforceplus.ultraman.sdk.bulk.controller.dto.ImportTemplateRequest;
import com.xforceplus.ultraman.sdk.bulk.controller.dto.RelatedMapping;
import com.xforceplus.ultraman.sdk.controller.constants.SDKContextKey;
import com.xforceplus.ultraman.sdk.core.bulk.BulkService;
import com.xforceplus.ultraman.sdk.core.cmd.ConditionExportCmd;
import com.xforceplus.ultraman.sdk.core.cmd.ExportCmdQuery;
import com.xforceplus.ultraman.sdk.core.cmd.GetImportTemplateCmd;
import com.xforceplus.ultraman.sdk.core.cmd.ImportCmd;
import com.xforceplus.ultraman.sdk.core.rel.legacy.ExpFactory;
import com.xforceplus.ultraman.sdk.core.rel.legacy.ExpRel;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.Tuple3;
import io.vavr.control.Either;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StreamUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.sdk.controller.constants.Constants.FAILED;

@RequestMapping
public class EntityBulkController {

    @Autowired
    private ContextService contextService;

    @Autowired
    private ServiceDispatcher dispatcher;

    private int exportMaxSize;

    private Logger logger = LoggerFactory.getLogger(EntityBulkController.class);

    public EntityBulkController(int exportMaxSize) {
        this.exportMaxSize = exportMaxSize;
    }

    /**
     * TODO
     *
     * @param boId
     * @param version
     * @param condition
     * @return
     */
    @PostMapping("/bos/{boId}/entities/export")
    @ResponseBody
    public CompletableFuture<Response<String>> conditionExport(
            @PathVariable("boId") String boId,
            @RequestParam(required = true, defaultValue = "sync", value = "exportType") String exportType,
            @RequestParam(required = false, value = "v") String version,
            @RequestParam(required = false, value = "appId") String appId,
            @RequestParam(required = false, value = "skip", defaultValue = "false") boolean skip,
            @RequestParam(required = false, value = "pageCode") String pageCode,
            @RequestParam(required = false, value = "type", defaultValue = "xls") String exportFileType,
            @RequestParam(required = false, value = "filename") String specifiedFileName,
            @RequestParam(required = false, value = "ignoreMax", defaultValue = "false") Boolean ignoreMax,
            @RequestBody ConditionQueryRequest condition) {

        String page = Optional.ofNullable(pageCode).orElse("PAGE");
        contextService.set(SDKContextKey.PAGE, page);

        //default
        if (condition != null) {
            if (condition.getPageNo() == null) {
                condition.setPageNo(1);
            }

            if (condition.getPageSize() == null) {
                if (!ignoreMax) {
                    condition.setPageSize(exportMaxSize);
                } else {
                    condition.setPageSize(-1);
                }
            }

            if (condition.getPageSize() > exportMaxSize) {
                if (!ignoreMax) {
                    condition.setPageSize(exportMaxSize);
                }
            }

            /**
             * make this valid
             */
            ConditionQueryRequestHelper.transformer(condition);
        }

        Map<String, ExportCmdQuery> map = new HashMap<>();
        map.put(boId, toExportCmdQuery(condition));

        CompletableFuture<Either<String, String>> exportResult = dispatcher
                .querySync(new ConditionExportCmd(
                                boId
                                , version
                                , exportType
                                , exportFileType
                                , appId
                                , map
                                , specifiedFileName
                                , Optional.ofNullable(condition).map(ConditionQueryRequest::getAttachment).orElse(null)
                                , skip)
                        , BulkService.class, "conditionExport");

        return exportResult.thenApply(x -> {
            if (x.isRight()) {
                Response<String> response = new Response<>();
                response.setResult(x.get());
                response.setMessage("OK");
                response.setCode("1");
                return response;
            } else {
                return Response.Error(x.getLeft());
            }
        });
    }

    /**
     * condition query => Export Query
     *
     * @param condition
     * @return
     */
    private ExportCmdQuery toExportCmdQuery(ConditionQueryRequest condition) {
        ExportCmdQuery exportQuery = new ExportCmdQuery();
        //setup main
        exportQuery.setMainQuery(ExpFactory.createFrom(condition));
        exportQuery.setMainMapping(condition.getMapping());
        exportQuery.setMaxRecord(condition.getPageSize());


        List<RelationQuery> toManyRelations = condition.getToManyRelations();

        Map<String, ExpRel> subQuery = new HashMap<>();
        Map<String, List<NameMapping>> subNameMapping = new HashMap<>();

        Optional.ofNullable(toManyRelations).orElseGet(Collections::emptyList).stream().map(x -> {
            String code = x.getCode();
            String name = x.getName();

            //new nameMapping
            NameMapping nm = new NameMapping();
            nm.setCode(code);
            nm.setText(name);

            ExpRel expQuery = ExpFactory.createFrom(x.getConditions(), x.getEntity(), x.getMapping(), x.getSort(), null, null);
            List<NameMapping> nameMapping = x.getMapping();

            List<NameMapping> newNameMapping = new ArrayList<>(nameMapping);
            newNameMapping.add(nm);
            return Tuple.of(code, expQuery, newNameMapping);
        }).forEach(x -> {
            subQuery.put(x._1, x._2);
            subNameMapping.put(x._1, x._3);
        });

        exportQuery.setSubMapping(subNameMapping);
        exportQuery.setSubQuery(subQuery);

        return exportQuery;
    }

    /**
     * TODO add template
     *
     * @param boId
     * @param version
     * @param pageCode
     * @param request
     * @return
     */
    @PostMapping("/bos/{boId}/entities/import/template")
    @ResponseBody
    public CompletableFuture<Response<String>> importTemplateCustom(
            @PathVariable String boId
            , @RequestParam(required = false, value = "v") String version
            , @RequestParam(required = false, value = "pageCode") String pageCode
            , @RequestBody ImportTemplateRequest request) {

        List<com.xforceplus.ultraman.sdk.bulk.controller.dto.Tuple2> inputNameMapping =
                Optional.ofNullable(request.getFields()).orElseGet(Collections::emptyList);

        List<RelatedMapping> relatedMappings = Optional
                .ofNullable(request.getToManyRelations())
                .orElseGet(Collections::emptyList);

        List<NameMapping> nameMappings = inputNameMapping.stream().map(x -> {
            NameMapping mapping = new NameMapping();
            mapping.setCode(x.getCode());
            mapping.setText(x.getName());
            mapping.setFormat(x.getFormat());
            return mapping;
        }).collect(Collectors.toList());

        /**
         * related name mapping
         */
        Map<String, Tuple3<List<NameMapping>, List<NameMapping>, String>> subMapping = new HashMap<>();

        relatedMappings.stream().forEach(x -> {
            List<NameMapping> nameMapping = x.getFields().stream().map(y -> {
                NameMapping mapping = new NameMapping();
                mapping.setCode(y.getCode());
                mapping.setText(y.getName());
                mapping.setFormat(x.getFormat());
                return mapping;
            }).collect(Collectors.toList());

            List<NameMapping> addOn = x.getRelationFields().stream().map(y -> {
                NameMapping mapping = new NameMapping();
                mapping.setCode(y.getCode());
                mapping.setText(y.getName());
                mapping.setFormat(x.getFormat());
                return mapping;
            }).collect(Collectors.toList());

            subMapping.put(x.getCode(), Tuple.of(nameMapping, addOn, Optional.ofNullable(x.getName()).orElse(x.getCode())));
        });


        CompletableFuture<Either<String, String>> importResult = dispatcher.querySync(
                new GetImportTemplateCmd(boId, version, nameMappings, subMapping)
                , BulkService.class, "importTemplateCustom");

        return importResult.thenApply(x -> {
            if (x.isRight()) {
                Response<String> response = new Response<>();
                response.setResult(x.get());
                response.setCode("1");
                return response;
            } else {
                return Response.Error(x.getLeft());
            }
        });
    }

    /**
     * download template
     *
     * @param boId
     * @return
     */
    @GetMapping("/bos/{boId}/entities/import/template")
    @ResponseBody
    public ResponseEntity<StreamingResponseBody> importTemplate(@PathVariable String boId
            , @RequestParam(required = false, value = "v") String version
            , @RequestParam(required = false, value = "pageCode") String pageCode
    ) {

        //TODO general PAGE context?
        String page = Optional.ofNullable(pageCode).orElse("PAGE");
        contextService.set(SDKContextKey.PAGE, page);

        Either<String, Tuple2<String, InputStream>> importTemplate = dispatcher
                .querySync(new GetImportTemplateCmd(boId, version)
                        , BulkService.class, "importTemplate");

        if (importTemplate.isRight()) {
            InputStream finalInput = importTemplate.get()._2();
            StreamingResponseBody responseBody = outputStream -> {
                StreamUtils.copy(finalInput, outputStream);
                outputStream.close();
            };

            String encodedName = importTemplate.get()._1();
            try {
                encodedName = URLEncoder.encode(importTemplate.get()._1(), "UTF-8");
            } catch (UnsupportedEncodingException e) {
                logger.error("{}", e);
            }

            return ResponseEntity.ok()
                    .header(HttpHeaders.CONTENT_DISPOSITION,
                            "attachment; filename=" + encodedName)
                    .body(responseBody);
        } else {
            logger.error("Download template failed {}", importTemplate.getLeft());
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
        }
    }


    @PostMapping("/bos/{boId}/entities/import")
    @ResponseBody
    public ResponseEntity<Response<Object>> importEntities(
            @PathVariable String boId
//            , @RequestParam(required = false, value = "v") String version
//            , @RequestParam(required = false, value = "appId") String appId
//            , @RequestParam(required = false, value = "pageCode") String pageCode
//            , @RequestParam(required = false, value = "importMode", defaultValue = "base") String importMode
//            , @RequestParam(required = false, value = "async", defaultValue = "false") boolean isAsync
//            , @RequestParam(required = false, value = "step", defaultValue = "1000") int step
//            , @RequestParam(required = false, value = "timeout", defaultValue = "300000") int timeout
//            , @RequestParam(required = false, value = "sheets") String sheets
//            , @RequestParam(required = false, value = "toManyRelations") String toManyRelations
//            , @RequestParam(required = false, value = "useBatch",defaultValue = "true") boolean useBatch
            , @ModelAttribute ImportRequest importRequest
    ) throws IOException {

        String page = Optional.ofNullable(importRequest.getPageCode()).orElse("PAGE");
        contextService.set(SDKContextKey.PAGE, page);

        List<ImportCmd.Sheet> sheets1 = null;
        List<ImportCmd.ToManyRelation> toManyRelations1 = null;

        if (!StringUtils.isEmpty(importRequest.getSheets())) {
            Optional<List<ImportCmd.Sheet>> sheets2 = JsonHelper.arrFromJsonStr(importRequest.getSheets(), ImportCmd.Sheet.class);
            sheets1 = sheets2.orElseThrow(() -> new RuntimeException("sheet param is not valid"));
            Optional<List<ImportCmd.ToManyRelation>> toManyRelations2 = JsonHelper.arrFromJsonStr(importRequest.getToManyRelations(), ImportCmd.ToManyRelation.class);
            toManyRelations1 = toManyRelations2.orElseThrow(() -> new RuntimeException("toManyRelation param is not valid"));
        }
        Optional<Map> extra2 = importRequest.getExtra() == null ? Optional.empty() : JsonHelper.fromJsonStr(importRequest.getExtra(), Map.class);

        //List<String> orderedCode = Optional.ofNullable(toManyRelations).orElseGet(Collections::emptyList).stream().sorted(Comparator.comparingInt(RelatedMapping::getSort)).map(x -> x.getCode()).collect(Collectors.toList());
        String extension = FilenameUtils.getExtension(Optional.ofNullable(importRequest.getFile().getOriginalFilename()).orElse(importRequest.getFile().getName()));

        ImportCmd importCmd = new ImportCmd(boId, importRequest.getV(), importRequest.getFile().getInputStream(), extension, importRequest.getFile().getOriginalFilename()
                , importRequest.getTimeout(), importRequest.getStep(), importRequest.getAsync(), importRequest.getAppId(), sheets1, toManyRelations1, importRequest.getImportMode(), importRequest.getUseBatch(), importRequest.getOnlyCheck(), extra2.orElseGet(null));

        Either<String, String> result = dispatcher.querySync(importCmd, BulkService.class, "batchImport");

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<Object> rep = new Response();
            rep.setCode("1");
            rep.setResult(x);
            rep.setMessage("OK");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<Object> rep = new Response();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str + ""));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }
}
