package com.xforceplus.ultraman.sdk.bulk.controller.config;

import akka.stream.ActorMaterializer;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.service.DictService;
import com.xforceplus.ultraman.sdk.bulk.controller.DefaultDownloadController;
import com.xforceplus.ultraman.sdk.bulk.controller.EntityBulkController;
import com.xforceplus.ultraman.sdk.core.bulk.BulkService;
import com.xforceplus.ultraman.sdk.core.bulk.exporter.*;
import com.xforceplus.ultraman.sdk.core.bulk.exporter.config.ExportConfig;
import com.xforceplus.ultraman.sdk.core.bulk.exporter.impl.*;
import com.xforceplus.ultraman.sdk.core.bulk.exporter.listener.ExportEventLoggerListener;
import com.xforceplus.ultraman.sdk.core.bulk.impl.BulkServiceImpl;
import com.xforceplus.ultraman.sdk.core.bulk.importer.ImportService;
import com.xforceplus.ultraman.sdk.core.bulk.importer.ImportServiceFactory;
import com.xforceplus.ultraman.sdk.core.bulk.importer.ImportTemplateService;
import com.xforceplus.ultraman.sdk.core.bulk.importer.impl.ImportDefaultExcelServiceImpl;
import com.xforceplus.ultraman.sdk.core.bulk.importer.impl.ImportDefaultExcelTemplateServiceImpl;
import com.xforceplus.ultraman.sdk.core.bulk.importer.impl.ImportWizardExcelServiceImpl;
import com.xforceplus.ultraman.sdk.core.facade.EntityFacade;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import com.xforceplus.ultraman.sdk.core.config.ExecutionConfig;
import com.xforceplus.ultraman.sdk.infra.base.id.IdGenerator;
import com.xforceplus.ultraman.sdk.infra.base.thread.ExecutorHelper;
import com.xforceplus.ultraman.sdk.infra.event.EventPublisher;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;

import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@ConditionalOnProperty(value = {"xplat.oqsengine.sdk.endpoints.enabled", "xplat.oqsengine.sdk.enabled"}, matchIfMissing = true)
public class BulkControllerAutoConfiguration {

    @Bean
    public ExportSource sequenceExportSource(EntityFacade entityFacade, ContextService contextService, ExportConfig exportConfig, ExecutionConfig executionConfig) {
        return new SequenceExportSource(entityFacade, exportConfig.getStepSize(), exportConfig.getMaxRetry(), contextService, executionConfig);
    }

    @Bean
    public ExportSource nestedExportSource(List<SequenceExportSource> sourceList, EntityFacade entityFacade, ProfileFetcher fetcher, ContextService contextService, ExportConfig exportConfig, ExecutionConfig executionConfig) {
        return new NestedExportSource(sourceList, entityFacade, fetcher, exportConfig, contextService, executionConfig);
    }

    @Bean("bulkThreadPool")
    public ExecutorService oqsThreadPool(@Value("${xplat.oqsengine.sdk.import.pool:20}") int worker, @Value("${xplat.oqsengine.sdk.import.queue:500}") int queue) {
        int useWorker = worker;
        int useQueue = queue;
        if (useWorker == 0) {
            useWorker = Runtime.getRuntime().availableProcessors() + 1;
        }

        if (useQueue < 500) {
            useQueue = 500;
        }

        return buildThreadPool(useWorker, useQueue, "bulk-common", false);
    }

    private ExecutorService buildThreadPool(int worker, int queue, String namePrefix, boolean daemon) {
        return new ThreadPoolExecutor(worker, worker, 0L, TimeUnit.MILLISECONDS, new ArrayBlockingQueue<>(queue), ExecutorHelper.buildNameThreadFactory(namePrefix, daemon), new ThreadPoolExecutor.AbortPolicy());
    }

    @Bean("defaultExportCustomFieldToString")
    public DefaultExportCustomFieldToString defaultExportCustomFieldToString(DictService dictService, ProfileFetcher fetcher, EntityFacade entityFacade) {
        return new DefaultExportCustomFieldToString(dictService, entityFacade, fetcher);
    }

    @Bean
    public ImportServiceFactory importServiceFactory(List<ImportService> importServiceList) {
        return new ImportServiceFactory(importServiceList);
    }

    @Bean
    public ImportTemplateService importTemplateService(EntityFacade entityFacade
            , @Autowired(required = false) ContextService contextService
            , ExportSink sink
            , ActorMaterializer mat
            , @Qualifier("bulkThreadPool") ExecutorService importThreadPool
    ) {
        return new ImportDefaultExcelTemplateServiceImpl(entityFacade, contextService, sink, mat, importThreadPool);
    }

    @Bean
    public ExportStringTransformer stringTransformer(List<ExportCustomFieldToString> customTransformers, @Qualifier("defaultExportCustomFieldToString") ExportCustomFieldToString exportCustomFieldToString) {
        return new ExportStringTransformerImpl(customTransformers, exportCustomFieldToString);
    }

    @ConditionalOnMissingBean(ExportSink.class)
    @ConditionalOnProperty(value = "xplat.oqsengine.sdk.export.local-sink", matchIfMissing = true)
    @Bean
    public ExportSink localFileSink(@Value("${xplat.oqsengine.sdk.export.local.root:/}") String root) {
        return new LocalFileExportSink(root);
    }

    @Bean
    public ExportService csvExportService(List<ExportSource> sourceList, ActorMaterializer materializer, ExportSink exportSink, ExportStringTransformer stringTransformer, @Autowired(required = false) ExportCallBack exportCallBack) {
        return new CSVEntityExportServiceImpl(sourceList, exportSink, exportCallBack, stringTransformer, materializer);
    }

    @Bean
    public ExportService excelExportService(ActorMaterializer materializer, List<ExportSource> sourceList, ExportSink exportSink, ExportStringTransformer stringTransformer, ExportConfig exportConfig, @Autowired(required = false) ExportCallBack exportCallBack, @Qualifier("bulkThreadPool") ExecutorService executorService) {

        return new ExcelEntityExportServiceImpl(sourceList, exportSink, exportCallBack, stringTransformer, exportConfig.getExcelConfig(), materializer, executorService);
    }

    @Bean
    @Primary
    public ImportService importDefaultExcelServiceImpl(EntityFacade entityFacade, @Autowired(required = false) ContextService contextService, @Qualifier("bulkThreadPool") ExecutorService importThreadPool, EventPublisher publisher) {
        return new ImportDefaultExcelServiceImpl(entityFacade, contextService, importThreadPool, publisher);
    }

    @Bean
    public ImportService importWizardSapExcelServiceImpl(EntityFacade entityFacade, @Autowired(required = false) ContextService contextService, @Qualifier("bulkThreadPool") ExecutorService importThreadPool, IdGenerator<Long> idGenerator, EventPublisher publisher, ExportSink sink, ActorMaterializer mat) {
        return new ImportWizardExcelServiceImpl(entityFacade, contextService, importThreadPool, idGenerator, publisher, sink, mat);
    }

    @ConditionalOnMissingBean(DownloadFilenameGenerator.class)
    @Bean
    public DownloadFilenameGenerator defaultFilenameGenerator() {
        return new DefaultFilenameGenerator();
    }

    @ConditionalOnMissingBean(BulkService.class)
    @Bean
    public BulkService bulkService(EntityFacade entityFacade, ImportTemplateService importTemplateService, ImportServiceFactory importServiceFactory, List<ExportService> exportService, ContextService contextService, ExportConfig exportConfig, DownloadFilenameGenerator filenameGenerator, ProfileFetcher fetcher, EntityClassEngine engine) {
        return new BulkServiceImpl(entityFacade, importTemplateService, exportService, contextService, exportConfig, filenameGenerator, fetcher, engine, importServiceFactory);
    }

    @ConditionalOnMissingBean(ExportCallBack.class)
    @Bean
    public ExportCallBack exportCallBack() {
        return new DefaultExportCallBack();
    }


    @ConditionalOnProperty(value = "xplat.oqsengine.sdk.export.log", matchIfMissing = true)
    @Bean
    public ExportEventLoggerListener loggerListener() {
        return new ExportEventLoggerListener();
    }

    @Bean
    public EntityBulkController entityBulkController(ExportConfig exportConfig) {
        return new EntityBulkController(exportConfig.getMaxPageSize());
    }

    @Bean
    public DefaultDownloadController defaultDownloadController(ExportSink exportSink, ImportService importService, ImportTemplateService templateService) {
        return new DefaultDownloadController(exportSink, importService, templateService);
    }
}
