package com.xforceplus.ultraman.sdk.graphql.gen;

import com.google.common.collect.Sets;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.*;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.EntityField;
import com.xforceplus.ultraman.sdk.graphql.input.*;
import graphql.scalars.ExtendedScalars;
import graphql.scalars.java.JavaPrimitives;
import graphql.scalars.object.JsonScalar;
import graphql.schema.*;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

import static com.xforceplus.metadata.schema.rels.MetadataRelationType.TO_MANY;
import static com.xforceplus.metadata.schema.rels.MetadataRelationType.TO_ONE;
import static com.xforceplus.ultraman.sdk.graphql.utils.GraphQLHelper.SUB_TOKEN;
import static graphql.Scalars.*;
import static graphql.introspection.Introspection.DirectiveLocation.MUTATION;
import static graphql.schema.FieldCoordinates.coordinates;
import static graphql.schema.GraphQLCodeRegistry.newCodeRegistry;
import static graphql.schema.GraphQLFieldDefinition.newFieldDefinition;
import static graphql.schema.GraphQLInputObjectField.newInputObjectField;
import static graphql.schema.GraphQLInputObjectType.newInputObject;
import static graphql.schema.GraphQLObjectType.newObject;

/**
 * TODO make this factory registerable
 * input types
 * including:
 * filter
 * order
 */
@Slf4j
public class OqsTypesFactory {

    private EntityClassEngine engine;

    private DataFetcher oqsDataFetcher;

    private Map<Long, GraphQLInputType> inputTypeCache;

    private final static String DescriptionTemplate = "id:%s, name:%s";

    private final static String FILTER_NAME = "%sFilter";

    private final static String CREATE_NAME = "%sAdd";

    private final static String BATCH_CREATE_NAME = "%sBatchAdd";

    private final static String UPDATE_NAME = "%sUpdate";

    private final static String BATCH_UPDATE_NAME = "%sBatchUpdate";

    private final static String DELETE_NAME = "%sDelete";

    private final static String BATCH_DELETE_NAME = "%sBatchDelete";

    private final static String COMPANION_NAME = "%sObj";

    public final static String CODE_ARG = "code";

    public final static String TRANSACTIONAL = "transactional";

    public final static String FILTER_ARG = "_filter";

    public final static String ORDER_ARG = "_order";

    public final static String INPUT_ARG = "_input";

    public final static String ID_ARG = "id";

    private final static GraphQLArgument ID = GraphQLArgument.newArgument()
            .name(ID_ARG)
            .type(GraphQLID)
            .build();

    private final static GraphQLArgument IDS = GraphQLArgument.newArgument()
            .name(ID_ARG)
            .type(GraphQLList.list(GraphQLID))
            .build();

    private final static GraphQLArgument PAGE_NO = GraphQLArgument.newArgument()
            .name("pageNo")
            .type(GraphQLInt)
            .build();

    private final static GraphQLArgument PAGE_SIZE = GraphQLArgument.newArgument()
            .name("pageSize")
            .type(GraphQLInt)
            .build();

    private final static GraphQLInputType ORDER_INPUT = GraphQLList.list(GraphQLInputObjectType
            .newInputObject()
            .name("_orderList")
            .field(newInputObjectField()
                    .name("code")
                    .type(GraphQLString).build())
            .field(newInputObjectField()
                    .name(ORDER_ARG)
                    .type(GraphQLEnumType.newEnum().name(ORDER_ARG).value("ASC", "asc").value("DESC", "desc").build()).build())
            .build());

    private final static GraphQLArgument ORDER = GraphQLArgument
            .newArgument()
            .name(ORDER_ARG)
            .type(ORDER_INPUT)
            .build();

    /**
     * @Coded(code:"xxx") to indicate which entityclass is related to this
     */

//    public final static GraphQLAppliedDirective CODED_APPLIED = GraphQLAppliedDirective.newDirective()
//            .name("coded")
//            .build();

    public final static GraphQLDirective TRANSACTIONAL_D = GraphQLDirective.newDirective()
            .name(TRANSACTIONAL)
            .validLocations(MUTATION)
            .build();

//    public final static GraphQLDirective REMOTE = GraphQLDirective.newDirective()
//            .name("remote")
//            .validLocations(FIELD, FRAGMENT_SPREAD, INLINE_FRAGMENT)
//            .build();

//    public final static GraphQLDirective ORIGIN = GraphQLDirective.newDirective()
//            .name("origin")
//            .validLocations(FIELD, FRAGMENT_SPREAD, INLINE_FRAGMENT)
//            .build();


//    public final static GraphQLDirective UNBOX = GraphQLDirective.newDirective()
//            .name("unbox")
//            .description("unbox from the list")
//            .validLocations(FIELD, FRAGMENT_SPREAD, INLINE_FRAGMENT)
//            .build();


    /**
     * applied
     */

    public final static GraphQLAppliedDirective REMOTE_APPLIED = GraphQLAppliedDirective.newDirective()
            .name("remote")
            .build();

    public final static GraphQLAppliedDirective TRANSACTIONAL_D_APPLIED = GraphQLAppliedDirective.newDirective()
            .name(TRANSACTIONAL)
            .build();
    /**
     * origin directive
     */
    public final static GraphQLAppliedDirective ORIGIN_APPLIED = GraphQLAppliedDirective.newDirective()
            .name("origin")
            .build();

    public final static GraphQLAppliedDirective CODED_APPLIED = GraphQLAppliedDirective.newDirective()
            .name("coded")
            .build();

    public final static GraphQLAppliedDirective CUSTOM_APPLIED = GraphQLAppliedDirective.newDirective()
            .name("custom")
            .build();

    public final static GraphQLAppliedDirective ONE2ONE_APPLIED = GraphQLAppliedDirective.newDirective()
            .name("one2one")
            .build();

    public final static GraphQLAppliedDirective ONE2MANY_APPLIED = GraphQLAppliedDirective.newDirective()
            .name("one2many")
            .build();

    public final static GraphQLAppliedDirective MULTI_VALUES_APPLIED = GraphQLAppliedDirective.newDirective()
            .name("multi")
            .build();

    public final static GraphQLAppliedDirective ADD = GraphQLAppliedDirective.newDirective()
            .name("add")
            .build();

    public final static GraphQLAppliedDirective BATCH_ADD = GraphQLAppliedDirective.newDirective()
            .name("batchAdd")
            .build();

    public final static GraphQLAppliedDirective UPDATE = GraphQLAppliedDirective.newDirective()
            .name("update")
            .build();

    public final static GraphQLAppliedDirective BATCH_UPDATE = GraphQLAppliedDirective.newDirective()
            .name("batchUpdate")
            .build();

    public final static GraphQLAppliedDirective DELETE = GraphQLAppliedDirective.newDirective()
            .name("delete")
            .build();

    public final static GraphQLAppliedDirective BATCH_DELETE = GraphQLAppliedDirective.newDirective()
            .name("batchDelete")
            .build();

    public final static GraphQLAppliedDirective UNBOX_APPLIED = GraphQLAppliedDirective.newDirective()
            .name("unbox")
            .description("unbox from the list")
            .build();

    public OqsTypesFactory(EntityClassEngine engine, OqsDataFetcher oqsDataFetcher) {
        this.engine = engine;
        this.oqsDataFetcher = oqsDataFetcher;
    }

    public GraphQLSchema genSchema(String profile) {
        List<IEntityClass> allEntities = engine.findAllEntities(profile);

        GraphQLObjectType.Builder queryBuilder = newObject().name("Query");
        GraphQLObjectType.Builder mutationBuilder = newObject().name("Mutation");

        /**
         * code to do with Interface
         */
        GraphQLCodeRegistry.Builder registryBuilder = newCodeRegistry();

        List<GraphQLInterfaceType> infList = new ArrayList<>();

        Map<String, GraphQLObjectType> typeMapping = new HashMap<>();

        Map<String, GraphQLInputType> inputTypeMapping = new HashMap<>();

        Map<String, GraphQLObjectType> listTypeMapping = new HashMap<>();

        Map<String, GraphQLInterfaceType> interfaceMapping = new HashMap<>();

        allEntities.stream().map(x -> genGraphQLGen(x, profile)).forEach(x -> {
            //append Query
            queryBuilder.field(x.getQueryType());
            //append Create
            mutationBuilder.field(x.getCreateType());
            //append Update
            mutationBuilder.field(x.getUpdateType());
            //append Delete
            mutationBuilder.field(x.getDeleteType());

            mutationBuilder.field(x.getBatchCreateType());

            mutationBuilder.field(x.getBatchUpdateType());

            mutationBuilder.field(x.getBatchDeleteType());

            x.getCustomTypes().forEach(mutationBuilder::field);

            registryBuilder.dataFetcher(coordinates("Query", x.getQueryType().getName())
                    , oqsDataFetcher);
            registryBuilder.dataFetcher(coordinates("Mutation", x.getCreateType().getName())
                    , oqsDataFetcher);
            registryBuilder.dataFetcher(coordinates("Mutation", x.getUpdateType().getName())
                    , oqsDataFetcher);
            registryBuilder.dataFetcher(coordinates("Mutation", x.getDeleteType().getName())
                    , oqsDataFetcher);
            registryBuilder.dataFetcher(coordinates("Mutation", x.getBatchDeleteType().getName())
                    , oqsDataFetcher);
            registryBuilder.dataFetcher(coordinates("Mutation", x.getBatchUpdateType().getName())
                    , oqsDataFetcher);
            registryBuilder.dataFetcher(coordinates("Mutation", x.getBatchCreateType().getName())
                    , oqsDataFetcher);

            /**
             * register custom
             */
            x.getCustomTypes().forEach(custom -> {
                registryBuilder.dataFetcher(coordinates("Mutation", custom.getName())
                        , oqsDataFetcher);
            });

            GraphQLObjectType objectType = x.getObjectType();
            objectType.getFieldDefinitions().stream().forEach(field -> {
                GraphQLAppliedDirective one2many = field.getAppliedDirective("one2many");
                GraphQLAppliedDirective one2one = field.getAppliedDirective("one2one");
                if (one2many != null) {
                    registryBuilder.dataFetcher(coordinates(objectType.getName(), field.getName()), oqsDataFetcher);
                } else if (one2one != null) {
                    registryBuilder.dataFetcher(coordinates(objectType.getName(), field.getName()), oqsDataFetcher);
                }
            });

            inputTypeMapping.put(x.getFilterTypeName(), x.getFilter());
            typeMapping.put(x.getTypeName(), x.getObjectType());
            listTypeMapping.put(x.getTypeName(), x.getListType());

            if (x.getInterfaceType() != null) {
                infList.add(x.getInterfaceType());
                interfaceMapping.put(x.getTypeName(), x.getInterfaceType());
                GraphQLInterfaceType interfaceType = x.getInterfaceType();
                interfaceType.getFieldDefinitions().stream().forEach(field -> {
                    GraphQLAppliedDirective one2many = field.getAppliedDirective("one2many");
                    GraphQLAppliedDirective one2one = field.getAppliedDirective("one2one");
                    if (one2many != null) {
                        registryBuilder.dataFetcher(coordinates(interfaceType.getName(), field.getName()), oqsDataFetcher);
                    } else if (one2one != null) {
                        registryBuilder.dataFetcher(coordinates(interfaceType.getName(), field.getName()), oqsDataFetcher);
                    }
                });
            }
        });

        /**
         * object mapping
         */
        infList.forEach(x -> {
            //do interface resolver
            registryBuilder.typeResolver(x, env -> {
                Object ret = env.getObject();
                if (ret instanceof Map) {
                    Object entityClassId = ((Map<String, Object>) ret).get("_entityClassId");
                    if (entityClassId != null) {
                        Optional<IEntityClass> retType = engine.load(entityClassId.toString(), "");
                        if (retType.isPresent()) {
                            return typeMapping.get(retType.get().code());
                        }
                    }
                }
                //TODO
                throw new RuntimeException("Not a Map");
            });
        });

        GraphQLSchema schema = GraphQLSchema
                .newSchema()
                .codeRegistry(registryBuilder.build())
                .query(queryBuilder)
                .additionalTypes(typeMapping.values().stream().collect(Collectors.toSet()))
                .additionalTypes(inputTypeMapping.values().stream().collect(Collectors.toSet()))
                .additionalTypes(interfaceMapping.values().stream().collect(Collectors.toSet()))
                .additionalTypes(listTypeMapping.values().stream().collect(Collectors.toSet()))
                .additionalDirectives(Sets.newHashSet(TRANSACTIONAL_D))
                .mutation(mutationBuilder)
                .build();
        return schema;
    }

    public GraphQLGenTypes genGraphQLGen(IEntityClass entityClass, String profile) {
        GraphQLGenTypes types = new GraphQLGenTypes();
        //TODO how dynamic profile
        IEntityClass targetEntityClass = entityClass;
        //TODO
        EntityClassGroup group = engine.describe(targetEntityClass, profile);

        types.setTypeName(targetEntityClass.code());

        if (!group.getChildrenEntityClass().isEmpty()) {
            //current is a interface
            types.setInterfaceType(genInterfaceType(group));
        }

        String code = group.getEntityClass().code();
        String filterName = String.format(FILTER_NAME, code);
        types.setFilterTypeName(filterName);
        types.setFilter(genFilter(group));
        types.setObjectType(genObjectType(group));
        types.setQueryType(genQueryType(group));

        types.setCustomTypes(genCustomType(group));
        types.setBatchCreateType(genBatchCreateType(group));
        types.setCreateType(genCreateType(group));
        types.setBatchUpdateType(genBatchUpdateType(group));
        types.setUpdateType(genUpdateType(group));
        types.setBatchDeleteType(genBatchDeleteType(group));
        types.setDeleteType(genDeleteType(group));
        types.setListType(genListType(group));
        return types;
    }

    private GraphQLObjectType genListType(EntityClassGroup group) {
        String objectName = group.getEntityClass().code();

        return newObject()
                .name(objectName.concat("List"))
                .field(newFieldDefinition()
                        .name("totalCount")
                        .type(ExtendedScalars.GraphQLLong)
                        .build())
                .field(newFieldDefinition().name("row").type(GraphQLList.list(GraphQLTypeReference.typeRef(objectName)))
                        .build())
                .build();
    }

    /**
     * TODO
     * gen a query field
     * name(filter: nameFilter, order:Order, pageNo:PageNo, pageSize:PageSize)
     *
     * @param group
     * @return
     */
    private GraphQLFieldDefinition genQueryType(EntityClassGroup group) {
        String name = group.getEntityClass().code();
        String filterName = String.format(FILTER_NAME, name);

        GraphQLArgument filter = GraphQLArgument.newArgument()
                .name(FILTER_ARG)
                .type(GraphQLTypeReference.typeRef(filterName))
                .build();

        return newFieldDefinition()
                .name(name)
                .arguments(Arrays.asList(filter, ORDER, PAGE_NO, PAGE_SIZE))
                .type(GraphQLTypeReference.typeRef(name + "List"))
                .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                        .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                .name(CODE_ARG)
                                .type(GraphQLString)
                                .valueProgrammatic(name).build())).build();
    }

    private List<GraphQLFieldDefinition> genCustomType(EntityClassGroup group) {
        String name = group.getEntityClass().code();

        Set<String> actions = engine.findCustomActionsById(group.getEntityClass().id());

//        /**
//         * TODO inherit
//         */
//        Set<String> actions = group.getEntityClass().actions();

        return actions.stream().map(x -> {
            String actionName = name.concat(x.substring(0, 1).toUpperCase().concat(x.substring(1)));
            GraphQLFieldDefinition fieldDefinition = newFieldDefinition()
                    .name(actionName)
                    .argument(GraphQLArgument
                            .newArgument().name(INPUT_ARG)
                            .type(JsonScalar.INSTANCE).build())
                    /**
                     * inline code directive
                     */
                    .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                            .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                    .name(CODE_ARG)
                                    .type(GraphQLString)
                                    .valueProgrammatic(name).build()).build())
                    .withAppliedDirective(GraphQLAppliedDirective.newDirective(CUSTOM_APPLIED)
                            .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                    .name(CODE_ARG)
                                    .type(GraphQLString)
                                    .valueProgrammatic(x)).build())
                    .type(GraphQLString)
                    .build();

            return fieldDefinition;
        }).collect(Collectors.toList());
    }

    private GraphQLFieldDefinition genBatchCreateType(EntityClassGroup group) {
        String name = group.getEntityClass().code();
        String createName = String.format(BATCH_CREATE_NAME, name);
        GraphQLFieldDefinition fieldDefinition = newFieldDefinition()
                .name(createName)
                .argument(GraphQLArgument
                        .newArgument().name(INPUT_ARG)
                        .type(GraphQLList.list(JsonScalar.INSTANCE))
                        .build())
                /**
                 * inline code directive
                 */
                .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                        .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                .name(CODE_ARG)
                                .type(GraphQLString)
                                .valueProgrammatic(name).build()).build())
                .withAppliedDirective(BATCH_ADD)
                .type(GraphQLString)
                .build();

        return fieldDefinition;
    }

    /**
     * create is gen a mutation
     * <p>
     * nameAdd(input:{})
     *
     * @param group
     * @return
     */
    private GraphQLFieldDefinition genCreateType(EntityClassGroup group) {
        String name = group.getEntityClass().code();
        String createName = String.format(CREATE_NAME, name);
        GraphQLFieldDefinition fieldDefinition = newFieldDefinition()
                .name(createName)
                .argument(GraphQLArgument
                        .newArgument().name(INPUT_ARG)
                        .type(JsonScalar.INSTANCE).build())
                /**
                 * inline code directive
                 */
                .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                        .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                .name(CODE_ARG)
                                .type(GraphQLString)
                                .valueProgrammatic(name).build()).build())
                .withAppliedDirective(ADD)
                //TODO more complex ?
                .type(GraphQLString)
                .build();

        return fieldDefinition;
    }

    private GraphQLFieldDefinition genBatchUpdateType(EntityClassGroup group) {
        String name = group.getEntityClass().code();
        String operationName = String.format(BATCH_UPDATE_NAME, name);
        GraphQLFieldDefinition fieldDefinition = newFieldDefinition()
                .name(operationName)
                .argument(GraphQLArgument
                        .newArgument().name(INPUT_ARG)
                        .type(GraphQLList.list(JsonScalar.INSTANCE))
                        .build())
                /**
                 * inline code directive
                 */
                .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                        .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                .name(CODE_ARG)
                                .type(GraphQLString)
                                .valueProgrammatic(name).build()).build())
                .withAppliedDirective(BATCH_UPDATE)
                //TODO more complex ?
                .type(GraphQLString)
                .build();

        return fieldDefinition;
    }


    /**
     * gen update
     * nameUpdate(id:Id, input)
     *
     * @param group
     * @return
     */
    private GraphQLFieldDefinition genUpdateType(EntityClassGroup group) {
        String name = group.getEntityClass().code();
        String createName = String.format(UPDATE_NAME, name);
        GraphQLFieldDefinition fieldDefinition = newFieldDefinition()
                .name(createName)
                .argument(ID)
                .argument(GraphQLArgument
                        .newArgument().name(INPUT_ARG)
                        .type(JsonScalar.INSTANCE).build())
                //TODO more complex ?
                .type(GraphQLString)
                .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                        .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                .name(CODE_ARG)
                                .type(GraphQLString)
                                .valueProgrammatic(name).build()).build())
                .withAppliedDirective(UPDATE)
                .build();

        return fieldDefinition;
    }

    private GraphQLFieldDefinition genBatchDeleteType(EntityClassGroup group) {
        String name = group.getEntityClass().code();
        String deleteName = String.format(BATCH_DELETE_NAME, name);
        GraphQLFieldDefinition fieldDefinition = newFieldDefinition()
                .name(deleteName)
                .argument(IDS)
                //TODO more complex ?
                .type(GraphQLString)
                .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                        .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                .name(CODE_ARG)
                                .type(GraphQLString)
                                .valueProgrammatic(name).build()).build())
                .withAppliedDirective(BATCH_DELETE)
                .build();

        return fieldDefinition;
    }

    private GraphQLFieldDefinition genDeleteType(EntityClassGroup group) {
        String name = group.getEntityClass().code();
        String deleteName = String.format(DELETE_NAME, name);
        GraphQLFieldDefinition fieldDefinition = newFieldDefinition()
                .name(deleteName)
                .argument(ID)
                //TODO more complex ?
                .type(GraphQLString)
                .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                        .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                .name(CODE_ARG)
                                .type(GraphQLString)
                                .valueProgrammatic(name).build()).build())
                .withAppliedDirective(DELETE)
                .build();

        return fieldDefinition;
    }

    /**
     * gen filter from boUp
     *
     * @return
     */
    private GraphQLInputObjectType genFilter(EntityClassGroup group) {
        String code = group.getEntityClass().code();
        String filterName = String.format(FILTER_NAME, code);

        /**
         * all plain fields in current and above
         */
        Collection<IEntityField> allFields = group.getAllFields();
        List<GraphQLInputObjectField> plainFields = allFields.stream().map(x -> {

            if (x.name().contains(".")) {
                GraphQLInputObjectField.Builder builder = newInputObjectField();
                //TODO
                builder.type(fieldTypeToFilterType(x.type()));
                String replace = x.name().replace('.', '_');
                return builder.name("_".concat(replace))
                        .withAppliedDirective(GraphQLAppliedDirective.newDirective(ORIGIN_APPLIED)
                                .argument(GraphQLAppliedDirectiveArgument.newArgument().name(CODE_ARG)
                                        .type(GraphQLString).valueProgrammatic(x.name())).build())
                        .build();
            } else {
                //map field to field definition
                GraphQLInputObjectField.Builder builder = newInputObjectField()
                        .name(x.name())
                        .description(x.cnName());
                GraphQLInputType fieldType = fieldTypeToFilterType(x.type());
                builder = builder.type(fieldType);
                return builder.build();
            }
        }).collect(Collectors.toList());

        Collection<IEntityClass> childrenEntityClass = group.getChildrenEntityClass();

        List<GraphQLInputObjectField> childInputObjectFields = Optional.ofNullable(childrenEntityClass).orElseGet(Collections::emptyList)
                .stream().flatMap(child -> {
                    String childCode = child.code();
                    Collection<IEntityField> fields = child.fields();
                    return fields.stream().map(x -> {
                        String name = x.name();
                        //remove all name contains
                        if (!name.contains(".")) {
                            String childFilterName = childCode.concat(SUB_TOKEN).concat(name);
                            GraphQLInputObjectField.Builder builder = newInputObjectField()
                                    .name(childFilterName)
                                    .description(x.cnName());
                            GraphQLInputType fieldType = fieldTypeToFilterType(x.type());
                            builder = builder.type(fieldType);
                            return builder.build();
                        }
                        return null;
                    }).filter(Objects::nonNull);
                }).collect(Collectors.toList());

        /**
         * if has same code
         */
        List<GraphQLInputObjectField> relatedFilter = group.getAllRelations().stream().map(rel -> {
            Optional<IEntityClass> related = group.relatedEntityClass(rel.getName());
            if (related.isPresent()) {
                String referFilter = String.format(FILTER_NAME, related.get().code());
                return newInputObjectField()
                        .name(rel.getName())
                        .type(GraphQLTypeReference
                                .typeRef(referFilter)).build();
            } else {
                log.warn("Related EntityClass not present {}", rel.getName());
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());

        List<GraphQLInputObjectField> fields = new ArrayList<>(plainFields.size() + relatedFilter.size() + 2);
        fields.addAll(plainFields);
        fields.addAll(childInputObjectFields);
        fields.addAll(relatedFilter);

        //and filter
        GraphQLInputObjectField and = newInputObjectField()
                .name("and")
                .type(GraphQLList.list(GraphQLTypeReference.typeRef(filterName)))
                .build();

        //or filter
        GraphQLInputObjectField or = newInputObjectField()
                .name("or")
                .type(GraphQLList.list(GraphQLTypeReference.typeRef(filterName)))
                .build();

        fields.add(and);
        fields.add(or);

        return newInputObject()
                .name(filterName)
                .fields(fields)
                .build();
    }

    private GraphQLInputType fieldTypeToFilterType(FieldType type) {
        GraphQLInputType fieldType = null;
        switch (type) {
            case STRINGS:
                fieldType = StringsFilter.INSTANCE;
                break;
            case LONG:
                fieldType = LongFilter.INSTANCE;
                break;
            case DECIMAL:
                fieldType = DecimalFilter.INSTANCE;
                break;
            case DATETIME:
                fieldType = LongFilter.INSTANCE;
                break;
            case BOOLEAN:
                fieldType = BooleanFilter.INSTANCE;
                break;
            case ENUM:
            case STRING:
            default:
                fieldType = StringFilter.INSTANCE;
        }

        return fieldType;
    }

    /**
     * @param type
     * @return
     */
    private GraphQLOutputType fieldTypeToObjectType(FieldType type) {
        GraphQLOutputType fieldType = null;
        switch (type) {
            case STRINGS:
                //fieldType = GraphQLList.list(GraphQLString);
                fieldType = GraphQLString;
                break;
            case LONG:
                fieldType = JavaPrimitives.GraphQLLong;
                break;
            case DECIMAL:
                fieldType = JavaPrimitives.GraphQLBigDecimal;
                break;
            case DATETIME:
                fieldType = JavaPrimitives.GraphQLLong;
                break;
            case BOOLEAN:
                fieldType = GraphQLBoolean;
                break;
            case ENUM:
            case STRING:
            default:
                fieldType = GraphQLString;
        }

        return fieldType;
    }

    /**
     * interface
     *
     * @param group
     * @return
     */
    private GraphQLInterfaceType genInterfaceType(EntityClassGroup group) {
        String name = group.getEntityClass().code();
        Collection<IEntityClass> fatherEntityClass = group.getFatherEntityClass();
        Collection<GraphQLFieldDefinition> fieldDefinitions = fieldsToGraphQLFieldDefinition(group.getAllFields());

        Map<String, List<IRelation>> relationsMapping = group.getAllRelations().stream()
                .collect(Collectors.groupingBy(x -> Optional.ofNullable(x.getRelationType()).orElse("")));

        List<GraphQLFieldDefinition> relatedFields = group.getAllRelations().stream().map(rel -> {
            Optional<IEntityClass> related = group.relatedEntityClass(rel.getName());
            if (related.isPresent()) {
                if (rel.getRelationType().equalsIgnoreCase(TO_MANY.toString())) {

                    //check if this rel is self-rel
                    boolean isSelf = Optional.ofNullable(relationsMapping.get(FieldLikeRelationType.MANY2ONE.getName())).orElseGet(Collections::emptyList)
                            .stream().anyMatch(x -> x.getName().equalsIgnoreCase(rel.getName().concat("MTO")));

                    return newFieldDefinition()
                            .name(rel.getName())
                            .withAppliedDirective(GraphQLAppliedDirective.newDirective(ONE2MANY_APPLIED)
                                    .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                            .name(CODE_ARG)
                                            .type(GraphQLString)
                                            .valueProgrammatic(isSelf ? rel.getName().concat("MTO") : rel.getName()).build()).build())
                            .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                                    .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                            .name(CODE_ARG)
                                            .type(GraphQLString)
                                            .valueProgrammatic(name).build()).build())
                            .type(GraphQLList.list(GraphQLTypeReference
                                    .typeRef(related.get().code())))
                            .build();
                } else if (rel.getRelationType().equalsIgnoreCase(TO_ONE.toString())) {
                    return newFieldDefinition()
                            .name(rel.getName())
                            .withAppliedDirective(ONE2ONE_APPLIED)
                            .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                                    .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                            .name(CODE_ARG)
                                            .type(GraphQLString)
                                            .valueProgrammatic(name).build()).build())
                            .type(GraphQLTypeReference
                                    .typeRef(related.get().code()))
                            .build();
                } else if (rel.getRelationType().equalsIgnoreCase(FieldLikeRelationType.MULTI_VALUES.getName())) {
                    //TODO
                    return newFieldDefinition()
                            .name(rel.getName())
                            .withAppliedDirective(MULTI_VALUES_APPLIED)
                            .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                                    .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                            .name(CODE_ARG)
                                            .type(GraphQLString)
                                            .valueProgrammatic(name).build()).build())
                            .type(GraphQLTypeReference
                                    .typeRef(related.get().code()))
                            .build();
                } else {
                    log.warn("Related Type is unknown {}", rel.getName());
                    return null;
                }
            } else {
                log.warn("Related EntityClass not present {}", rel.getName());
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());
        GraphQLInterfaceType.Builder infBuilder = GraphQLInterfaceType.newInterface()
                .name(name)
                .fields(new ArrayList<>(fieldDefinitions))
                .fields(relatedFields);
        if (!fatherEntityClass.isEmpty()) {
            fatherEntityClass.forEach(x ->
                    infBuilder.withInterface(GraphQLTypeReference.typeRef(x.code())));
        }

        return infBuilder.build();
    }

    private List<GraphQLFieldDefinition> fieldsToGraphQLFieldDefinition(Collection<IEntityField> allFields) {
        List<GraphQLFieldDefinition> plainFields = allFields.stream()
                .map(x -> {
                    //escape the `.`
                    if (x.name().contains(".")) {
                        GraphQLFieldDefinition.Builder builder = newFieldDefinition();
                        builder.type(fieldTypeToObjectType(x.type()));
                        String replace = x.name().replace('.', '_');
                        return builder.name("_".concat(replace))
                                .withAppliedDirective(GraphQLAppliedDirective.newDirective(ORIGIN_APPLIED)
                                        .argument(GraphQLAppliedDirectiveArgument.newArgument().name(CODE_ARG)
                                                .type(GraphQLString)
                                                .valueProgrammatic(x.name()).build()).build())
                                .build();
                    } else {
                        //map field to field definition
                        GraphQLFieldDefinition.Builder builder = newFieldDefinition();
                        GraphQLOutputType fieldType = fieldTypeToObjectType(x.type());
                        builder.name(x.name());
                        builder = builder.type(fieldType);
                        return builder.build();
                    }
                }).collect(Collectors.toList());

        return plainFields;
    }

    /**
     * id is required primary
     */
    private Collection<IEntityField> makeupId(Collection<IEntityField> allFields) {
        Optional<IEntityField> fieldOp = allFields.stream().filter(x -> "id".equals(x.name())).findAny();
        if (!fieldOp.isPresent()) {
            List<IEntityField> newList = new ArrayList<>(allFields);
            newList.add(new EntityField(-1, "id", FieldType.STRING));
            return newList;
        }

        return allFields;
    }

    /**
     * gen object type
     *
     * @param group
     * @return
     */
    private GraphQLObjectType genObjectType(EntityClassGroup group) {
        String objectName = group.getEntityClass().code();

        boolean isCompanion = false;
        if (!group.getChildrenEntityClass().isEmpty()) {
            objectName = String.format(COMPANION_NAME, objectName);
            isCompanion = true;
        }

        /**
         * all plain fields
         */
        Collection<IEntityField> allFields = group.getAllFields();
        allFields = makeupId(allFields);

        Map<String, List<IRelation>> relationsMapping = group.getAllRelations().stream()
                .collect(Collectors.groupingBy(x -> Optional.ofNullable(x.getRelationType()).map(String::toLowerCase).orElse("")));

        //makeup fields
        List<GraphQLFieldDefinition> plainFields = fieldsToGraphQLFieldDefinition(allFields);
        List<GraphQLFieldDefinition> relatedFields = group.getAllRelations().stream().map(rel -> {
            Optional<IEntityClass> related = group.relatedEntityClass(rel.getName());
            if (related.isPresent()) {
                if (rel.getRelationType().equalsIgnoreCase(TO_MANY.name())) {

                    //check if this rel is self-rel
                    boolean isSelf = Optional.ofNullable(relationsMapping.get(FieldLikeRelationType.MANY2ONE.getName())).orElseGet(Collections::emptyList)
                            .stream().anyMatch(x -> x.getName().equalsIgnoreCase(rel.getName().concat("MTO")));

                    return newFieldDefinition()
                            .name(rel.getName())
                            .withAppliedDirective(GraphQLAppliedDirective.newDirective(ONE2MANY_APPLIED)
                                    .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                            .name(CODE_ARG)
                                            .type(GraphQLString)
                                            .valueProgrammatic(isSelf ? rel.getName().concat("MTO") : rel.getName()).build()).build())
                            .withAppliedDirective(UNBOX_APPLIED)
                            .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                                    .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                            .name(CODE_ARG)
                                            .type(GraphQLString)
                                            .valueProgrammatic(related.get().code()).build()).build())
                            .type(GraphQLList.list(GraphQLTypeReference
                                    .typeRef(related.get().code())))
                            .build();
                } else if (rel.getRelationType().equalsIgnoreCase(TO_ONE.name())) {
                    return newFieldDefinition()
                            .name(rel.getName())
                            .withAppliedDirective(ONE2ONE_APPLIED)
                            .withAppliedDirective(UNBOX_APPLIED)
                            .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                                    .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                            .name(CODE_ARG)
                                            .type(GraphQLString)
                                            .valueProgrammatic(related.get().code()).build()).build())
                            .type(GraphQLTypeReference
                                    .typeRef(related.get().code()))
                            .build();
                } else if (rel.getRelationType().equalsIgnoreCase(FieldLikeRelationType.MULTI_VALUES.getName())) {
                    return newFieldDefinition()
                            .name(rel.getName())
                            .withAppliedDirective(MULTI_VALUES_APPLIED)
                            .withAppliedDirective(GraphQLAppliedDirective.newDirective(CODED_APPLIED)
                                    .argument(GraphQLAppliedDirectiveArgument.newArgument()
                                            .name(CODE_ARG)
                                            .type(GraphQLString)
                                            .valueProgrammatic(related.get().code()).build()).build())
                            .type(GraphQLTypeReference
                                    .typeRef(related.get().code()))
                            .build();
                } else {
                    log.warn("Related Type is unknown {}", rel.getName());
                    return null;
                }
            } else {
                log.warn("Related EntityClass not present {}", rel.getName());
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());

        List<GraphQLFieldDefinition> fields = new ArrayList<>(plainFields.size() + relatedFields.size());

        fields.addAll(plainFields);
        fields.addAll(relatedFields);

        String objectCName = group.getEntityClass().name();
        long objectId = group.getEntityClass().id();

        String description = String.format(DescriptionTemplate, objectId, objectCName);

        GraphQLObjectType.Builder builder = newObject()
                .name(objectName)
                .description(description)
                .fields(fields);

        if (group.getEntityClass().getType() == 1) {
            builder.withAppliedDirective(REMOTE_APPLIED);
        }

        /**
         * add self
         */
        if (isCompanion) {
            builder.withInterface(GraphQLTypeReference.typeRef(group.getEntityClass().code()));
        }

        if (!group.getFatherEntityClass().isEmpty()) {
            group.getFatherEntityClass().forEach(parent -> {
                builder.withInterface(GraphQLTypeReference.typeRef(parent.code()));
            });
        }

        return builder.build();
    }
}
