package com.xforceplus.ultraman.sdk.graphql.utils;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.xforceplus.ultraman.metadata.domain.record.EmptyValue;
import com.xforceplus.ultraman.metadata.domain.vo.dto.*;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.helper.RequestBuilder;
import com.xforceplus.ultraman.sdk.core.cmd.*;
import com.xforceplus.ultraman.sdk.graphql.gen.OqsTypesFactory;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * GraphQL helper
 */
public class GraphQLHelper {

    public final static String SUB_TOKEN = "__SUB__";

    private final static String DELETE_TEMPLATE = "mutation { %sDelete(id:%s) }";

    private final static String UPDATE_TEMPLATE = "mutation { %sUpdate(id:%s, " + OqsTypesFactory.INPUT_ARG + ":%s) }";

    private final static String CREATE_TEMPLATE = "mutation { %sAdd(" + OqsTypesFactory.INPUT_ARG + ":%s) }";

    private final static String BATCH_DELETE_TEMPLATE = "mutation { %sBatchDelete(id:[%s]) }";

    private final static String BATCH_CREATE_TEMPLATE = "mutation { %sBatchAdd(" + OqsTypesFactory.INPUT_ARG + ":%s) }";

    private final static String BATCH_UPDATE_TEMPLATE = "mutation { %sBatchUpdate(" + OqsTypesFactory.INPUT_ARG + ":%s) }";

    private final static String ORDER_TEMPLATE = "{code:\"%s\", " + OqsTypesFactory.ORDER_ARG + ":%s}";

    private final static String CUSTOM_TEMPLATE = "mutation { %s%s(" + OqsTypesFactory.INPUT_ARG + ":%s) }";

    private static ObjectMapper mapper = new ObjectMapper();

    static {
        mapper.configure(JsonGenerator.Feature.QUOTE_FIELD_NAMES, false);
        SimpleModule simpleModule = new SimpleModule();
        simpleModule.addSerializer(EmptyValue.class, new JsonSerializer<EmptyValue>() {
            @Override
            public void serialize(EmptyValue emptyValue, JsonGenerator jsonGenerator, SerializerProvider serializerProvider) throws IOException {
                jsonGenerator.writeNull();
            }
        });

        mapper.registerModule(simpleModule);

    }

    /**
     * condition to graphQL String
     *
     * @param entityClass
     * @param cmd
     * @return
     */
    public static String conditionToQueryString(IEntityClass entityClass, EntityClassEngine engine, String profile, ConditionSearchCmd cmd, boolean isStrict) {
        ConditionQueryRequest conditionQueryRequest = cmd.getConditionQueryRequest();
        return toQueryString(entityClass, engine, profile, conditionQueryRequest, isStrict);
    }

    public static String singleUpdateString(IEntityClass entityClass, SingleUpdateCmd cmd) {
        String code = entityClass.code();
        Long id = cmd.getId();
        Map<String, Object> body = cmd.getBody();
        String jsonStr = toJson(normalize(body));
        return String.format(UPDATE_TEMPLATE, code, id, jsonStr);
    }

    public static String singleCreateString(IEntityClass entityClass, SingleCreateCmd cmd) {
        String code = entityClass.code();
        Map<String, Object> body = cmd.getBody();
        String jsonStr = toJson(normalize(body));
        return String.format(CREATE_TEMPLATE, code, jsonStr);
    }

    public static String batchDeleteString(IEntityClass entityClass, BatchDeleteCmd cmd) {
        String code = entityClass.code();
        List<String> ids = cmd.getIds();
        String idsStr = ids.stream().collect(Collectors.joining(","));
        return String.format(BATCH_DELETE_TEMPLATE, code, idsStr);
    }

    public static String batchCreateString(IEntityClass entityClass, BatchCreateCmd cmd) {
        String code = entityClass.code();
        List<Map<String, Object>> bodies = cmd.getBodies();
        List<Map<String, Object>> newBodies = bodies.stream().map(x -> normalize(x)).collect(Collectors.toList());
        String inputStr = toJson(newBodies);
        return String.format(BATCH_CREATE_TEMPLATE, code, inputStr);
    }

    /**
     * transform argument with dot‘.’ to underline‘_’
     *
     * @param argument
     * @return
     */
    public static Map<String, Object> normalize(Map<String, Object> argument) {
        Map<String, Object> map = new HashMap<>();
        argument.forEach((k, v) -> {
            String key = k;
            //
            if (k.contains(".")) {
                key = "_".concat(k.replace('.', '_'));
            } else if (k.contains("/") || k.contains(":")) {
                //sub field
                k = k.replaceFirst("/", SUB_TOKEN);
                k = k.replaceFirst(":", SUB_TOKEN);
                key = k;
            }
            map.put(key, v);
        });

        return map;
    }

    public static String normalize(String name) {
        if (name.contains(".")) {
            //related field
            return "_".concat(name.replace('.', '_'));
        } else if (name.contains("/") || name.contains(":")) {
            //sub field
            name = name.replaceFirst("/", SUB_TOKEN);
            name = name.replaceFirst(":", SUB_TOKEN);
            return name;
        } else {
            return name;
        }
    }

    public static Map<String, Object> reNormalize(Map<String, Object> argument) {
        Map<String, Object> map = new HashMap<>();
        argument.forEach((k, v) -> {
            map.put(reNormalize(k), v);
        });

        return map;
    }

    /**
     * unescape the name
     *
     * @param k
     * @return
     */
    public static String reNormalize(String k) {
        String key = k;
        if (key.startsWith("_")) {
            key = key.substring(1).replaceFirst("_", ".");
        }

        if (key.contains(SUB_TOKEN)) {
            key = key.replaceFirst(SUB_TOKEN, "/");
        }

        return key;
    }

    /**
     * @param entityClass
     * @param cmd
     * @return
     */
    public static String customString(IEntityClass entityClass, CustomActionCmd cmd) {
        String actionCode = cmd.getActionName();
        String actionCodeFormat = actionCode.substring(0, 1).toUpperCase() + actionCode.substring(1);
        String code = entityClass.code();
        //TODO is ok?
        Map<String, Object> customInput = cmd.getCustomInput();
        String inputStr = toJson(customInput);
        return String.format(CUSTOM_TEMPLATE, code, actionCodeFormat, inputStr);
    }

    public static String batchUpdateString(IEntityClass entityClass, BatchUpdateCmd cmd) {
        String code = entityClass.code();
        List<Map<String, Object>> bodies = cmd.getBodies();
        List<Map<String, Object>> newBodies = bodies.stream().map(x -> normalize(x)).collect(Collectors.toList());
        String inputStr = toJson(newBodies);
        return String.format(BATCH_UPDATE_TEMPLATE, code, inputStr);
    }

    private static String toJson(Object body) {
        try {
            String json = mapper.writeValueAsString(body);
            return json;
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    public static String singleDeleteString(IEntityClass entityClass, SingleDeleteCmd cmd) {
        String id = cmd.getId();
        String code = entityClass.code();
        return String.format(DELETE_TEMPLATE, code, id);
    }

    public static String singleQueryString(IEntityClass entityClass, EntityClassEngine engine, String profile, SingleQueryCmd cmd, boolean isStrict) {
        String id = cmd.getId();

        EntityClassGroup group = engine.describe(entityClass, profile);

        String[] codes = group.getAllFields().stream().map(x -> {
            String name = x.name();
            if (name.contains(".")) {
                return "_".concat(name.replace('.', '_'));
            } else {
                return name;
            }
        }).toArray(String[]::new);

        /**
         * shift
         */
        ConditionQueryRequest idRequest = new RequestBuilder()
                .field("id", ConditionOp.eq, id)
                .item(codes)
                .pageNo(1)
                .pageSize(1)
                .build();

        return toQueryString(entityClass, engine, profile, idRequest, isStrict);
    }

    /**
     * expand
     *
     * @param sub
     * @return
     */
    private static String expandSubField(String sub) {
        if (sub.contains(SUB_TOKEN)) {
            String[] split = sub.split(SUB_TOKEN);
            if (split.length > 1) {
                StringBuilder fragement = new StringBuilder();
                fragement.append("... on ");
                fragement.append(split[0]);
                fragement.append(" { ");
                //do the alias
                fragement.append(sub).append(":").append(split[1]);
                fragement.append(" } ");
                return fragement.toString();
            }
        }

        return sub;
    }

    /**
     * to condition
     * @param conditions
     * @return
     */
    private static String toCondition(Conditions conditions) {
        StringBuilder filterBuilder = new StringBuilder();
        Optional.ofNullable(conditions).ifPresent(x -> {
            if(conditions.getGroups() != null && !conditions.getGroups().isEmpty()) {
                String operation = conditions.getOperation();
                filterBuilder.append(operation);
                filterBuilder.append(":[");
                String condition = conditions.getGroups().stream().map(group -> toCondition(group))
                        .map(filter -> "{ ".concat(filter).concat("}"))
                        .collect(Collectors.joining(","));
                filterBuilder.append(condition);
                filterBuilder.append("]");
            } else {
                Optional.ofNullable(x.getFields()).orElseGet(Collections::emptyList).forEach(field -> {
                    String fieldCode = field.getCode();
                    ConditionOp operation = field.getOperation();
                    List<String> value = field.getValue();
                    filterBuilder.append(" ")
                            .append(normalize(fieldCode))
                            .append(":")
                            .append(" {")
                            .append(operation.name()).append(":").append(valueString(operation, value))
                            .append(" },");
                });

                Optional.ofNullable(x.getEntities()).orElseGet(Collections::emptyList).forEach(sub -> {
                    filterBuilder.append(sub.getCode()).append(":{");
                    Optional.ofNullable(sub.getFields()).ifPresent(subF -> {
                        subF.stream().forEach(field -> {
                            String fieldCode = field.getCode();
                            ConditionOp operation = field.getOperation();
                            List<String> value = field.getValue();
                            filterBuilder.append(" ")
                                    .append(normalize(fieldCode))
                                    .append(" :{")
                                    .append(operation.name()).append(":").append(valueString(operation, value))
                                    .append(" },");
                        });
                    });

                    Optional.ofNullable(sub.getEntities()).orElseGet(Collections::emptyList).forEach(nestedSub -> {
                        filterBuilder.append("")
                                        .append(nestedSub.getCode()).append(": {");
                        Optional.ofNullable(nestedSub.getFields()).ifPresent(subF -> {
                            subF.stream().forEach(field -> {
                                String fieldCode = field.getCode();
                                ConditionOp operation = field.getOperation();
                                List<String> value = field.getValue();
                                filterBuilder.append(" ")
                                        .append(normalize(fieldCode))
                                        .append(" :{")
                                        .append(operation.name()).append(":").append(valueString(operation, value))
                                        .append(" },");
                            });
                        });

                        filterBuilder.append("}");
                    });
                    filterBuilder.append("}");
                });
            }
        });

        return filterBuilder.toString();
    }

    public static String toQueryString(IEntityClass entityClass, EntityClassEngine engine, String profile, ConditionQueryRequest conditionQueryRequest, boolean isStrict) {

        EntityClassGroup group = engine.describe(entityClass, profile);

        //TODO
//        if(!isStrict) {
//            ConditionQueryRequestHelper.removeUselessField(engine, group, profile, conditionQueryRequest);
//        }

        StringBuilder stringBuilder = new StringBuilder();
        String code = entityClass.code();

        /**
         * to value
         */
        StringBuilder subBuilder = new StringBuilder();
        subBuilder.append("row {");
        EntityItem entity = conditionQueryRequest.getEntity();
        List<String> fields = Optional
                .ofNullable(entity)
                .map(EntityItem::getFields)
                .map(x -> {
                    return x.stream().map(y -> normalize(y)).collect(Collectors.toList());
                })
                .orElseGet(Collections::emptyList);
        if (fields.isEmpty()) {

            fields = group.getAllFields().stream().map(x -> normalize(x.name())).collect(Collectors.toList());
        }
        fields.forEach(x -> {
            subBuilder.append(expandSubField(x)).append(" ");
        });

        Optional.ofNullable(entity).map(EntityItem::getEntities).orElseGet(Collections::emptyList).stream().forEach(sub -> {
            subBuilder.append(sub.getCode()).append(" ");
            List<String> subFields = sub.getFields();
            if (subFields != null && !subFields.isEmpty()) {
                subBuilder.append(" {");
                subFields.forEach(s -> {
                    subBuilder.append(expandSubField(normalize(s))).append(" ");
                });
                subBuilder.append(" }");
            }
        });

        subBuilder.append(" }");
        subBuilder.append(" totalCount");

        /**
         * filter: aaFilter, order: Orderable, pageNo: Int, pageSize: Int
         */
        Conditions conditions = conditionQueryRequest.getConditions();

        StringBuilder filterBuilder = new StringBuilder(OqsTypesFactory.FILTER_ARG + ":{");
        filterBuilder.append(toCondition(conditions));
        filterBuilder.append("}");

        StringBuilder orderBuilder = new StringBuilder();

        List<FieldSort> sort = conditionQueryRequest.getSort();
//        Map<String, List<FieldSort>> collect = Optional.ofNullable(sort).orElseGet(Collections::emptyList).stream()
//                .collect(Collectors.groupingBy(x -> x.getOrder(), Collectors.toList()));
//        String asc = Optional.ofNullable(collect.get("asc")).orElseGet(Collections::emptyList).stream().map(x -> "\"".concat(x.getField()).concat("\"")).collect(Collectors.joining(","));
//        String desc = Optional.ofNullable(collect.get("desc")).orElseGet(Collections::emptyList).stream().map(x -> "\"".concat(x.getField()).concat("\"")).collect(Collectors.joining(","));
        String orderListStr = Optional.ofNullable(sort).orElseGet(Collections::emptyList).stream().map(x -> {
            return String.format(ORDER_TEMPLATE, x.getField(), "desc".equals(x.getOrder()) ? "DESC" : "ASC");
        }).collect(Collectors.joining(","));


        orderBuilder.append(OqsTypesFactory.ORDER_ARG);
        orderBuilder.append(": [ ");
        orderBuilder.append(orderListStr);
        orderBuilder.append("]");

        StringBuilder rangeBuilder = new StringBuilder();
        Integer pageNo = conditionQueryRequest.getPageNo();
        Integer pageSize = conditionQueryRequest.getPageSize();
        rangeBuilder.append("pageNo:").append(pageNo).append(",");
        rangeBuilder.append("pageSize:").append(pageSize);

        stringBuilder.append("query {")
                .append(code).append("(")
                .append(filterBuilder).append(",")
                .append(orderBuilder).append(",")
                .append(rangeBuilder)
                .append(")")
                .append(" {");

        stringBuilder.append(subBuilder);

        stringBuilder.append(" }}");
        return stringBuilder.toString();
    }


    /**
     * @param op
     * @param value
     * @return
     */
    private static String valueString(ConditionOp op, List<String> value) {
        StringBuilder valueBuilder = new StringBuilder();
        if (op.getSize() > 2) {
            valueBuilder.append("[");
            valueBuilder.append(value.stream()
                    .map(v -> "\"".concat(escapeValue(v)).concat("\""))
                    .collect(Collectors.joining(",")));
            valueBuilder.append("]");
            return valueBuilder.toString();
        } else {
            return value.isEmpty() ? null : "\"".concat(escapeValue(value.get(0))).concat("\"");
        }
    }

    private static String escapeValue(String value) {
        if (value != null) {
            return value.replaceAll("\"", "\\\\\"");
        }

        return value;
    }
}
