package com.xforceplus.ultraman.sdk.graphql.gen;

import com.xforceplus.tech.base.core.context.ContextKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.metadata.domain.record.EmptyValue;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.sdk.core.facade.MutationProvider;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import com.xforceplus.ultraman.sdk.core.facade.QueryProvider;
import com.xforceplus.ultraman.sdk.core.facade.option.QueryOption;
import com.xforceplus.ultraman.sdk.core.rel.legacy.*;
import graphql.GraphQLContext;
import graphql.language.Directive;
import graphql.language.OperationDefinition;
import graphql.schema.DataFetcher;
import graphql.schema.DataFetchingEnvironment;
import graphql.schema.GraphQLAppliedDirective;
import graphql.schema.GraphQLAppliedDirectiveArgument;
import io.vavr.Tuple;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.dataloader.DataLoader;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.sdk.graphql.gen.OqsTypesFactory.CODE_ARG;
import static graphql.language.OperationDefinition.Operation.MUTATION;
import static graphql.language.OperationDefinition.Operation.QUERY;


/**
 * data fetcher framework
 */
@Slf4j
public class OqsDataFetcher implements DataFetcher {

    @Autowired
    private List<MutationProvider> mutationProviders;

    @Autowired
    private List<QueryProvider> queryProviders;

    @Autowired
    private ProfileFetcher fetcher;

    @Autowired
    private EntityClassEngine engine;

    @Autowired
    private ContextService contextService;

    public List<MutationProvider> getMutationProviders() {
        return mutationProviders;
    }

    public void setMutationProviders(List<MutationProvider> mutationProviders) {
        this.mutationProviders = mutationProviders;
    }

    public List<QueryProvider> getQueryProviders() {
        return queryProviders;
    }

    public void setQueryProviders(List<QueryProvider> queryProviders) {
        this.queryProviders = queryProviders;
    }

    public ProfileFetcher getFetcher() {
        return fetcher;
    }

    public void setFetcher(ProfileFetcher fetcher) {
        this.fetcher = fetcher;
    }

    public EntityClassEngine getEngine() {
        return engine;
    }

    public void setEngine(EntityClassEngine engine) {
        this.engine = engine;
    }

    public ContextService getContextService() {
        return contextService;
    }

    public void setContextService(ContextService contextService) {
        this.contextService = contextService;
    }

    @Override
    public Object get(DataFetchingEnvironment environment) throws Exception {

        String profile = fetcher.getProfile(contextService.getAll());

        try {
            List<GraphQLAppliedDirective> directives = environment.getFieldDefinition().getAppliedDirectives();
            if (!directives.isEmpty()) {
                Optional<GraphQLAppliedDirective> coded = directives.stream()
                        .filter(x -> x.getName().equals(OqsTypesFactory.CODED_APPLIED.getName()))
                        .findAny();

                String entityCode = null;
                if (coded.isPresent()) {
                    GraphQLAppliedDirectiveArgument argument = coded.get().getArgument(CODE_ARG);
                    if (argument != null) {
                        entityCode = argument.getValue().toString();
                        Optional<IEntityClass> entityClassOp = engine.loadByCode(entityCode, profile);

                        if (entityClassOp.isPresent()) {
                            OperationDefinition operationDefinition = environment.getOperationDefinition();
                            if (operationDefinition.getOperation() == MUTATION) {
                                //do mutation
                                return mutation(environment, entityClassOp.get());
                            } else if (operationDefinition.getOperation() == QUERY) {
                                //do query
                                Object remoteQuery = query(environment, entityClassOp.get());
                                GraphQLAppliedDirective one2one = environment.getFieldDefinition().getAppliedDirective("one2one");
                                if (one2one != null) {

                                    //batch one or plain one
                                    if (remoteQuery instanceof CompletableFuture) {
                                        return ((CompletableFuture<?>) remoteQuery).thenApply(x -> {
                                            if (x instanceof List && !((List) x).isEmpty()) {
                                                return ((List) x).get(0);
                                            }
                                            return x;
                                        });
                                    } else if (remoteQuery instanceof List && !((List<?>) remoteQuery).isEmpty()) {
                                        return ((List) remoteQuery).get(0);
                                    } else {
                                        return null;
                                    }
                                }
                                return remoteQuery;
                            } else {
                                throw new RuntimeException("Operation is not supported");
                            }
                        } else {
                            throw new RuntimeException("Related entityClass not found");
                        }
                    }
                }

            }
        } catch (Exception ex) {
            log.error("inner ex {}", ex);
            throw new RuntimeException(ex);
        } finally {
        }
        return null;
    }

    /**
     * query can changed via the input selection
     *
     * @param environment
     * @param entityClass
     * @return
     */
    private Object query(DataFetchingEnvironment environment, IEntityClass entityClass) {
        Map<String, Object> condition = environment.getArgument(OqsTypesFactory.FILTER_ARG);
        List<Map<String, Object>> sort = environment.getArgument(OqsTypesFactory.ORDER_ARG);
        Integer pageSize = environment.getArgumentOrDefault("pageSize", 20);
        Integer pageNo = environment.getArgumentOrDefault("pageNo", 1);

        ExpCondition fromGraphQLMap = ExpCondition.alwaysTrue();
        if (condition != null) {
            fromGraphQLMap = ExpFactory.createFromGraphQLMap(condition);
        }

        ExpSort expSort = new ExpSort();
        if (sort != null && !sort.isEmpty()) {
//            List<String> asc = Optional.ofNullable(sort.get("asc")).map(x -> (List<String>) x).orElseGet(Collections::emptyList);
//            List<String> desc = Optional.ofNullable(sort.get("desc")).map(x -> (List<String>) x).orElseGet(Collections::emptyList);
//            if (!asc.isEmpty()) {
//                for (String ascStr : asc) {
//                    expSort.withSort(ascStr, "asc");
//                }
//            }
//
//            if (!desc.isEmpty()) {
//                for (String descStr : desc) {
//                    expSort.withSort(descStr, "desc");
//                }
//            }
            sort.forEach(entry -> {
                Object code = entry.get(CODE_ARG);
                /**
                 * TODO
                 */
                Object order = entry.get(OqsTypesFactory.ORDER_ARG);
                if (code != null && order != null) {
                    expSort.withSort(code.toString(), order.toString());
                }
            });
        }

        ExpCondition finalCondition = fromGraphQLMap;
        Map<String, Object> source = environment.getSource();
        if (source != null) {
            //it's a related query build a related query
            boolean one2one = environment.getFieldDefinition().getAppliedDirectives().stream().anyMatch(x -> x.getName().equals("one2one"));
            boolean one2Many = environment.getFieldDefinition().getAppliedDirectives().stream().anyMatch(x -> x.getName().equals("one2many"));
            String relatedCode = environment.getFieldDefinition().getName();
            if (one2one) {
                Object id = source.get("_".concat(relatedCode.concat("_id")));
                if (id != null) {
                    /**
                     * always has a dotaloader
                     */
                    DataLoader<Object, Object> dataLoader = environment.getDataLoader(entityClass.code());
                    if (dataLoader != null) {
                        return dataLoader.load(Tuple.of(entityClass, id.toString()), environment);
                    }
                } else {
                    return null;
                }
            } else if (one2Many) {
                Object id = source.get("id");

                Optional<GraphQLAppliedDirective> one2many = environment.getFieldDefinition()
                        .getAppliedDirectives().stream().filter(x -> x.getName().equals("one2many")).findAny();
                //we checked outside
                GraphQLAppliedDirective one2may = one2many.get();
                Object value = one2may.getArgument(CODE_ARG).getValue();
                if (value != null) {
                    relatedCode = value.toString();
                }

                if (id != null) {
                    ExpCondition relatedId = ExpCondition.call(ExpOperator.EQUALS, ExpField.field(relatedCode.concat(".id")), ExpValue.from(id));
                    finalCondition = ExpCondition.call(ExpOperator.AND, fromGraphQLMap, relatedId);
                } else {
                    return null;
                }
            }
        }

        //root is entityClass
        ExpQuery query = new ExpQuery().filters(finalCondition).sort(expSort).range(pageNo, pageSize);

        return doQuery(entityClass, query, environment);
    }

    /**
     * query
     *
     * @return
     */
    private Object doQuery(IEntityClass entityClass, ExpRel expRel, DataFetchingEnvironment environment) {

        Optional<QueryProvider> queryProvider = queryProviders
                .stream().filter(x -> x.accept(entityClass)).findFirst();

        if (queryProvider.isPresent()) {
            /**
             * current is UNBOX
             */
            boolean unbox = environment.getFieldDefinition().getAppliedDirectives().stream().anyMatch(x -> OqsTypesFactory.UNBOX_APPLIED.getName().equals(x.getName()));
            Object queryResult = queryProvider.get().query(entityClass, expRel, unbox ? QueryOption.UNBOX : QueryOption.NONE);
            return queryResult;
        } else {
            throw new RuntimeException();
        }
    }

    private String mutation(DataFetchingEnvironment environment, IEntityClass entityClass) {
        List<GraphQLAppliedDirective> directives = environment.getFieldDefinition().getAppliedDirectives();
        if (!directives.isEmpty()) {
            if (directives.stream().anyMatch(x -> x.getName().equals(OqsTypesFactory.ADD.getName()))) {
                //create
                return doCreate(environment, entityClass);
            } else if (directives.stream().anyMatch(x -> x.getName().equals(OqsTypesFactory.UPDATE.getName()))) {
                return doUpdate(environment, entityClass);
            } else if (directives.stream().anyMatch(x -> x.getName().equals(OqsTypesFactory.DELETE.getName()))) {
                return doDelete(environment, entityClass);
            } else if (directives.stream().anyMatch(x -> x.getName().equals(OqsTypesFactory.BATCH_ADD.getName()))) {
                //create
                return doBatchCreate(environment, entityClass);
            } else if (directives.stream().anyMatch(x -> x.getName().equals(OqsTypesFactory.BATCH_UPDATE.getName()))) {
                //create
                return doBatchUpdate(environment, entityClass);
            } else if (directives.stream().anyMatch(x -> x.getName().equals(OqsTypesFactory.BATCH_DELETE.getName()))) {
                //create
                return doBatchDelete(environment, entityClass);
            } else if (directives.stream().anyMatch(x -> x.getName().equals(OqsTypesFactory.CUSTOM_APPLIED.getName()))) {
                return doCustom(environment, entityClass);
            } else {
                throw new RuntimeException("Not supported operation");
            }
        }
        return "操作失败";
    }


    /**
     * transform argument with underline'_' to dot‘.’
     *
     * @param argument
     * @return
     */
    private Map<String, Object> recreate(Map<String, Object> argument) {
        Map<String, Object> map = new HashMap<>();
        argument.forEach((k, v) -> {
            String key = k;
            if (k.startsWith("_")) {
                key = k.substring(1).replace('_', '.');
            }

            if (v == null) {
                v = EmptyValue.emptyValue;
            }
            map.put(key, v);
        });

        return map;
    }

    public String getTransactionId(DataFetchingEnvironment environment) {
        String transactionId = null;
        List<Directive> mutationDirectives = environment.getOperationDefinition().getDirectives();
        if (!mutationDirectives.isEmpty()) {
            if (mutationDirectives.stream().anyMatch(x -> x.getName().equalsIgnoreCase(OqsTypesFactory.TRANSACTIONAL_D_APPLIED.getName()))) {
                //transactional operation
                GraphQLContext context = environment.getGraphQlContext();
                if(context != null) {
                    Object o = context.get(OqsTypesFactory.TRANSACTIONAL);
                    if (o != null) {
                        transactionId = o.toString();
                    }
                }
            }
        }
        return transactionId;
    }

    private String doCreate(DataFetchingEnvironment env, IEntityClass entityClass) {
        Map<String, Object> argument = env.getArgument(OqsTypesFactory.INPUT_ARG);

        Optional<MutationProvider> mutationProvider = mutationProviders
                .stream().filter(x -> x.accept(entityClass)).findFirst();
        if (mutationProvider.isPresent()) {
            Map<String, Object> context = getContext(env);
            String transactionId = getTransactionId(env);
            if (!StringUtils.isEmpty(transactionId)) {
                context.put(ContextKeys.StringKeys.TRANSACTION_KEY.name(), transactionId);
            }
            return mutationProvider.get().create(entityClass, recreate(Optional.ofNullable(argument).orElseGet(Collections::emptyMap)), context);
        } else {
            throw new RuntimeException("Create Mutation is not Support");
        }
    }

    private String doBatchCreate(DataFetchingEnvironment env, IEntityClass entityClass) {
        List<Map<String, Object>> argument = env.getArgument(OqsTypesFactory.INPUT_ARG);

        Optional<MutationProvider> mutationProvider = mutationProviders
                .stream().filter(x -> x.accept(entityClass)).findFirst();

        if (mutationProvider.isPresent()) {
            Map<String, Object> context = getContext(env);
            String transactionId = getTransactionId(env);
            if (!StringUtils.isEmpty(transactionId)) {
                context.put(ContextKeys.StringKeys.TRANSACTION_KEY.name(), transactionId);
            }
            List<Map<String, Object>> inputBodies = Optional.ofNullable(argument).orElseGet(Collections::emptyList).stream().map(x -> recreate(x)).collect(Collectors.toList());
            return mutationProvider.get().batchCreate(entityClass, inputBodies, context);
        } else {
            throw new RuntimeException("Batch Create Mutation is not Support");
        }
    }

    private String doBatchUpdate(DataFetchingEnvironment env, IEntityClass entityClass) {
        List<Map<String, Object>> argument = env.getArgument(OqsTypesFactory.INPUT_ARG);

        Optional<MutationProvider> mutationProvider = mutationProviders
                .stream().filter(x -> x.accept(entityClass)).findFirst();

        if (mutationProvider.isPresent()) {
            Map<String, Object> context = getContext(env);
            String transactionId = getTransactionId(env);
            if (!StringUtils.isEmpty(transactionId)) {
                context.put(ContextKeys.StringKeys.TRANSACTION_KEY.name(), transactionId);
            }
            List<Map<String, Object>> inputBodies = Optional.ofNullable(argument).orElseGet(Collections::emptyList).stream().map(x -> recreate(x)).collect(Collectors.toList());
            return mutationProvider.get().batchUpdate(entityClass, inputBodies, context);
        } else {
            throw new RuntimeException("Batch Update Mutation is not Support");
        }
    }

    private String doUpdate(DataFetchingEnvironment env, IEntityClass entityClass) {
        String id = env.getArgument(OqsTypesFactory.ID_ARG);
        Map<String, Object> argument = env.getArgument(OqsTypesFactory.INPUT_ARG);
        Optional<MutationProvider> mutationProvider = mutationProviders
                .stream().filter(x -> x.accept(entityClass)).findFirst();
        if (mutationProvider.isPresent()) {
            Map<String, Object> context = getContext(env);
            String transactionId = getTransactionId(env);
            if (!StringUtils.isEmpty(transactionId)) {
                context.put(ContextKeys.StringKeys.TRANSACTION_KEY.name(), transactionId);
            }
            return mutationProvider.get().update(entityClass, Long.parseLong(id), recreate(Optional.ofNullable(argument).orElseGet(Collections::emptyMap))
                    , context);
        } else {
            throw new RuntimeException("Update Mutation is not Support");
        }
    }

    private String doDelete(DataFetchingEnvironment env, IEntityClass entityClass) {
        String id = env.getArgument(OqsTypesFactory.ID_ARG);
        Optional<MutationProvider> mutationProvider = mutationProviders
                .stream().filter(x -> x.accept(entityClass)).findFirst();

        if (mutationProvider.isPresent()) {
            Map<String, Object> context = getContext(env);
            String transactionId = getTransactionId(env);
            if (!StringUtils.isEmpty(transactionId)) {
                context.put(ContextKeys.StringKeys.TRANSACTION_KEY.name(), transactionId);
            }
            return mutationProvider.get().delete(entityClass, Long.parseLong(id), context);
        } else {
            throw new RuntimeException("Delete Mutation is not Support");
        }
    }

    private String doBatchDelete(DataFetchingEnvironment env, IEntityClass entityClass) {
        List<String> id = env.getArgument(OqsTypesFactory.ID_ARG);
        Optional<MutationProvider> mutationProvider = mutationProviders
                .stream().filter(x -> x.accept(entityClass)).findFirst();

        if (mutationProvider.isPresent()) {
            Map<String, Object> context = getContext(env);
            String transactionId = getTransactionId(env);
            if (!StringUtils.isEmpty(transactionId)) {
                context.put(ContextKeys.StringKeys.TRANSACTION_KEY.name(), transactionId);
            }
            return mutationProvider.get().batchDelete(entityClass
                    , id.stream().map(Long::parseLong).collect(Collectors.toList())
                    , context);
        } else {
            throw new RuntimeException("Delete Mutation is not Support");
        }
    }

    private String doCustom(DataFetchingEnvironment env, IEntityClass entityClass) {
        Map<String, Object> argument = env.getArgument(OqsTypesFactory.INPUT_ARG);

        Optional<MutationProvider> mutationProvider = mutationProviders
                .stream().filter(x -> x.accept(entityClass)).findFirst();

        List<GraphQLAppliedDirective> directives = env.getFieldDefinition().getAppliedDirectives();
        String actionName = null;
        if (!directives.isEmpty()) {
            Optional<GraphQLAppliedDirective> custom = directives.stream()
                    .filter(x -> x.getName().equals(OqsTypesFactory.CUSTOM_APPLIED.getName()))
                    .findAny();

            if (custom.isPresent()) {
                GraphQLAppliedDirective customDirective = custom.get();
                GraphQLAppliedDirectiveArgument actionCodeArg = customDirective.getArgument(CODE_ARG);
                actionName = actionCodeArg.getValue().toString();
            } else {
                throw new RuntimeException("Mutation custom action is not present");
            }
        }

        if (mutationProvider.isPresent()) {
            Map<String, Object> context = getContext(env);
            String transactionId = getTransactionId(env);
            if (!StringUtils.isEmpty(transactionId)) {
                context.put(ContextKeys.StringKeys.TRANSACTION_KEY.name(), transactionId);
            }
            return mutationProvider.get().custom(entityClass, actionName, argument, context);
        } else {
            throw new RuntimeException("Custom Mutation is not Support");
        }
    }

    private Map<String, Object> getContext(DataFetchingEnvironment env) {

        Map<String, Object> context = new HashMap<>(contextService.getAll());
        GraphQLContext graphQlContext = env.getGraphQlContext();
        if (graphQlContext != null) {
            graphQlContext.stream().forEach(x -> {
                context.put(x.getKey().toString(), x.getValue());
            });
        }
        return context;
    }
}
