package com.xforceplus.ultraman.sdk.graphql.service;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.tech.base.core.dispatcher.anno.QueryHandler;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.FieldLikeRelationType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IRelation;
import com.xforceplus.ultraman.metadata.repository.MetadataRepository;
import com.xforceplus.ultraman.sdk.core.cmd.*;
import com.xforceplus.ultraman.sdk.core.facade.MutationProvider;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import com.xforceplus.ultraman.sdk.core.facade.QueryProvider;
import com.xforceplus.ultraman.sdk.core.facade.remote.RemoteExecutionResponse;
import com.xforceplus.ultraman.sdk.core.service.EntityHandlerService;
//import com.xforceplus.ultraman.sdk.core.transaction.OqsTransactional;
import com.xforceplus.ultraman.sdk.graphql.config.GraphQLSchemaHolder;
import com.xforceplus.ultraman.sdk.graphql.gen.BatchDataLoader;
import graphql.ExecutionInput;
import graphql.ExecutionResult;
import graphql.GraphQLContext;
import graphql.GraphQLError;
import graphql.execution.ExecutionId;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.control.Either;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.sql.parser.Span;
import org.apache.commons.lang3.StringUtils;
import org.dataloader.DataLoader;
import org.dataloader.DataLoaderRegistry;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import java.util.*;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.sdk.graphql.utils.GraphQLHelper.*;

@Slf4j
public class DefaultEntityHandlerService implements EntityHandlerService {

    @Autowired
    private EntityClassEngine engine;

    @Autowired
    private GraphQLSchemaHolder holder;

    @Autowired
    private List<QueryProvider> queryProviders;

    @Autowired
    private ObjectMapper mapper;

    @Autowired
    private ProfileFetcher fetcher;

    /**
     * TODO
     * can only for inner
     */
    @Autowired
    private MutationProvider mutationProvider;

    @Autowired
    private ContextService contextService;

    @Value("${xplat.oqsengine.sdk.query.strict:true}")
    private boolean isStrict;

    public boolean isStrict() {
        return isStrict;
    }

    public void setStrict(boolean strict) {
        isStrict = strict;
    }

    private static final String MISSING_ENTITIES = "查询对象不存在";

    private static final String MISSING_ENTITY = "查询记录不存在";

    public GraphQLSchemaHolder getHolder() {
        return holder;
    }

    public void setHolder(GraphQLSchemaHolder holder) {
        this.holder = holder;
    }

    public EntityClassEngine getEngine() {
        return engine;
    }

    public void setEngine(EntityClassEngine engine) {
        this.engine = engine;
    }

    public List<QueryProvider> getQueryProviders() {
        return queryProviders;
    }

    public void setQueryProviders(List<QueryProvider> queryProviders) {
        this.queryProviders = queryProviders;
    }

    public ObjectMapper getMapper() {
        return mapper;
    }

    public void setMapper(ObjectMapper mapper) {
        this.mapper = mapper;
    }

    public ProfileFetcher getFetcher() {
        return fetcher;
    }

    public void setFetcher(ProfileFetcher fetcher) {
        this.fetcher = fetcher;
    }

    public ContextService getContextService() {
        return contextService;
    }

    public void setContextService(ContextService contextService) {
        this.contextService = contextService;
    }

    /**
     * get related entityClass
     *
     * @param cmd
     * @return
     */
    private Optional<IEntityClass> getEntityClass(MetaDataLikeCmd cmd) {
        return Optional.ofNullable(cmd.version()).map(x -> {
            return engine.load(cmd.getBoId()
                    , fetcher.getProfile(Collections.emptyMap())
                    , cmd.version());
        }).orElseGet(() -> engine.load(cmd.getBoId()
                , fetcher.getProfile(Collections.emptyMap())));
    }

    /**
     * single query cmd to map
     *
     * @param cmd
     * @return
     */
    private Map<String, Object> queryCmdToMap(SingleQueryCmd cmd) {
        Map<String, Object> queryMap = new HashMap<>();
        queryMap.put("_objId", cmd.getId());
        queryMap.put("_classId", cmd.getBoId());
        queryMap.put("_version", cmd.version());
        return queryMap;
    }

    /**
     * condition query to map
     *
     * @param cmd
     * @return
     */
    private Map<String, Object> conditionQueryCmdToMap(ConditionSearchCmd cmd) {
        Map<String, Object> queryMap = new HashMap<>();
        queryMap.put("_condition", cmd.getConditionQueryRequest());
        queryMap.put("_version", cmd.version());
        queryMap.put("_classId", cmd.getBoId());
        return queryMap;
    }

    /**
     * delete cmd to map
     *
     * @param cmd
     * @return
     */
    private Map<String, Object> deleteCmdToMap(SingleDeleteCmd cmd) {
        Map<String, Object> queryMap = new HashMap<>();
        queryMap.put("_objId", cmd.getId());
        queryMap.put("_classId", cmd.getBoId());
        queryMap.put("_version", cmd.version());
        return queryMap;
    }

    /**
     * multi delete cmd to map
     *
     * @param cmd
     * @return
     */
    private Map<String, Object> multiDeleteCmdToMap(BatchDeleteCmd cmd) {
        Map<String, Object> queryMap = new HashMap<>();
        queryMap.put("_objIds", cmd.getIds());
        queryMap.put("_classId", cmd.getBoId());
        queryMap.put("_version", cmd.version());
        return queryMap;
    }

    /**
     * update cmd to map
     *
     * @param cmd
     * @return
     */
    private Map<String, Object> updateCmdToMap(SingleUpdateCmd cmd) {
        Map<String, Object> bodyMap = new HashMap<>();
        bodyMap.putAll(cmd.getBody());
        bodyMap.put("_objId", cmd.getId());
        bodyMap.put("_version", cmd.version());
        bodyMap.put("_classId", cmd.getBoId());
        return bodyMap;
    }

    /**
     * batch update cmd to map
     *
     * @param cmd
     * @return
     */
    private Map<String, Object> batchUpdateCmdToMap(BatchUpdateCmd cmd) {
        Map<String, Object> bodyMap = new HashMap<>();
        bodyMap.put("_bodies", cmd.getBodies());
        bodyMap.put("_version", cmd.version());
        bodyMap.put("_classId", cmd.getBoId());
        return bodyMap;
    }

    /**
     * create cmd to map
     *
     * @param cmd
     * @return
     */
    private Map<String, Object> createCmdToMap(SingleCreateCmd cmd) {
        Map<String, Object> bodyMap = new HashMap<>();
        bodyMap.putAll(cmd.getBody());
        bodyMap.put("_version", cmd.version());
        bodyMap.put("_classId", cmd.getBoId());
        return bodyMap;
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Map<String, Object>> singleQuery(SingleQueryCmd cmd) {

        String profile = fetcher.getProfile(Collections.emptyMap());
        try {
            /**
             * get related entityClass
             */
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, engine, profile, cmd, null, isStrict);

            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);
            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                Map<String, Object> data = execute.getData();
                Object o = data.get(entityClass.code());
                if (o instanceof Map) {
                    Object row = ((Map<?, ?>) o).get("row");
                    List<Object> rawList = Optional.ofNullable(row).map(x -> (List) x).orElseGet(Collections::emptyList);
                    List<Map<String, Object>> flattened = rawList.stream().map(x -> {
                        if (x instanceof Map) {
                            Map<String, Object> retMap = new HashMap<>();
                            flattenMap((Map<String, Object>) x, retMap, "");
                            return retMap;
                        } else {
                            return null;
                        }
                    }).filter(Objects::nonNull).collect(Collectors.toList());

                    if (!flattened.isEmpty()) {
                        return Either.right(reNormalize(flattened.get(0)));
                    } else {
                        return Either.left(MISSING_ENTITY);
                    }
                } else {
                    throw new RuntimeException("Type error");
                }
            }
        } catch (Exception ex) {
            log.error("ConditionSearch Error {}", ex);
            return Either.left(ex.getMessage());
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Integer> batchDelete(BatchDeleteCmd cmd) {
        String profile = fetcher.getProfile(Collections.emptyMap());

        try {
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, cmd, "");
            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);

            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                Object data = execute.getData();
                if (data instanceof Map) {
                    Map<String, Object> dataMap = (Map<String, Object>) data;
                    Integer affectRow = Optional.ofNullable(dataMap.get(entityClass.code().concat("BatchDelete"))).map((x -> Integer.parseInt(x.toString()))).orElse(-1);
                    return Either.right(affectRow);
                }
                return Either.right(-1);
            }
        } catch (Exception ex) {
            log.error("Delete error {}", ex);
            return Either.left(ex.getMessage());
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Integer> singleDelete(SingleDeleteCmd cmd) {
        String profile = fetcher.getProfile(Collections.emptyMap());
        try {
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, cmd, "");

            Object context = executionInput.getContext();
            if (context instanceof GraphQLContext) {
                String relations = cmd.getRelations();
                if (!StringUtils.isEmpty(relations)) {
                    log.info("delete with details {} {}", entityClass.code(), relations);
                    ((GraphQLContext) context).put("relations", cmd.getRelations());
                }
            }

            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);


            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                return Either.right(1);
            }
        } catch (Exception ex) {
            log.error("Delete error {}", ex);
            return Either.left(ex.getMessage());
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Integer> batchCreate(BatchCreateCmd cmd) {
        String profile = fetcher.getProfile(Collections.emptyMap());

        try {
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, cmd, "");

            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);
            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                Object data = execute.getData();
                if (data instanceof Map) {
                    Map<String, Object> dataMap = (Map<String, Object>) data;
                    Integer affectRow = Optional.ofNullable(dataMap.get(entityClass.code().concat("BatchAdd"))).map((x -> Integer.parseInt(x.toString()))).orElse(-1);
                    return Either.right(affectRow);
                }
                return Either.right(-1);
            }
        } catch (Exception ex) {
            log.error("Batch Create error {}", ex);
            return Either.left(ex.getMessage());
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Long> singleCreate(SingleCreateCmd cmd) {
        String profile = fetcher.getProfile(Collections.emptyMap());
        try {
            /**
             * get related entityClass
             */
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, cmd, "");
            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);
            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                Object data = execute.getData();
                if (data instanceof Map) {
                    Map<String, Object> dataMap = (Map<String, Object>) data;
                    Long id = Optional.ofNullable(dataMap.get(entityClass.code().concat("Add"))).map((x -> Long.parseLong(x.toString()))).orElse(-1L);
                    return Either.right(id);
                }
                return Either.right(-1L);
            }
        } catch (Exception ex) {
            return Either.left(ex.getMessage());
        }
    }

    private Map<String, Object> getResult(RemoteExecutionResponse response) {
        Object result = response.getResult();
        Map<String, Object> resultMapping = new HashMap<>();
        if (result != null) {
            if (result instanceof Tuple2) {
                Tuple2<Integer, List<Map>> tuple = ((Tuple2<Integer, List<Map>>) result);
                List<Map> maps = tuple._2();
                if (!maps.isEmpty()) {
                    resultMapping = maps.get(0);
                }
            } else {
                resultMapping = (Map<String, Object>) result;
            }

            return resultMapping;
        } else {
            return Collections.emptyMap();
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Integer> batchUpdate(BatchUpdateCmd cmd) {
        String profile = fetcher.getProfile(Collections.emptyMap());

        try {
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, cmd, "");
            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);

            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                Object data = execute.getData();
                if (data instanceof Map) {
                    Map<String, Object> dataMap = (Map<String, Object>) data;
                    Integer affectRow = Optional.ofNullable(dataMap.get(entityClass.code().concat("BatchUpdate"))).map((x -> Integer.parseInt(x.toString()))).orElse(-1);
                    return Either.right(affectRow);
                }
                return Either.right(-1);
            }
        } catch (Exception ex) {
            log.error("Batch Update error {}", ex);
            return Either.left(ex.getMessage());
        }
    }

    /**
     * current for inner cause the transaction
     *
     * @param cmd
     * @return
     */
    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Integer> headAndDetailsCreate(HeadAndDetailsCreateCmd cmd) {

        //profile related
        String profile = fetcher.getProfile(Collections.emptyMap());

        try {
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            EntityClassGroup group = engine.describe(entityClass, profile);

            ExecutionInput mainCreateInput = toInput(entityClass, new SingleCreateCmd(cmd.getBoId(), cmd.getMainBody(), cmd.version()), "");
            ExecutionResult execute = holder.getGraphQL(profile).execute(mainCreateInput);

            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                Object data = execute.getData();
                if (data instanceof Map) {
                    Map<String, Object> dataMap = (Map<String, Object>) data;
                    Long id = Optional.ofNullable(dataMap.get(entityClass.code().concat("Add"))).map((x -> Long.parseLong(x.toString()))).orElse(-1L);
                    //get id
                    Collection<IRelation> relations = Optional.ofNullable(entityClass.relations()).orElseGet(Collections::emptyList);
                    //filter the one 2 many
                    cmd.getRelatedBodies().stream()
                            .filter(x -> relations.stream().anyMatch(rel ->
                                    rel.getName().equals(x._1)
                                            && FieldLikeRelationType.ONE2MANY.getName().equalsIgnoreCase(rel.getRelationType())))
                            .forEach(tuple -> {
                                Optional<IEntityClass> relatedEntityClass = group.relatedEntityClass(tuple._1());
                                List<Map<String, Object>> relatedDataList = tuple._2;
                                relatedDataList.forEach(relatedData -> {
                                    relatedData.put(tuple._1.concat(".id"), id);
                                });

                                ExecutionInput subCreateInput = toInput(relatedEntityClass.get()
                                        , new BatchCreateCmd(Long.toString(relatedEntityClass.get().id()), relatedDataList, cmd.version()), "");
                                ExecutionResult subExecutionResult = holder.getGraphQL(profile).execute(subCreateInput);
                                List<GraphQLError> subErrors = subExecutionResult.getErrors();
                                if (!subErrors.isEmpty()) {
                                    throw new RuntimeException(subErrors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
                                }
                            });
                    return Either.right(1);
                }

                throw new RuntimeException("Record Id is Missing");
            }
        } catch (Exception ex) {
            log.error("Head Detail Update error {}", ex);
            throw ex;
        }
    }

    /**
     * custom action TODO how to do this?
     *
     * @param cmd
     * @return
     */
    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Object> customAction(CustomActionCmd cmd) {
        String profile = fetcher.getProfile(Collections.emptyMap());

        try {
            /**
             * get related entityclass
             */
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, cmd, "");
            log.debug("[SDK]:{}", executionInput);

            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);
            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                return Either.right(execute.getData());
            }
        } catch (Exception ex) {
            return Either.left(ex.getMessage());
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Integer> singleUpdate(SingleUpdateCmd cmd) {
        String profile = fetcher.getProfile(Collections.emptyMap());

        try {
            /**
             * get related entityClass
             */
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, cmd, "null");
            log.debug("[SDK]:{}", executionInput);
            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);
            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                return Either.right(1);
            }
        } catch (Exception ex) {
            return Either.left(ex.getMessage());
        }
    }

    private ExecutionInput toInput(String queryString, DataLoaderRegistry registry, String traceId) {
        ExecutionInput executionInput = ExecutionInput.newExecutionInput()
                .query(queryString)
                //TODO
                .dataLoaderRegistry(registry)
                .variables(contextService.getAll())
                .executionId(ExecutionId.from(traceId))
                .build();

        GraphQLContext context = (GraphQLContext) executionInput.getContext();
        contextService.getAll().forEach((k, v) -> {
            if(v == null) {
                log.warn("{} has a null value", k);
            } else {
                context.put(k, v);
            }
        });


        log.debug("[QueryString] {}", queryString);
        return executionInput;
    }

    private ExecutionInput toInput(String queryString, String traceId) {
        ExecutionInput executionInput = ExecutionInput.newExecutionInput()
                .query(queryString)
                //TODO
                .variables(contextService.getAll())
                .executionId(ExecutionId.from(traceId))
                .build();

        GraphQLContext context = (GraphQLContext) executionInput.getContext();
        contextService.getAll().forEach((k, v) -> {
            if(v == null) {
                log.warn("{} has a null value", k);
            } else {
                context.put(k, v);
            }
        });


        log.debug("[QueryString] {}", queryString);
        return executionInput;
    }

    private ExecutionInput toInput(IEntityClass entityClass, CustomActionCmd cmd, String traceId) {
        String queryString = customString(entityClass, cmd);
        return toInput(queryString, traceId);
    }

    private ExecutionInput toInput(IEntityClass entityClass, BatchCreateCmd cmd, String traceId) {
        String queryString = batchCreateString(entityClass, cmd);
        return toInput(queryString, traceId);
    }

    private ExecutionInput toInput(IEntityClass entityClass, BatchUpdateCmd cmd, String traceId) {
        String queryString = batchUpdateString(entityClass, cmd);
        return toInput(queryString, traceId);
    }

    private ExecutionInput toInput(IEntityClass entityClass, BatchDeleteCmd cmd, String traceId) {
        String queryString = batchDeleteString(entityClass, cmd);
        return toInput(queryString, traceId);
    }

    private ExecutionInput toInput(IEntityClass entityClass, SingleCreateCmd cmd, String traceId) {
        String queryString = singleCreateString(entityClass, cmd);
        return toInput(queryString, traceId);
    }

    private ExecutionInput toInput(IEntityClass entityClass, SingleUpdateCmd cmd, String traceId) {
        String queryString = singleUpdateString(entityClass, cmd);
        return toInput(queryString, traceId);
    }

    private ExecutionInput toInput(IEntityClass entityClass, SingleDeleteCmd cmd, String traceId) {
        String queryString = singleDeleteString(entityClass, cmd);
        return toInput(queryString, traceId);
    }

    private ExecutionInput toInput(IEntityClass entityClass, EntityClassEngine engine, String profile, SingleQueryCmd cmd, String traceId, boolean isStrict) {
        String queryString = singleQueryString(entityClass, engine, profile, cmd, isStrict);
        return toInput(queryString, traceId);
    }

    private ExecutionInput toInput(IEntityClass entityClass, EntityClassEngine engine, String profile, ConditionSearchCmd cmd, String traceId, boolean isStrict) {
        DataLoaderRegistry registry = new DataLoaderRegistry();
        //prepare
        engine.codes().forEach(code -> {
            registry.computeIfAbsent(code, lazy -> DataLoader.newDataLoader(new BatchDataLoader(queryProviders)));
        });

        String queryString = conditionToQueryString(entityClass, engine, profile, cmd, isStrict);
        return toInput(queryString, registry, traceId);
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Tuple2<Integer, List<Map<String, Object>>>> conditionSearch(ConditionSearchCmd cmd) {
        String profile = fetcher.getProfile(Collections.emptyMap());

        try {
            /**
             * get related entityClass
             */
            Optional<IEntityClass> entityClassOp = getEntityClass(cmd);
            IEntityClass entityClass;
            if (entityClassOp.isPresent()) {
                entityClass = entityClassOp.get();
            } else {
                return Either.left(MISSING_ENTITIES);
            }

            ExecutionInput executionInput = toInput(entityClass, engine, profile, cmd, "", isStrict);
            ExecutionResult execute = holder.getGraphQL(profile).execute(executionInput);

            List<GraphQLError> errors = execute.getErrors();
            if (!errors.isEmpty()) {
                throw new RuntimeException(errors.stream().map(x -> "" + x.getMessage()).collect(Collectors.joining(",")));
            } else {
                Map<String, Object> data = execute.getData();
                Object o = data.get(entityClass.code());
                if (o instanceof Map) {
                    Object row = ((Map<?, ?>) o).get("row");
                    Object totalCount = ((Map<?, ?>) o).get("totalCount");

                    List<Object> rawList = Optional.ofNullable(row).map(x -> (List) x).orElseGet(Collections::emptyList);

                    List<Map<String, Object>> flattened = rawList.stream().map(x -> {
                        if (x instanceof Map) {
                            Map<String, Object> retMap = new HashMap<>();
                            flattenMap((Map<String, Object>) x, retMap, "");
                            return reNormalize(retMap);
                        } else {
                            return null;
                        }
                    }).filter(Objects::nonNull).collect(Collectors.toList());

                    return Either.right(Tuple.of(Integer.parseInt(Optional.ofNullable(totalCount)
                                    .orElse("0").toString())
                            , flattened));
                } else {
                    throw new RuntimeException("Type error");
                }
            }
        } catch (Exception ex) {
            log.error("ConditionSearch Error {}", ex);
            return Either.left(ex.getMessage());
        }
    }

    private void flattenMap(Map<String, Object> value, Map<String, Object> dest, String prefix) {
        for (Map.Entry<String, Object> entry : value.entrySet()) {
            if (entry.getValue() instanceof Map) {
                flattenMap((Map<String, Object>) entry.getValue(), dest, StringUtils.isEmpty(prefix) ? entry.getKey() : prefix.concat(".").concat(entry.getKey()));
            } else {
                dest.put(StringUtils.isEmpty(prefix) ? entry.getKey() : prefix.concat(".").concat(entry.getKey()), entry.getValue());
            }
        }
    }
}
