package com.xforceplus.ultraman.sdk.controller;

import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.tech.base.core.dispatcher.ServiceDispatcher;
import com.xforceplus.ultraman.metadata.domain.vo.dto.*;
import com.xforceplus.ultraman.metadata.helper.ConditionQueryRequestHelper;
import com.xforceplus.ultraman.sdk.core.cmd.StatisticsQueryCmd;
import com.xforceplus.ultraman.sdk.controller.constants.SDKContextKey;
import com.xforceplus.ultraman.sdk.core.cmd.*;
import com.xforceplus.ultraman.sdk.core.service.EntityHandlerService;
import io.swagger.annotations.*;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.control.Either;
import org.owasp.esapi.ESAPI;
import org.owasp.esapi.errors.EncodingException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ResolvableType;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;

import static com.xforceplus.ultraman.metadata.domain.record.EmptyValue.emptyValue;
import static com.xforceplus.ultraman.sdk.controller.constants.Constants.FAILED;

/**
 * @author admin
 */
@RequestMapping
@Api(value = "/", tags = "基本数据操作")
public class EntityController {

    //TODO
    @Autowired
    private ServiceDispatcher dispatcher;

    @Autowired
    private ContextService contextService;

    private void preparePageContext(String pageCode) {
        //PAGE is DUMMY for permission
        String page = Optional.ofNullable(pageCode).orElse("PAGE");
        contextService.set(SDKContextKey.PAGE, page);
    }

    @PostMapping("/bos/{boId}/entities/statistics")
    @ResponseBody
    public ResponseEntity<Response<RowItem<Map<String, Object>>>> statisticsQuery (
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "查询条件, 其中field名称为 xx/xx时表示为子类字段" +
                    ", entity 中 fields指定返回当前对象返回字段, entities 指定关联对象返回字段" +
                    ", conditions 中 fields指定当前对象条件字段, entities 指定关联对象条件字段") @RequestBody ConditionQueryRequest condition,
            HttpServletRequest request
    ) {

        preparePageContext(pageCode);

        //default
        if (condition != null) {
            if (condition.getPageNo() == null) {
                condition.setPageNo(1);
            }

            if (condition.getPageSize() == null) {
                condition.setPageSize(10);
            }

            /**
             * make this valid
             */
            ConditionQueryRequestHelper.transformer(condition);
        }

        Either<String, Tuple2<Integer, List<Map<String, Object>>>> result =
                dispatcher.querySync(new StatisticsQueryCmd(boId, condition, version, pageCode)
                        , EntityHandlerService.class, "statistics");

        return extractRepList(Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")));
    }

    @ApiOperation(
            value = "根据元数据对象ID和数据ID查询单条数据"
            , notes = "如果结果存在返回Response<Map>对象，如果没有则抛出异常")
    @ApiResponses(value = {
                    @ApiResponse(code = 200, message = "key-value map, code == 1", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1, 数据不存在或者对象不存在都会报错", response = Response.class)}
    )
    @GetMapping("/bos/{boId}/entities/{id}")
    @ResponseBody
    public ResponseEntity<Response<Map<String, Object>>> singleQuery(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "数据ID") @PathVariable String id,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode
    ) {
        preparePageContext(pageCode);

        Either<String, Map<String, Object>> result = dispatcher.querySync(new SingleQueryCmd(boId, id, version)
                , EntityHandlerService.class, "singleQuery");

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<Map<String, Object>> rep = new Response<>();
            rep.setCode("1");
            rep.setMessage("获取成功");
            rep.setResult(x);
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<Map<String, Object>> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

    @ApiOperation(
            value = "根据元数据对象ID和数据ID删除多条数据"
            , notes = "如果结果存在返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为 1", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1, 数据不存在或者对象不存在都会报错", response = Response.class)}
    )
    @PostMapping("/bos/{boId}/entities/delete")
    @ResponseBody
    public ResponseEntity<Response<String>> multiDelete(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "删除的IDs") @RequestBody(required = true) List<String> ids
    ) {

        preparePageContext(pageCode);

        Either<String, Integer> result = dispatcher.querySync(new BatchDeleteCmd(boId, ids, version)
                , ResolvableType.forClassWithGenerics(Either.class, String.class, Integer.class));

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<String> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(String.valueOf(x));
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<String> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

    @ApiOperation(
            value = "根据元数据对象ID和数据ID删除单条数据"
            , notes = "如果结果存在返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为 1", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1, 数据不存在或者对象不存在都会报错", response = Response.class)}
    )
    @DeleteMapping("/bos/{boId}/entities/{id}")
    @ResponseBody
    public ResponseEntity<Response<String>> singleDelete(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "数据ID") @PathVariable String id,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "对一关系") @RequestParam(required = false, value = "relation") String relations
    ) {

        preparePageContext(pageCode);

        Either<String, Integer> result = dispatcher.querySync(new SingleDeleteCmd(boId, id, version, relations)
                , ResolvableType.forClassWithGenerics(Either.class, String.class, Integer.class));

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<String> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(String.valueOf(x));
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<String> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

    @ApiOperation(
            value = "根据元数据对象ID和key-value map修改数据"
            , notes = "如果创建成功返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为成功创建记录的ID", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1 以及报错信息 ", response = Response.class)}
    )
    @PostMapping("/bos/{boId}/entities/update")
    @ResponseBody
    public ResponseEntity<Response<String>> multiUpdate(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "key-value map") @RequestBody List<Map<String, Object>> bodies
    ) {

        preparePageContext(pageCode);

        Either<String, Integer> result = dispatcher
                .querySync(new BatchUpdateCmd(boId, bodies, version)
                        , EntityHandlerService.class, "batchUpdate");

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<String> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(String.valueOf(x));
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<String> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

    @ApiOperation(
            value = "根据元数据对象ID和key-value map新增数据"
            , notes = "如果创建成功返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为成功创建记录的ID", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1 以及报错信息 ", response = Response.class)}
    )
    @PostMapping("/bos/{boId}/entities/createWithDetails")
    @ResponseBody
    public ResponseEntity<Response<String>> headDetailsCreate(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "key-value map") @RequestBody DataWithHeadAndDetails headAndDetails
    ) {

        preparePageContext(pageCode);

        List<RelatedData> toManyRelations = Optional.ofNullable(headAndDetails.getToManyRelations()).orElseGet(Collections::emptyList);

        Either<String, Integer> result = dispatcher
                .querySync(new HeadAndDetailsCreateCmd(boId, headAndDetails.getData(), toManyRelations.stream().map(x -> Tuple.of(x.getCode(), x.getDataList())).collect(Collectors.toList()), version)
                        , EntityHandlerService.class, "headAndDetailsCreate");

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<String> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(String.valueOf(x));
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<String> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

    @ApiOperation(
            value = "根据元数据对象ID和key-value map新增数据"
            , notes = "如果创建成功返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为成功创建记录的ID", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1 以及报错信息 ", response = Response.class)}
    )
    @PostMapping("/bos/{boId}/entities/create")
    @ResponseBody
    public ResponseEntity<Response<String>> multiCreate(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "key-value map") @RequestBody List<Map<String, Object>> bodies
    ) {

        preparePageContext(pageCode);

        Either<String, Integer> result = dispatcher
                .querySync(new BatchCreateCmd(boId, bodies, version)
                        , EntityHandlerService.class, "batchCreate");

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<String> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(String.valueOf(x));
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<String> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

    @ApiOperation(
            value = "json调用自定义action"
            , notes = "如果创建成功返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为成功创建记录的ID", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1 以及报错信息 ", response = Response.class)}
    )
    @PostMapping("/bos/{boId}/entities/custom/{action}")
    @ResponseBody
    public ResponseEntity<Response<Object>> customAction(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "自定义action") @PathVariable String action,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "key-value map") @RequestBody Map<String, Object> body
    ) {

        preparePageContext(pageCode);

        Either<String, Object> result = dispatcher
                .querySync(new CustomActionCmd(boId, version, body, action)
                        , EntityHandlerService.class, "customAction");

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<Object> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(x);
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<Object> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }


    /**
     * 新增
     * request: {
     * url: '/api/{tenantId}/{appCode}/bos/{boid}/entities',
     * method: 'post'
     * body: {
     * key: value
     * }
     * response: {code:string, message:string}
     * }
     */
    @ApiOperation(
            value = "根据元数据对象ID和key-value map新增数据"
            , notes = "如果创建成功返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为成功创建记录的ID", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1 以及报错信息 ", response = Response.class)}
    )
    @PostMapping("/bos/{boId}/entities")
    @ResponseBody
    public ResponseEntity<Response<String>> singleCreate(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "key-value map") @RequestBody Map<String, Object> body
    ) {

        preparePageContext(pageCode);

        //TODO Tracer
//        Span span = tracer.activeSpan();
//        if (span != null) {
//            span.log("start dispatcher");
//        }

        Either<String, Long> result = dispatcher
                .querySync(new SingleCreateCmd(boId, body, version)
                        , EntityHandlerService.class, "singleCreate");

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<String> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(String.valueOf(x));
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<String> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

//    request: {
//        url: '/api/{tenantId}/{appCode}/bos/{boid}/entities/{id}',
//                method: 'put'
//        body: {
//            key: value
//        }
//    }
//    response: {code:string, message:string}

    @ApiOperation(
            value = "根据元数据对象ID和数据IDS修改数据"
            , notes = "如果结果存在返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为1", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1 以及报错信息 ", response = Response.class)}
    )
    @PutMapping("/bos/{boId}/entities/update")
    @ResponseBody
    public ResponseEntity<Response<String>> singleModify(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "需要修改的key-value map, 如果显示设置为null则置空该属性") @RequestBody List<Map<String, Object>> bodies
    ) {

        preparePageContext(pageCode);

        List<Map<String, Object>> newBodies = bodies.stream().map(body -> {
            Map<String, Object> newBody = body.entrySet().stream().map(k -> {
                if (k.getValue() == null) {
                    return Tuple.of(k.getKey(), emptyValue);
                } else {
                    return Tuple.of(k.getKey(), k.getValue());
                }
            }).collect(Collectors.toMap(x -> x._1(), x -> x._2(), (a, b) -> a));
            return newBody;
        }).collect(Collectors.toList());

        //to identify
        Either<String, Integer> result = dispatcher.querySync(new BatchUpdateCmd(boId, newBodies, version)
                , ResolvableType.forClassWithGenerics(Either.class, String.class, Integer.class));

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<String> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(String.valueOf(x));
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<String> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

    @ApiOperation(
            value = "根据元数据对象ID和数据ID修改数据"
            , notes = "如果结果存在返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为1", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1 以及报错信息 ", response = Response.class)}
    )
    @PutMapping("/bos/{boId}/entities/{id}")
    @ResponseBody
    public ResponseEntity<Response<String>> singleModify(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "数据ID") @PathVariable Long id,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "需要修改的key-value map, 如果显示设置为null则置空该属性") @RequestBody Map<String, Object> body
    ) {

        preparePageContext(pageCode);

        //to identify
        Map<String, Object> newBody = body.entrySet().stream().map(k -> {
            if (k.getValue() == null) {
                return Tuple.of(k.getKey(), emptyValue);
            } else {
                return Tuple.of(k.getKey(), k.getValue());
            }
        }).collect(Collectors.toMap(x -> x._1(), x -> x._2(), (a, b) -> a));

        Either<String, Integer> result = dispatcher.querySync(new SingleUpdateCmd(boId, id, newBody, version)
                , ResolvableType.forClassWithGenerics(Either.class, String.class, Integer.class));

        return Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")).map(x -> {
            Response<String> rep = new Response<>();
            rep.setCode("1");
            rep.setResult(String.valueOf(x));
            rep.setMessage("操作成功");
            return ResponseEntity.ok(rep);
        }).getOrElseGet(str -> {
            Response<String> rep = new Response<>();
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(str));
            rep.setResult(str);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        });
    }

    /**
     * 条件查询
     * request: {
     * url: '/api/{tenantId}/{appCode}/bos/{boid}/entities',
     * method: 'post',
     * body: {
     * pageNo: number,
     * pageSize: number,
     * conditions: {
     * fields: [
     * {
     * code: string,
     * operation: enum{equal, like, in, gteq&lteq, gteq&lt, gt&lteq, gt&lt, gt, gteq, lt, lteq},
     * value: Array
     * }
     * ],
     * entities: [
     * {
     * code: 'otherEntity',
     * fields: [
     * {
     * code: string,
     * operation: enum{equal, like, in, gteq&lteq, gteq&lt, gt&lteq, gt&lt, gt, gteq, lt, lteq},
     * value: Array
     * }
     * ],
     * }
     * ]
     * },
     * sort: [
     * {
     * field: string,
     * order: enum{asc, desc}
     * }
     * ],
     * entity: {
     * fields: ['id', 'name', 'field1', 'field2', 'field3'],
     * entities: [
     * {
     * code: 'otherEntity1',
     * fields: ['name'],
     * },
     * {
     * code: 'otherEntity2',
     * fields: ['name'],
     * },
     * ],
     * },
     * }
     * }
     * response: {
     * code: string,
     * message: string,
     * result: {
     * rows: [
     * {key(${EntityCode.FieldCode}): value}
     * ],
     * summary: {
     * total: 100,
     * },
     * }
     * }
     */

    @ApiOperation(
            value = "根据元数据对象ID和数据ID查询多条数据"
            , notes = "如果结果存在返回Response<String>对象，如果没有则抛出异常")
    @ApiResponses(value = {
            @ApiResponse(code = 200, message = "code == 1 返回值为成功创建记录的ID", response = Response.class),
            @ApiResponse(code = 500, message = "任意错误, code == -1 以及报错信息 ", response = Response.class)}
    )
    @PostMapping("/bos/{boId}/entities/query")
    @ResponseBody
    public ResponseEntity<Response<RowItem<Map<String, Object>>>> conditionQuery(
            @ApiParam(value = "元数据对象ID") @PathVariable String boId,
            @ApiParam(value = "版本, 暂时无特别作用") @RequestParam(required = false, value = "v") String version,
            @ApiParam(value = "页面编码") @RequestParam(required = false, value = "pageCode") String pageCode,
            @ApiParam(value = "忽略count") @RequestParam(required = false, value = "simplePagination") boolean simplePagination,
            @ApiParam(value = "查询条件, 其中field名称为 xx/xx时表示为子类字段" +
                    ", entity 中 fields指定返回当前对象返回字段, entities 指定关联对象返回字段" +
                    ", conditions 中 fields指定当前对象条件字段, entities 指定关联对象条件字段") @RequestBody ConditionQueryRequest condition
            , HttpServletRequest request) {

        preparePageContext(pageCode);
        if(simplePagination) {
            contextService.getAll().put("only_query", true);
        }

        //default
        if (condition != null) {
            if (condition.getPageNo() == null) {
                condition.setPageNo(1);
            }

            if (condition.getPageSize() == null) {
                condition.setPageSize(10);
            }

            /**
             * make this valid
             */
            ConditionQueryRequestHelper.transformer(condition);
        }

        Either<String, Tuple2<Integer, List<Map<String, Object>>>> result =
                dispatcher.querySync(new ConditionSearchCmd(boId, condition, version, pageCode, request.getParameterMap())
                        , EntityHandlerService.class, "conditionSearch");

        return extractRepList(Optional.ofNullable(result).orElseGet(() -> Either.left("没有返回值")));
    }

    private <T> ResponseEntity<Response<RowItem<T>>> extractRepList(Either<String, Tuple2<Integer, List<T>>> result) {
        Response rep = new Response();
        if (result.isRight()) {
            rep.setCode("1");
            Tuple2<Integer, List<T>> tuple = result.get();
            RowItem<T> rowItem = new RowItem<>();
            rowItem.setSummary(new SummaryItem(tuple._1()));
            rowItem.setRows(tuple._2());
            rep.setResult(rowItem);
            rep.setMessage("操作成功");

            Object o = contextService.getAll().get("invocation");
            HttpHeaders httpHeaders = new HttpHeaders();
            if(o != null) {
                httpHeaders.add("invocation", o.toString());
            }

            Object errors = contextService.getAll().get("errors");
            if(errors instanceof List) {
                String errMsg = ((List<?>) errors).stream().map(Object::toString)
                        .distinct().collect(Collectors.joining(","));
                try {
                    httpHeaders.add("err", ESAPI.encoder().encodeForURL(errMsg));
                } catch (EncodingException e) {

                }
            }
            
            return ResponseEntity.ok().headers(httpHeaders).body(rep);
        } else {
            rep.setCode("-1");
            rep.setMessage(FAILED.concat(result.getLeft()));
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(rep);
        }
    }
}
