package com.xforceplus.ultraman.sdk.controller;

import com.google.common.collect.Lists;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.utils.EntityProfileUtils;
import com.xforceplus.ultraman.oqsengine.plus.devops.DevOpsService;
import com.xforceplus.ultraman.oqsengine.plus.devops.dto.DevOpsTaskInfo;
import com.xforceplus.ultraman.oqsengine.plus.history.dto.HistoryTaskInfo;
import com.xforceplus.ultraman.oqsengine.plus.history.handler.HistoryStorage;
import com.xforceplus.ultraman.sdk.controller.devops.Documentations;
import com.xforceplus.ultraman.sdk.controller.devops.RebuildIndexes;
import com.xforceplus.ultraman.sdk.controller.devops.SingleRebuild;
import com.xforceplus.ultraman.sdk.infra.base.id.LongIdGenerator;
import com.xforceplus.ultraman.sdk.infra.base.id.SnowflakeLongIdGenerator;
import com.xforceplus.ultraman.sdk.infra.base.id.node.TimeRandomNodeIdGenerator;
import com.xforceplus.ultraman.sdk.infra.exceptions.EntityClassMissingException;
import com.xforceplus.ultraman.sdk.infra.exceptions.InvalidInputsException;
import io.swagger.annotations.Api;
import io.vavr.Tuple2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.sql.SQLException;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by justin.xu on 08/2023.
 *
 * @since 1.8
 */
@Api(value = "/", tags = "运维接口")
@RequestMapping
public class DevopsController {

    private ZoneId zoneId = ZoneId.of("Asia/Shanghai");

    @Autowired
    private EntityClassEngine engine;
    @Autowired
    private DevOpsService devOpsService;

    @Resource
    private HistoryStorage historyStorage;



    private LongIdGenerator idGenerator = new SnowflakeLongIdGenerator(new TimeRandomNodeIdGenerator());

    private ZoneOffset zoneOffset = OffsetDateTime.now().getOffset();

    private static final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    private static final Integer START_OF_MONTH = 1;

    private static final Integer END_OF_MONTH = 12;

    /**
     * 批量重建索引.
     */
    @PostMapping(value = "/dev-ops/rebuild-indexes", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<DevOpsTaskInfo>> rebuildIndexes(@RequestBody RebuildIndexes rebuildIndexes) {

        List<IEntityClass> entityClasses = null;
        if (null != rebuildIndexes.getAppCode() && !rebuildIndexes.getAppCode().isEmpty()) {
            entityClasses = engine.findAllEntities(rebuildIndexes.getProfile());
        } else {
            entityClasses = engine.findAllEntities(rebuildIndexes.getProfile()).stream().filter(
                    e -> {
                        return rebuildIndexes.getEntityClassIds().contains(Long.toString(e.id()));
                    }
            ).collect(Collectors.toList());
        }

        if (null == entityClasses || entityClasses.isEmpty()) {
            return ResponseEntity.ok(null);
        }

        return ResponseEntity.ok(
                devOpsService.rebuildIndexes(entityClasses,
                        LocalDateTime.parse(rebuildIndexes.getStart(), dateTimeFormatter),
                        LocalDateTime.parse(rebuildIndexes.getEnd(), dateTimeFormatter),
                        rebuildIndexes.isUseCDC(), rebuildIndexes.getProfile()));
    }

    @PutMapping(value = "/dev-ops/single-rebuild")
    public ResponseEntity<DevOpsTaskInfo> rebuildIndex(@RequestBody SingleRebuild singleRebuild) {

        Optional<IEntityClass> entityClass = engine.findAllEntities(singleRebuild.getProfile()).stream().filter(
                e -> {
                    return singleRebuild.getEntityClassId().contains(Long.toString(e.id()));
                }
        ).findFirst();

        return entityClass.map(iEntityClass -> ResponseEntity.ok(devOpsService.rebuildIndex(iEntityClass,
                LocalDateTime.parse(singleRebuild.getStart(), dateTimeFormatter),
                LocalDateTime.parse(singleRebuild.getEnd(), dateTimeFormatter),
                singleRebuild.isUseCDC(),
                singleRebuild.getStartId(), singleRebuild.getProfile()))).orElseGet(() -> ResponseEntity.of(Optional.empty()));

    }

    /**
     * 取消任务.
     */
    @PutMapping(value = "/dev-ops/cancel-rebuild-index/", consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Boolean> cancelTask(@RequestBody List<Long> ids) throws Exception {

        int canceledSize = 0;
        for (Long id : ids) {
            if (devOpsService.cancel(id)) {
                canceledSize++;
            }
        }

        return ResponseEntity.ok(canceledSize == ids.size());
    }

    /**
     * 按照batchId获取当前的任务列表.
     */
    @GetMapping(value = "/dev-ops/list-rebuild-indexes/", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<Collection<DevOpsTaskInfo>> queryTasks(@RequestParam("batchId") long batchId) throws SQLException {
        return ResponseEntity.ok(devOpsService.listTasks(batchId));
    }

    /**
     * 按照TaskId获取当前的任务.
     */
    @GetMapping(value = "/query-rebuild-index", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<DevOpsTaskInfo> queryTask(@RequestParam("maintainId") long maintainId) throws SQLException {
        Optional<DevOpsTaskInfo> devOpsTaskInfo = devOpsService.listTask(maintainId);
        return ResponseEntity.ok(devOpsTaskInfo.orElse(null));
    }

    /**
     * 获取当前的(活动)任务列表.
     */
    @PostMapping(value = "/dev-ops/documentation/create", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<Collection<HistoryTaskInfo>> documentations(@RequestBody Documentations documentations) throws SQLException {
        List<IEntityClass> entityClasses = null;

        if (null != documentations.getAppCode() && !documentations.getAppCode().isEmpty()) {
            entityClasses = engine.findAllEntities("");
        } else {
            entityClasses = engine.findAllEntities("").stream().filter(
                    e -> {
                        return documentations.getEntityClassIds().contains(Long.toString(e.id()));
                    }
            ).collect(Collectors.toList());
        }

        if (null == entityClasses) {
            throw new EntityClassMissingException("oqs documentations, entityClass not found...");
        }

        String filterProfile = null;
        Map<Long, List<IEntityClass>> finalEntityClass = new HashMap<>();
        for (IEntityClass entityClass : entityClasses) {
            finalEntityClass.computeIfAbsent(entityClass.id(), (x) -> new ArrayList<>()).add(entityClass);

            List<String> profiles = null;

            //  如果不传入profile，则需要对该entityClass下的所有数据进行查询.
            if (null == documentations.getProfile() || documentations.getProfile().isEmpty()) {
                profiles = EntityProfileUtils.activeProfiles(entityClass.id());
            } else {
                profiles = Lists.newArrayList(documentations.getProfile());
                filterProfile = documentations.getProfile();
            }

            if (profiles != null) {
                for (String profile : profiles) {
                    engine.loadByCode(entityClass.code(), profile)
                            .ifPresent(iEntityClass -> finalEntityClass.get(entityClass.id()).add(iEntityClass));
                }
            }
        }

        Tuple2<LocalDateTime, LocalDateTime> rangeTime = toStartEndTime(documentations.getYear(), documentations.getMonth());
        if (null == rangeTime) {
            throw new InvalidInputsException("oqs documentations, time range not found...");
        }

        int execMonth = (null != documentations.getMonth() && validMonth(documentations.getMonth())) ? documentations.getMonth() : 0;

        Collection<HistoryTaskInfo> historyTaskInfos = new ArrayList<>();
        long batchId = idGenerator.next();
        for (Map.Entry<Long, List<IEntityClass>> entry : finalEntityClass.entrySet()) {
            long taskId = idGenerator.next();
            int part = 1;
            for (IEntityClass entityClass : entry.getValue()) {

                HistoryTaskInfo historyTaskInfo = pending(batchId, taskId, entityClass, rangeTime._1(), rangeTime._2(),
                        documentations.getYear(), execMonth, 0L,
                        filterProfile, part++, 0, "");

                historyStorage.doDocumentation(entityClass, historyTaskInfo);

                historyTaskInfos.add(historyTaskInfo);
            }
        }

        return ResponseEntity.ok(historyTaskInfos);
    }

    @GetMapping(value = "/dev-ops/documentation/query", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<Collection<HistoryTaskInfo>> query(@RequestParam(value = "taskId", required = false) Long taskId,
                                                      @RequestParam(value = "batchId", required = false) Long batchId) throws SQLException {
        if (null != taskId) {
            return ResponseEntity.ok(historyStorage.queryByTaskId(taskId));
        } else if (null != batchId) {
            return ResponseEntity.ok(historyStorage.list(batchId));
        }

        return ResponseEntity.ok(Collections.emptyList());
    }

    /**
     * 获取当前的(活动)任务列表.
     */
    @PutMapping(value = "/dev-ops/documentation/delete", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<Collection<HistoryTaskInfo>> deleteDocuments(@RequestBody List<String> ids) throws SQLException {
        List<Long> idList = ids.stream().map(Long::parseLong).collect(Collectors.toList());

        Collection<HistoryTaskInfo> historyTaskInfos =
                historyStorage.doDeletes(idList, 0L, "");

        return ResponseEntity.ok(historyTaskInfos);
    }



    private Tuple2<LocalDateTime, LocalDateTime> toStartEndTime(Integer year, Integer month) {

        if (null == year) {
            return null;
        }

        Integer startMonth;
        Integer endMonth;
        if (null != month) {
            if (!validMonth(month)) {
                return null;
            }
            startMonth = month;
            endMonth = month;
        } else {
            startMonth = START_OF_MONTH;
            endMonth = END_OF_MONTH;
        }

        //  最初的时间戳
        LocalDateTime start =
                LocalDateTime.of(year, startMonth, 1, 0, 0, 0,0);

        //  最后的时间戳
        LocalDateTime end =
                YearMonth.from(Year.of(year).atMonth(Month.of(endMonth)))
                        .atEndOfMonth()
                        .atTime(23, 59, 59,999999999);

        return new Tuple2<>(start, end);
    }


    private boolean validMonth(Integer month) {
        return month >= START_OF_MONTH && month <= END_OF_MONTH;
    }

    private HistoryTaskInfo pending(long batchId, long taskId,
                                    IEntityClass entityClass, LocalDateTime start,
                                    LocalDateTime end,
                                    int year, int month,
                                    long checkpoint,
                                    String filterProfile, int part,
                                    long userId, String userName) {

        HistoryTaskInfo historyTaskInfo = new HistoryTaskInfo(
                idGenerator.next(),
                batchId,
                taskId,
                entityClass.id(),
                entityClass.profile(),
                HistoryTaskInfo.Op.DOCUMENTATION.ordinal()
        );

        historyTaskInfo.setRangeStart(start.toInstant(zoneOffset).toEpochMilli());
        historyTaskInfo.setRangeEnd(end.toInstant(zoneOffset).toEpochMilli());
        historyTaskInfo.setCheckPoint(checkpoint);
        historyTaskInfo.setOpUser(userId);
        historyTaskInfo.setYear(year);
        historyTaskInfo.setMonth(month);
        historyTaskInfo.setOpUserName(userName);
        historyTaskInfo.setFilterProfile(filterProfile);
        historyTaskInfo.setPart(part);
        historyTaskInfo.setMessage("TASK INIT");

        return historyTaskInfo;
    }

}
