/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.xforceplus.ultraman.adapter.elasticsearch;

import com.alibaba.google.common.base.Supplier;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.ImmutableMap;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.adapter.elasticsearch.ElasticsearchJson.SearchHit;
import com.xforceplus.ultraman.adapter.elasticsearch.ElasticsearchJson.SqlResult;
import com.xforceplus.ultraman.adapter.elasticsearch.query.utils.ParseSqlNodeUtils;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.entity.IRelation;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.DataContext;
import org.apache.calcite.adapter.java.AbstractQueryableTable;
import org.apache.calcite.linq4j.Enumerable;
import org.apache.calcite.linq4j.Enumerator;
import org.apache.calcite.linq4j.Linq4j;
import org.apache.calcite.linq4j.QueryProvider;
import org.apache.calcite.linq4j.Queryable;
import org.apache.calcite.linq4j.function.Function1;
import org.apache.calcite.plan.RelOptCluster;
import org.apache.calcite.plan.RelOptTable;
import org.apache.calcite.rel.RelFieldCollation;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.RelDataTypeFactory;
import org.apache.calcite.schema.SchemaPlus;
import org.apache.calcite.schema.TranslatableTable;
import org.apache.calcite.schema.impl.AbstractTableQueryable;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.sql.dialect.MysqlSqlDialect;
import org.apache.calcite.util.Pair;
import org.apache.calcite.util.Util;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.client.ElasticsearchTransport;

/**
 * Table based on an Elasticsearch index.
 */
@Slf4j
public class ElasticsearchTable extends AbstractQueryableTable implements TranslatableTable {

    /**
     * Used for constructing (possibly nested) Elastic aggregation nodes.
     */
    private static final String AGGREGATIONS = "aggregations";
    public final ObjectMapper mapper;
    private final ElasticsearchVersion version;
    private final EntityClassEngine engine;
    private final ContextService contextService;
    private final ProfileFetcher fetcher;
    private final String indexName;
    private final ElasticsearchTransport transport;
    private final String entityCode;
    private final String prefix;

    public List<String> list = new ArrayList<>();


    /**
     * Creates an ElasticsearchTable.
     */
    public ElasticsearchTable(EntityClassEngine engine, ElasticsearchTransport transport,
                              ContextService contextService, ProfileFetcher fetcher, String prefix) {
        super(Object[].class);
        this.engine = engine;
        this.transport = transport;
        this.version = Optional.ofNullable(transport).map(x -> x.version).orElse(null);
        this.mapper = Optional.ofNullable(transport).map(x -> x.mapper()).orElse(null);
        this.contextService = contextService;
        this.fetcher = fetcher;
        this.indexName = Optional.ofNullable(transport).map(x -> x.indexName).orElse(null);
        this.entityCode = Optional.ofNullable(transport).map(x -> x.entityCode).orElse(null);
        this.prefix = prefix;
    }

    /**
     * In ES 5.x scripted fields start with {@code params._source.foo} while in ES2.x {@code _source.foo}. Helper method to build correct query based on
     * runtime version of elastic. Used to keep backwards compatibility with ES2.
     *
     * @return string to be used for scripted fields
     * @see <a href="https://github.com/elastic/elasticsearch/issues/20068">_source variable</a>
     * @see <a href="https://www.elastic.co/guide/en/elasticsearch/reference/master/modules-scripting-fields.html">Scripted
     * Fields</a>
     */
    public String scriptedFieldPrefix() {
        // ES2 vs ES5 scripted field difference
        return version == ElasticsearchVersion.ES2
                ? ElasticsearchConstants.SOURCE_GROOVY
                : ElasticsearchConstants.SOURCE_PAINLESS;
    }

    private void merge(ObjectNode query, ObjectNode newNode, String key) {
        if (query.has(key)) {
            ObjectNode jsonNode = (ObjectNode) query.get(key);
            JsonNode newTargetNode = newNode.get("query");
            //TODO
            JsonNode merge = merge(jsonNode, newTargetNode);
//            query.setAll((ObjectNode) merge);
        } else {
            query.setAll(newNode);
        }
    }

    public static JsonNode merge(JsonNode mainNode, JsonNode updateNode) {

        Iterator<String> fieldNames = updateNode.fieldNames();

        while (fieldNames.hasNext()) {
            String updatedFieldName = fieldNames.next();
            JsonNode valueToBeUpdated = mainNode.get(updatedFieldName);
            JsonNode updatedValue = updateNode.get(updatedFieldName);

            // If the node is an @ArrayNode
            if (valueToBeUpdated != null && valueToBeUpdated.isArray()) {
                
                if(updatedValue.isArray()) {
                    // running a loop for all elements of the updated ArrayNode
                    for (int i = 0; i < updatedValue.size(); i++) {
                        JsonNode updatedChildNode = updatedValue.get(i);
                        // Create a new Node in the node that should be updated, if there was no corresponding node in it
                        // Use-case - where the updateNode will have a new element in its Array
                        if (valueToBeUpdated.size() <= i) {
                            ((ArrayNode) valueToBeUpdated).add(updatedChildNode);
                        }
                        // getting reference for the node to be updated
                        JsonNode childNodeToBeUpdated = valueToBeUpdated.get(i);
                        merge(childNodeToBeUpdated, updatedChildNode);
                    }
                } else {
                    ((ArrayNode) valueToBeUpdated).add(updatedValue);
                }
                // if the Node is an @ObjectNode
            } else if (valueToBeUpdated != null && valueToBeUpdated.isObject()) {
                merge(valueToBeUpdated, updatedValue);
            } else {
                if (mainNode instanceof ObjectNode) {
                    ((ObjectNode) mainNode).replace(updatedFieldName, updatedValue);
                }
            }
        }
        return mainNode;
    }


    /**
     * Executes a "find" operation on the underlying index.
     *
     * @param ops          List of operations represented as Json strings.
     * @param fields       List of fields to project; or null to return map
     * @param sort         list of fields to sort and their direction (asc/desc)
     * @param aggregations aggregation functions
     * @return Enumerator of results
     */
    public Enumerable<Object> find(List<String> ops,
                                   List<Map.Entry<String, Class>> fields,
                                   List<Map.Entry<String, RelFieldCollation.Direction>> sort,
                                   List<String> groupBy,
                                   List<Map.Entry<String, String>> aggregations,
                                   Map<String, String> mappings,
                                   Long offset, Long fetch, RelNode rawTree, DataContext dataContext) throws IOException {

        if (transport == null) {
            return Linq4j.emptyEnumerable();
        }

        final Function1<ElasticsearchJson.SearchHit, Object> getter =
                ElasticsearchEnumerators.getter(fields, ImmutableMap.copyOf(mappings));
        try {
            if (!aggregations.isEmpty() || !groupBy.isEmpty()) {
                // process aggregations separately
                return aggregate(ops, fields, sort, groupBy, aggregations, mappings, offset, fetch);
            }

            final ObjectNode query = mapper.createObjectNode();
            // manually parse from previously concatenated string;
            Boolean join_query = contextService.getAll().get("join_query") != null && (Boolean) contextService.getAll().get("join_query");
            for (String op : ops) {
                if (mapper.readTree(op).get("query") != null) {
                    if (join_query) {
                        /**根据filter过滤条件拆成has_parent与has_child查询方式 过滤parent-child index索引表**/
                        generateJoinModelQuery(query, op);
                    } else {
                        merge(query, (ObjectNode) mapper.readTree(op), "query");
//                        query.setAll((ObjectNode) mapper.readTree(op));
                    }
                    if (contextService.getAll().get("hasCount") != null && (boolean) contextService.getAll().get("hasCount")) {
                        //query.put("track_total_hits", true);
                        executeQueryCount(mappings, query.toString());
                    }
                    continue;
                }
                query.setAll((ObjectNode) mapper.readTree(op));
            }

            if (!sort.isEmpty()) {
                ArrayNode sortNode = query.withArray("sort");
                sort.forEach(e -> {
                            String key = e.getKey();
                            if (!StringUtils.isEmpty(key)) {
                                boolean isJson = key.startsWith("__json");
                                JsonNode input = null;
                                if (isJson) {
                                    if (key.contains("$order$")) {
                                        key = key.replace("$order$", e.getValue().isDescending() ? "desc" : "asc");
                                    }
                                    try {
                                        input = mapper.readTree(key.substring(6));
                                    } catch (JsonProcessingException ex) {
                                        ex.printStackTrace();
                                    }
                                } else {
                                    input = mapper.createObjectNode()
                                            .put(key.toLowerCase(Locale.ROOT),
                                                    e.getValue().isDescending() ? "desc" : "asc");
                                }
                                if (input != null) {
                                    sortNode.add(input);
                                }
                            }
                        }
                );
            }
            if (offset != null) {
                query.put("from", offset);
            }
            if (fetch != null) {
                query.put("size", fetch);
            }

            query.put("_source", false);


            Iterable<ElasticsearchJson.SearchHit> iter;
            if (offset == null) {
                // apply scrolling when there is no offsets
                iter = () -> new Scrolling(transport, contextService).query(query);
            } else {
                final ElasticsearchJson.Result search = transport.search().apply(query);
                ElasticsearchJson.SearchHits searchHits = search.searchHits();
                ElasticsearchJson.SearchTotal total = searchHits.total();
                if (total != null && contextService.getAll().get("show_count") == null) {
                    contextService.getAll().put("show_count", total.value());
                }
                iter = () -> search.searchHits().flattenHits().iterator();
            }
            return Linq4j.asEnumerable(iter).select(getter);

        } catch (Throwable throwable) {
            log.warn("elasticsearch table execute query failed!, {}", throwable.getMessage());
            contextService.getAll().compute("errors", (k, v) -> {
                if (v == null) {
                    v = new ArrayList<>();
                }
                ((List<String>) v).add(throwable.getMessage());
                return v;
            });
            throw throwable;
        }
    }

    /**
     * 构建count查询DQL语句
     *
     * @param mappings
     * @param op
     **/
    private void executeQueryCount(Map<String, String> mappings, String op) throws IOException {
        Map<String, String> aggregationMap = new HashMap();
        Map<String, Class> fieldMap = new HashMap();
        aggregationMap.put("c", "{\"value_count\":{\"field\":\"_id\"}}");
        fieldMap.put("c", Long.class);
        List<Entry<String, Class>> fieldEntries = new ArrayList<>();
        List<Entry<String, String>> aggregationEntries = new ArrayList<>();
        aggregationEntries.addAll(aggregationMap.entrySet());
        fieldEntries.addAll(fieldMap.entrySet());
        aggregate(Collections.singletonList(op), fieldEntries, new ArrayList<>(), new ArrayList<>(), aggregationEntries, mappings, null, null);
    }

    /**
     * 适配join(chind-parent) elastic 索引模型查询语句
     *
     * @param query
     * @param op
     **/
    private void generateJoinModelQuery(ObjectNode query, String op) throws JsonProcessingException {
        JsonNode queryNodes = mapper.readTree(op).get("query");
        /**取出filter中的过滤条件**/
        JsonNode jsonNode = queryNodes.get("constant_score").get("filter");
        if (jsonNode.get("bool") != null) {
            jsonNode = jsonNode.get("bool").get("must");
        }
        /**
         * {
         *   "query": {
         *     "bool": {
         *       "must": [
         *         {
         *           "term": {
         *             "test0719005.zfc1": "test"
         *           }
         *         },
         *         {
         *           "has_parent": {
         *             "parent_type": "test0719002",
         *             "query": {
         *               "match": {
         *                 "id": "1681957790106517504"
         *               }
         *             }
         *           }
         *         }
         *       ]
         *     }
         *   }
         * }
         * **/
        Map<String, Object> parentTerms = new HashMap<>();
        Map<String, Object> childTerms = new HashMap<>();
        Map<String, JsonNode> terms = new HashMap<>();
        if (jsonNode instanceof ArrayNode) {
            for (JsonNode term : jsonNode) {
                praseTerms(parentTerms, childTerms, terms, term);
            }
        } else if (jsonNode instanceof ObjectNode) {
            praseTerms(parentTerms, childTerms, terms, jsonNode);
        }
        ObjectNode parentAndChildQuery = mapper.createObjectNode();
        /**inner_hits带出子明细数据配置参数**/
        ObjectNode innerHits = mapper.createObjectNode();
        /**has_parent**/
        if (parentTerms.get("id") != null) {
            ObjectNode newBuildQuery = mapper.createObjectNode();
            ObjectNode hasParent = mapper.createObjectNode();
            ObjectNode hasParentItems = mapper.createObjectNode();
            JsonNode bool = constructQuery(childTerms, terms);
            if (childTerms.size() >= 1) {
                bool = constructQuery(childTerms, terms).get("bool");
                ArrayNode must = (ArrayNode) bool.get("must");
                must.add(hasParent);
            }
            hasParentItems.put("parent_type", entityCode);
            hasParentItems.put("query", constructQuery(parentTerms, terms));
            hasParentItems.put("inner_hits", innerHits);
            hasParent.put("has_parent", hasParentItems);
            newBuildQuery.put("bool", bool);
            parentAndChildQuery.put("query", newBuildQuery);
        } else {
            /**has_child**/
            Optional<Entry<String, Object>> first = childTerms.entrySet().stream().findFirst();
            if (first.isPresent()) {
                String type = first.get().getKey().split("\\.")[0];
                ObjectNode hasChild = mapper.createObjectNode();
                ObjectNode hasChildItems = mapper.createObjectNode();
                hasChildItems.put("type", type);
                hasChildItems.put("query", constructQuery(childTerms, terms));
                hasChildItems.put("inner_hits", innerHits);
                hasChild.put("has_child", hasChildItems);
                parentAndChildQuery.put("query", hasChild);
                if (parentTerms.size() >= 1) {
                    parentAndChildQuery.put("post_filter", constructQuery(parentTerms, terms));
                }
            }
        }
        query.setAll(parentAndChildQuery);
    }

    /**
     * 解析过滤条件
     *
     * @param terms
     * @param childTerms
     * @param parentTerms
     * @param term
     **/
    private void praseTerms(Map<String, Object> parentTerms, Map<String, Object> childTerms, Map<String, JsonNode> terms, JsonNode term) {
        JsonNode termTuple = term.get("term");
        Supplier<Iterator<Entry<String, JsonNode>>> termFields = termTuple::fields;
        Iterator<Entry<String, JsonNode>> entryIterator = termFields.get();
        Entry<String, JsonNode> next = entryIterator.next();
        if (!next.getKey().contains(".")) {
            terms.put(next.getKey(), term);
            parentTerms.put(next.getKey(), next.getValue());
        } else {
            terms.put(next.getKey(), term);
            childTerms.put(next.getKey(), next.getValue());
        }
    }

    /**
     * 构建elasticsearch dsl过滤查询条件
     *
     * @param parentChild
     * @param terms
     * @return
     **/
    private ObjectNode constructQuery(Map<String, Object> parentChild, Map<String, JsonNode> terms) {
        ObjectNode parentBool = mapper.createObjectNode();
        ObjectNode boolNode = mapper.createObjectNode();
        ArrayNode must = mapper.createArrayNode();
        parentChild.entrySet().forEach(entry -> {
            must.add(terms.get(entry.getKey()));
        });
        parentBool.put("must", must);
        boolNode.put("bool", parentBool);
        return boolNode;
    }

    private Enumerable<Object> aggregate(List<String> ops,
                                         List<Map.Entry<String, Class>> fields,
                                         List<Map.Entry<String, RelFieldCollation.Direction>> sort,
                                         List<String> groupBy,
                                         List<Map.Entry<String, String>> aggregations,
                                         Map<String, String> mapping,
                                         Long offset, Long fetch) throws IOException {

        if ((null != groupBy && !groupBy.isEmpty()) && offset != null) {
            String message = "Currently ES doesn't support generic pagination "
                    + "with aggregations. You can still use LIMIT keyword (without OFFSET). "
                    + "For more details see https://github.com/elastic/elasticsearch/issues/4915";
            throw new IllegalStateException(message);
        }
        ObjectNode query = mapper.createObjectNode();
        // manually parse into JSON from previously concatenated strings
        for (String op : ops) {
            if (contextService.getAll().get("join_query") != null && (Boolean) contextService.getAll().get("join_query")) {
                /**根据filter过滤条件拆成has_parent与has_child查询方式 过滤parent-child index索引表**/
                if (mapper.readTree(op).get("query") != null) {
                    generateJoinModelQuery(query, op);
                } else {
                    query.setAll((ObjectNode) mapper.readTree(op));
                }
            } else {
                query.setAll((ObjectNode) mapper.readTree(op));
            }
        }
        // remove / override attributes which are not applicable to aggregations
        query.put("_source", false);
        query.put("size", 0);
        query.remove("script_fields");
        // set _source = false and size = 0, `FetchPhase` would still be executed
        // to fetch the metadata fields and visit the Lucene stored_fields,
        // which would lead to performance declined dramatically.
        // `stored_fields = _none` can prohibit such behavior entirely
        query.put("stored_fields", "_none_");

        // allows to detect aggregation for count(*)
        final Predicate<Map.Entry<String, String>> isCountStar = e -> e.getValue()
                .contains("\"" + ElasticsearchConstants.ID + "\"");

        // list of expressions which are count(*)
        final Set<String> countAll = aggregations.stream()
                .filter(isCountStar)
                .map(Map.Entry::getKey).collect(Collectors.toSet());

        final Map<String, String> fieldMap = new HashMap<>();

        // due to ES aggregation format. fields in "order by" clause should go first
        // if "order by" is missing. order in "group by" is un-important
        final Set<String> orderedGroupBy = new LinkedHashSet<>();
        orderedGroupBy.addAll(sort.stream().map(Map.Entry::getKey).collect(Collectors.toList()));
        orderedGroupBy.addAll(groupBy);

        // construct nested aggregations node(s)
        ObjectNode parent = query.with(AGGREGATIONS);
        for (String name : orderedGroupBy) {
            final String aggName = "g_" + name;
            fieldMap.put(aggName, name);

            final ObjectNode section = parent.with(aggName);
            final ObjectNode terms = section.with("terms");
            terms.put("field", name);

            transport.mapping.missingValueFor(name).ifPresent(m -> {
                // expose missing terms. each type has a different missing value
                terms.set("missing", m);
            });

            if (fetch != null) {
                terms.put("size", fetch);
            }

            sort.stream().filter(e -> e.getKey().equals(name)).findAny()
                    .ifPresent(s ->
                            terms.with("order")
                                    .put("_key", s.getValue().isDescending() ? "desc" : "asc"));

            parent = section.with(AGGREGATIONS);
        }

        // simple version for queries like "select count(*), max(col1) from table" (no GROUP BY cols)
        if (!groupBy.isEmpty() || !aggregations.stream().allMatch(isCountStar)) {
            for (Map.Entry<String, String> aggregation : aggregations) {
                JsonNode value = mapper.readTree(aggregation.getValue());
                parent.set(aggregation.getKey(), value);
            }
        }

        final Consumer<JsonNode> emptyAggRemover = new Consumer<JsonNode>() {
            @Override
            public void accept(JsonNode node) {
                if (!node.has(AGGREGATIONS)) {
                    node.elements().forEachRemaining(this);
                    return;
                }
                JsonNode agg = node.get(AGGREGATIONS);
                if (agg.size() == 0) {
                    ((ObjectNode) node).remove(AGGREGATIONS);
                } else {
                    this.accept(agg);
                }
            }
        };

        // cleanup query. remove empty AGGREGATIONS element (if empty)
        emptyAggRemover.accept(query);

        // This must be set to true or else in 7.X and 6/7 mixed clusters
        // will return lower bounded count values instead of an accurate count.
        if (groupBy.isEmpty()
                && version.elasticVersionMajor() >= ElasticsearchVersion.ES6.elasticVersionMajor()) {
            query.put("track_total_hits", true);
        }

        ElasticsearchJson.Result res = transport.search(Collections.emptyMap()).apply(query);

        final List<Map<String, Object>> result = new ArrayList<>();
        if (res.aggregations() != null) {
            // collect values
            ElasticsearchJson.visitValueNodes(res.aggregations(), m -> {
                // using 'Collectors.toMap' will trigger Java 8 bug here
                Map<String, Object> newMap = new LinkedHashMap<>();
                for (String key : m.keySet()) {
                    newMap.put(fieldMap.getOrDefault(key, key), m.get(key));
                }
                result.add(newMap);
            });
        } else {
            // probably no group by. add single result
            result.add(new LinkedHashMap<>());
        }

        // elastic exposes total number of documents matching a query in "/hits/total" path
        // this can be used for simple "select count(*) from table"
        final long total = res.searchHits().total().value();
        contextService.getAll().put("show_count", total);
        if (groupBy.isEmpty()) {
            // put totals automatically for count(*) expression(s), unless they contain group by
            for (String expr : countAll) {
                result.forEach(m -> m.put(expr, total));
            }
        }

        final Function1<ElasticsearchJson.SearchHit, Object> getter =
                ElasticsearchEnumerators.getter(fields, ImmutableMap.copyOf(mapping));

        ElasticsearchJson.SearchHits hits =
                new ElasticsearchJson.SearchHits(res.searchHits().total(), result.stream()
                        .map(r -> new ElasticsearchJson.SearchHit("_id", r, null, null))
                        .collect(Collectors.toList()));
        //hits.hits().remove(0);
        return Linq4j.asEnumerable(hits.hits()).select(getter);
    }

    private RelDataType fieldTypeToRelDataType(RelDataTypeFactory relDataTypeFactory, Class type) {
        return relDataTypeFactory.createJavaType(type);
    }

    @Override
    public RelDataType getRowType(RelDataTypeFactory relDataTypeFactory) {
        String profile = fetcher.getProfile(contextService.getAll());
        Optional<IEntityClass> targetClass = engine.loadByCode(entityCode,
                profile);
        IEntityClass iEntityClass = targetClass.get();
        EntityClassGroup describe = engine.describe(targetClass.get(), profile);
        Collection<IEntityField> allFields = describe.getAllFields();
        final List<Map.Entry<String, RelDataType>> names = allFields.stream().map(
                        x -> Pair.of(x.name().toLowerCase(Locale.ROOT).replace(".", "_"),
                                fieldTypeToRelDataType(relDataTypeFactory, x.type().getJavaType())))
                .collect(Collectors.toList());
        allFields.stream().forEach(x -> {
            if (x.type() == FieldType.STRINGS) {
                names.add(Pair.of(x.name().toLowerCase(Locale.ROOT).replace(".", "_").concat("@raw"),
                        fieldTypeToRelDataType(relDataTypeFactory, x.type().getJavaType())));
            }
        });
        /**当有重复relations对象依赖，对关联关系不一样时，只取一遍relations对象的元数据信息，否则calcite会抛SqlValidatorException异常信息**/
        Map<String, List<Pair<String, RelDataType>>> relationSet = new HashMap<>();
        /**加入宽表所有的依赖字段，避免SQL解析验证元数据时抛出找不到表字段**/
        for (IRelation iRelation : iEntityClass.relations()) {
            Optional<IEntityClass> iRelationEntityClassGroup = engine.load(String.valueOf(iRelation.getEntityClassId()), profile);
            IEntityClass iRelationEntityClass = iRelationEntityClassGroup.get();
            Collection<IEntityField> relationAllFields = engine.describe(iRelationEntityClass, profile).getAllFields();
            List<Pair<String, RelDataType>> relationNames = relationAllFields.stream().map(
                            x -> Pair.of(iRelationEntityClass.code().toLowerCase(Locale.ROOT).concat("_").concat(x.name().replace(".", "_")),
                                    fieldTypeToRelDataType(relDataTypeFactory, x.type().getJavaType())))
                    .collect(Collectors.toList());
            relationAllFields.stream().forEach(x -> {
                if (x.type() == FieldType.STRINGS) {
                    relationNames.add(Pair.of(iRelationEntityClass.code().toLowerCase(Locale.ROOT).concat("_").concat(x.name().toLowerCase(Locale.ROOT)).replace(".", "_").concat("@raw"),
                            fieldTypeToRelDataType(relDataTypeFactory, x.type().getJavaType())));
                }
            });
            relationSet.put(iRelationEntityClass.code(), relationNames);
        }
        for (List<Pair<String, RelDataType>> relationValue : relationSet.values()) {
            names.addAll(relationValue);
        }
        RelDataType structType = relDataTypeFactory.createStructType(names);
        return structType;
    }

    @Override
    public String toString() {
        return "ElasticsearchTable{" + indexName + "}";
    }

    @Override
    public <T> Queryable<T> asQueryable(QueryProvider queryProvider, SchemaPlus schema,
                                        String tableName) {
        return new ElasticsearchQueryable<>(queryProvider, schema, this, tableName);
    }

    @Override
    public RelNode toRel(RelOptTable.ToRelContext context, RelOptTable relOptTable) {
        final RelOptCluster cluster = context.getCluster();
        return new ElasticsearchTableScan(cluster, cluster.traitSetOf(ElasticsearchRel.CONVENTION),
                relOptTable, this, null);
    }

/**
 * Implementation of {@link Queryable} based on a {@link ElasticsearchTable}.
 *
 * @param <T> element type
 */
public static class ElasticsearchQueryable<T> extends AbstractTableQueryable<T> {

    ElasticsearchQueryable(QueryProvider queryProvider, SchemaPlus schema,
                           ElasticsearchTable table, String tableName) {
        super(queryProvider, schema, table, tableName);
    }

    @Override
    public Enumerator<T> enumerator() {
        return null;
    }

    private ElasticsearchTable getTable() {
        return (ElasticsearchTable) table;
    }

    /**
     * Called via code-generation.
     *
     * @param ops    list of queries (as strings)
     * @param fields projection
     * @return result as enumerable
     * @see ElasticsearchMethod#ELASTICSEARCH_QUERYABLE_FIND
     */
    @SuppressWarnings("UnusedDeclaration")
    public Enumerable<Object> find(List<String> ops,
                                   List<Map.Entry<String, Class>> fields,
                                   List<Map.Entry<String, RelFieldCollation.Direction>> sort,
                                   List<String> groupBy,
                                   List<Map.Entry<String, String>> aggregations,
                                   Map<String, String> mappings,
                                   Long offset, Long fetch, RelNode rawTree, DataContext dataContext) {
        try {
//                long currentTimeMillis = System.currentTimeMillis();
//                System.out.println("Come into ElasticSearch Table " + currentTimeMillis);
            Enumerable<Object> objects = getTable().find(ops, fields, sort, groupBy, aggregations, mappings, offset, fetch, rawTree, dataContext);
//                System.out.println("Query cost is " + (System.currentTimeMillis() - currentTimeMillis));
            return objects;
        } catch (Throwable e) {
            throw new RuntimeException("Failed to query " + getTable().indexName, e);
        }
    }
}
}
