package com.xforceplus.ultraman.adapter.elasticsearch.listener;


import com.xforceplus.ultraman.adapter.elasticsearch.config.BocpElasticsearchConfiguration;
import com.xforceplus.ultraman.adapter.elasticsearch.query.dto.ElasticTenantProfile;
import com.xforceplus.ultraman.adapter.elasticsearch.query.po.BocpElasticConfigPo;
import com.xforceplus.ultraman.adapter.elasticsearch.service.ManageBocpMetadataService;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.CommonProperty;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.SegmentFieldType;
import com.xforceplus.ultraman.adapter.elasticsearch.service.impl.ManageBocpMetadataServiceImpl;
import com.xforceplus.ultraman.adapter.elasticsearch.utils.ParseBocpJsonUtils;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.jsonschema.pojo.SchemaSdkSetting;
import com.xforceplus.ultraman.sdk.core.config.CdcConfig;
import com.xforceplus.ultraman.sdk.core.pipeline.operation.validator.dynamic.DynamicRegexChecked;
import com.xforceplus.ultraman.sdk.core.pipeline.operation.validator.dynamic.DynamicRegexCheckedConfig;
import com.xforceplus.ultraman.sdk.core.pipeline.operation.validator.dynamic.DynamicRequired;
import com.xforceplus.ultraman.sdk.core.pipeline.operation.validator.dynamic.DynamicRequiredConfig;
import com.xforceplus.ultraman.sdk.infra.Refreshable;
import com.xforceplus.ultraman.transfer.common.event.SDKMetadataEvent;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

import com.xforceplus.ultraman.adapter.elasticsearch.service.utils.BocpMetabaseCacheUtils;

import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

/**
 * @program: ultraman-oqsengine-plus
 * @ClassName SdkMetadataListener
 * @description: 监听元数据发布后的变化，实现对elasticsearch 索引库进行修改
 * @author: WanYi
 * @create: 2023-05-15 16:27
 * @Version 1.0
 **/
@Slf4j
public class ElasticSearchSdkMetadataListener {

    //@Autowired
    // private EntityClassRelationService entityClassRelationService;

    @Autowired
    private ManageBocpMetadataService manageBocpMetadataService;

    @Autowired(required = false)
    private BocpElasticsearchConfiguration bocpElasticsearchConfiguration;

    @Autowired
    public EntityClassEngine entityClassEngine;


    @Autowired(required = false)
    DynamicRequiredConfig dynamicRequiredConfig;

    @Autowired(required = false)
    DynamicRegexCheckedConfig dynamicRegexCheckedConfig;

    @Autowired
    private CdcConfig cdcConfig;

/*
  public void setEntityClassRelationService(EntityClassRelationService entityClassRelationService) {
    this.entityClassRelationService = entityClassRelationService;
  }*/

    public void setManageBocpMetadataService(ManageBocpMetadataService manageBocpMetadataService) {
        this.manageBocpMetadataService = manageBocpMetadataService;
    }

    @EventListener(SDKMetadataEvent.class)
    public void onApplicationEvent(SDKMetadataEvent payload) {
        if (payload instanceof SDKMetadataEvent) {
            /**初始化元数据缓存信息**/
            // entityClassRelationService.initEntityClassRelation();
            List<SchemaSdkSetting> sdkSettings = payload.getSdkSettings();
            AtomicBoolean isParsedFromBocp = new AtomicBoolean(false);
            sdkSettings.forEach(sdkSetting -> {
                if (StringUtils.equalsIgnoreCase(sdkSetting.getType(), "sdk20")) {
                    cdcConfig.setInclude(new ArrayList<>(cdcConfig.getInclude()));
                    cdcConfig.setExclude(Collections.emptyList());
                    /**解析bocp配置,转换成elasticsearch configuration 配置对象**/
                    BocpElasticsearchConfiguration bocpElasticConfig = ParseBocpJsonUtils.parseBocpJson(sdkSetting.getSetting());
                    /**根据bocp配置创建更新index索引表**/
                    parseBocpConfig(bocpElasticConfig, cdcConfig);
                    //do extra from this config
                    prepareExtraConfig();
                    isParsedFromBocp.set(true);
                }
            });

            //load from file
            if (!isParsedFromBocp.get() && bocpElasticsearchConfiguration != null) {
                parseBocpConfig(bocpElasticsearchConfiguration, cdcConfig);
            }

            manageBocpMetadataService.setLoadFinish(true);
        }
    }

    /**
     * prepare extra config
     */

    /***
     * 解析bocp的配置信息，根据bocp配置信息，建立相应的elastic index索引表
     * @param bocpElasticsearchConfiguration
     */
    private void parseBocpConfig(BocpElasticsearchConfiguration bocpElasticsearchConfiguration, CdcConfig cdcConfig) {
        List<Tuple2<String, String>> errors = new ArrayList<>();
        bocpElasticsearchConfiguration.getBocpElasticConfigs().forEach(bocpElasticConfig -> {
            if (bocpElasticConfig.isEnableSync()) {
                long entityClassId = bocpElasticConfig.getEntityClassId();
                /**创建默认索引**/
                IEntityClass iEntityClass = entityClassEngine.load(String.valueOf(entityClassId), CommonProperty.defaultProfile).get();
                if (iEntityClass != null) {
                    cdcConfig.getInclude().add(iEntityClass.code());
                    Collection<ElasticTenantProfile> selfSegmentRule;
                    if (bocpElasticConfig.getSelfSegmentRule() == null) {
                        selfSegmentRule = Collections.EMPTY_LIST;
                    } else {
                        bocpElasticConfig.getSelfSegmentRule().setTenantCode(CommonProperty.defaultProfile);
                        selfSegmentRule = Collections.singletonList(bocpElasticConfig.getSelfSegmentRule());
                    }
                    manageBocpMetadataService.updateMetadataCache(selfSegmentRule, iEntityClass, errors);
                    if (bocpElasticConfig.getElasticTenantProfiles() != null) {
                        List<ElasticTenantProfile> enableSegmentTenantProfiles = bocpElasticConfig.getElasticTenantProfiles().stream().filter(tenantProfile -> {
                            if (!tenantProfile.isEnableSegment()) {
                                return true;
                            } else {
                                return false;
                            }
                        }).collect(Collectors.toList());
                        /**当租户没有开启索引拆分时，创建默认的租户隔离索引**/
                        updateMetadataCache(errors, iEntityClass, enableSegmentTenantProfiles);
                        enableSegmentTenantProfiles = bocpElasticConfig.getElasticTenantProfiles().stream().filter(tenantProfile -> {
                            if (tenantProfile.isEnableSegment()) {
                                return true;
                            } else {
                                return false;
                            }
                        }).collect(Collectors.toList());
                        updateMetadataCache(errors, iEntityClass, enableSegmentTenantProfiles);
                    }
                }
            }
        });
        ((ManageBocpMetadataServiceImpl) manageBocpMetadataService).printErrors(errors);
    }

    private void prepareExtraConfig() {

        BocpMetabaseCacheUtils.getBocpConfigAllKey().forEach(k -> {
            Map<Long, BocpElasticConfigPo> bocpConfig = BocpMetabaseCacheUtils.getBocpConfig(k);
            if (bocpConfig != null) {
                bocpConfig.forEach((key, value) -> {
                    String tenantCode = value.getTenantCode();
                    if (StringUtils.isEmpty(tenantCode)) {
                        tenantCode = "default";
                    }
                    if (value.isEnableSegment()) {
                        String entityClassCode = value.getEntityClassCode();
                        String segmentFieldName = value.getSegmentFieldName();
                        if (dynamicRequiredConfig != null) {
                            dynamicRequiredConfig.buildTenant(tenantCode, DynamicRequired::new).addRequired(entityClassCode, segmentFieldName);
                        }

                        if (dynamicRegexCheckedConfig != null) {
                            if (!StringUtils.isEmpty(value.getSegmentDateFormat()) && value.getSegmentFieldType() == SegmentFieldType.STRING) {
                                dynamicRegexCheckedConfig.buildTenant(tenantCode, DynamicRegexChecked::new).addRegexChecked(entityClassCode, segmentFieldName, value.getSegmentDateFormat());
                            }
                        }
                    }
                });
            }
        });
    }

    /**
     * 更新租户隔离元数据缓存信息
     *
     * @param errors
     * @param iEntityClass
     * @param enableSegmentTenantProfiles
     **/
    private void updateMetadataCache(List<Tuple2<String, String>> errors, IEntityClass iEntityClass,
                                     List<ElasticTenantProfile> enableSegmentTenantProfiles) {
        if (enableSegmentTenantProfiles.size() >= 1) {
            manageBocpMetadataService.updateMetadataCache(enableSegmentTenantProfiles, iEntityClass, errors);
        }
    }
}

