package com.xforceplus.ultraman.adapter.elasticsearch.query;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.ImmutableMap;
import com.xforceplus.metadata.schema.runtime.MetadataEngine;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.adapter.elasticsearch.*;
import com.xforceplus.ultraman.adapter.elasticsearch.query.utils.ElasticSearchSqlConverter;
import com.xforceplus.ultraman.adapter.elasticsearch.query.utils.ParseSqlNodeUtils;
import com.xforceplus.ultraman.adapter.elasticsearch.rules.ElasticsearchFilter;
import com.xforceplus.ultraman.adapter.elasticsearch.service.ManageBocpMetadataService;
import com.xforceplus.ultraman.adapter.elasticsearch.transport.ElasticsearchTransportExecutor;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.MysqlSqlDialectEx;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.select.SelectConfig;
import com.xforceplus.ultraman.sdk.core.calcite.oqs.*;
import com.xforceplus.ultraman.sdk.core.config.ExecutionConfig;
import com.xforceplus.ultraman.sdk.core.datasource.route.TransportExecutor;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import io.micrometer.core.annotation.Timed;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.DataContext;
import org.apache.calcite.adapter.enumerable.JavaRowFormat;
import org.apache.calcite.adapter.enumerable.PhysType;
import org.apache.calcite.adapter.enumerable.PhysTypeImpl;
import org.apache.calcite.adapter.java.JavaTypeFactory;
import org.apache.calcite.jdbc.CalciteConnection;
import org.apache.calcite.jdbc.JavaTypeFactoryImpl;
import org.apache.calcite.linq4j.Enumerable;
import org.apache.calcite.linq4j.Linq4j;
import org.apache.calcite.linq4j.function.Function1;
import org.apache.calcite.rel.RelFieldCollation;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.RelShuttleImpl;
import org.apache.calcite.rel.core.AggregateCall;
import org.apache.calcite.rel.hint.RelHint;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.RelDataTypeField;
import org.apache.calcite.rel.type.StructKind;
import org.apache.calcite.rex.RexLiteral;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.schema.Schema;
import org.apache.calcite.schema.SchemaPlus;
import org.apache.calcite.sql.SqlKind;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.sql.SqlOrderBy;
import org.apache.calcite.sql.SqlSelect;
import org.apache.calcite.sql.parser.SqlParseException;
import org.apache.calcite.util.Pair;
import org.apache.calcite.util.Util;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.client.ElasticsearchTransport;
import org.elasticsearch.client.RestClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;

import javax.sql.DataSource;
import java.io.IOException;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.sql.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

@Slf4j
public class ElasticSearchQueryProvider implements
        DataQueryProvider {

    @Lazy
    @Qualifier("elasticSearchDS")
    @Autowired
    private DataSource esDataSource;

    @Autowired
    private ContextService contextService;
    @Autowired
    private EntityClassEngine engine;
    @Lazy
    @Autowired
    private ExecutionConfig executionConfig;

    @Autowired
    private TransportExecutor transportExecutor;

    private int fetchSize;

    @Autowired
    private ManageBocpMetadataService manageBocpMetadataService;

    private Map<Tuple2<String, String>, ElasticsearchTransport> transportMap = new ConcurrentHashMap<>();

    @Autowired
    private ProfileFetcher fetcher;

    @Autowired
    private ObjectMapper mapper;

    @Autowired
    private EntityClassEngine classEngine;

    public ElasticSearchQueryProvider(int fetchSize) {
        this.fetchSize = fetchSize;
    }

    public void setEsDataSource(DataSource esDataSource) {
        this.esDataSource = esDataSource;
    }

    public void setContextService(ContextService contextService) {
        this.contextService = contextService;
    }

    public void setEngine(EntityClassEngine engine) {
        this.engine = engine;
    }

    public void setExecutionConfig(ExecutionConfig executionConfig) {
        this.executionConfig = executionConfig;
    }

    @Override
    public QueryProviderType type() {
        return QueryProviderType.INDEX;
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"query", "es"}
    )
    @Override
    public List<Object> query(String app, IEntityClass entityClass, String profile, RelDataType type,
                              List<RexNode> ops, List<Map.Entry<String, Tuple2<StructKind, Class>>> fields,
                              List<Pair<RexNode, String>> projects,
                              List<Map.Entry<String, RelFieldCollation.Direction>> sort, Long offset, Long fetch,
                              List<String> groupBy, List<AggregateCall> aggs, List<RelHint> hints, RelNode rawTree,
                              DataContext dataContext) {

        contextService.getAll().put("invocation", "index");
        
        if(offset == null) {
            offset = 0L;
        }

        Tuple2<String, String> searchIndex = manageBocpMetadataService.getSearchSegmentIndex(profile, entityClass.code());
        if (searchIndex != null && (searchIndex._2 == null || searchIndex._2.equalsIgnoreCase("$TEMP$"))) {
            return Collections.emptyList();
        }

        SelectConfig selectConfig = ParseSqlNodeUtils.getSelectConfig(profile, type, ops, fields, projects, sort,
                offset, fetch, groupBy, aggs, hints, rawTree, dataContext);

        /**校验SQL是否超过join上限，超过抛出异常**/
        ElasticCustomShuttle elasticCustomShuttle = new ElasticCustomShuttle();
        rawTree.accept(elasticCustomShuttle);
        List<Object> retList = new ArrayList<>();
        if (elasticCustomShuttle.getJoinsCounter()) {
            //do something
            return Collections.emptyList();
        } else {
            ElasticsearchTable searchTable = getSearchTable(app, entityClass.code());
            ElasticsearchRel.Implementor implementor = new ElasticsearchRel.Implementor();
            implementor.elasticsearchTable = searchTable;
            rawTree.accept(new ToElasticsearchQueryShuttle(implementor));

            
            /**
             * final Expression fields = block.append("fields",
             *                 constantArrayList(
             *                         Pair.zip(ElasticsearchRules.elasticsearchFieldNames(rowType),
             *                                 new AbstractList<Class>() {
             *                                     @Override
             *                                     public Class get(int index) {
             *                                         return physType.fieldClass(index);
             *                                     }
             *
             *                                     @Override
             *                                     public int size() {
             *                                         return rowType.getFieldCount();
             *                                     }
             *                                 }),
             *                         Pair.class));
             *  final Expression ops = block.append("ops", Expressions.constant(implementor.list));
             *         final Expression sort = block.append("sort", constantArrayList(implementor.sort, Pair.class));
             *         final Expression groupBy = block.append("groupBy", Expressions.constant(implementor.groupBy));
             *         final Expression aggregations = block.append("aggregations",
             *                 constantArrayList(implementor.aggregations, Pair.class));
             *
             *         final Expression mappings = block.append("mappings",
             *                 Expressions.constant(implementor.expressionItemMap));
             *
             *         final Expression offset = block.append("offset", Expressions.constant(implementor.offset));
             *         final Expression fetch = block.append("fetch", Expressions.constant(implementor.fetch));
             *         final Expression tree = block.append("tree", relImplementor.stash(getInput(), RelNode.class));
             *         final Expression dataContext = block.append("dataCtx", DataContext.ROOT);
             *         Expression enumerable = block.append("enumerable",
             *                 Expressions.call(table, ElasticsearchMethod.ELASTICSEARCH_QUERYABLE_FIND.method, ops,
             *                         fields, sort, groupBy, aggregations, mappings, offset, fetch, tree, dataContext));
             *         block.add(Expressions.return_(null, enumerable));
             *         Result result = relImplementor.result(physType, block.toBlock());
             *         System.out.println("Converter Table " + (System.currentTimeMillis() - start));
             */
            /**
             * List<String> ops,
             * List<Map.Entry<String, Class>> fields,
             * List<Map.Entry<String, RelFieldCollation.Direction>> sort,
             * List<String> groupBy,
             * List<Map.Entry<String, String>> aggregations,
             * Map<String, String> mappings,
             * Long offset, Long fetch, RelNode rawTree, DataContext dataContext
             */

            try {
                List<String> targetFields = new ArrayList<>(implementor.list);
                if(projects.isEmpty()) {
                    //append all current fields
                    Collection<IEntityField> allSelfFields = entityClass.fields();
                    String findString = allSelfFields.stream().map(x -> {
                                return "\"".concat(x.name().toLowerCase()).concat("\"");
                            }).collect(Collectors.joining(","));
                    targetFields.add("{\"fields\" : [".concat(findString).concat("]}"));
                } 
                
                
                //add create_time exists
                StringWriter writer = new StringWriter();
                JsonGenerator generator = mapper.getFactory().createGenerator(writer);
                QueryBuilders.constantScoreQuery(QueryBuilders.boolQuery().must(QueryBuilders.existsQuery("create_time")))
                        .writeJson(generator);
                generator.flush();
                generator.close();

                targetFields.add("{\"query\":".concat(writer.toString()).concat("}"));
                
                Enumerable<Object> objects = searchTable.find(targetFields,
                        fields.stream().map(x -> Pair.of(x.getKey().toLowerCase(), x.getValue()._2)).collect(Collectors.toList())
                        , implementor.sort, implementor.groupBy, implementor.aggregations, implementor.expressionItemMap, offset, implementor.fetch, rawTree, dataContext);
                Iterator<Object> iterator = objects.iterator();
               
                while(iterator.hasNext()) {
                    List<Object> row = new ArrayList<>();
                    Object[] next = ( Object[])iterator.next();
                   
//                    fields.forEach(f -> {
//                        String key = f.getKey();
//                        if (key.contains(".")) {
//                            key = f.getKey().replace(".", "_");
//                        }
////                        Object rs = MasterStorageHelper.getRs(iterator.next(), key, f.getValue()._2);
////                        row.add();
//                       
//                    });
                    if (next.length > 1) {
                        retList.add(next);
                    } else {
                        retList.add(next[0]);
                    }
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
//        ElasticSearchSqlConverter elasticSearchSqlConverter = new ElasticSearchSqlConverter(engine, contextService, elasticCustomShuttle);
//        SqlNode sqlNode = elasticSearchSqlConverter.oqsRelNodeConverterElasticSql(entityClass, selectConfig);
//        //TODO
//        contextService.getAll().put("hasCount", hasCount(selectConfig));
        //create query

        return retList;
    }

    private void genSort(ElasticsearchRel.Implementor implementor, OqsengineSort relNode) {
        final List<RelDataTypeField> fields = relNode.getRowType().getFieldList();

        for (RelFieldCollation fieldCollation : relNode.collation.getFieldCollations()) {
            final String name = fields.get(fieldCollation.getFieldIndex()).getName();
            final String rawName = implementor.expressionItemMap.getOrDefault(name, name);
            implementor.addSort(rawName, fieldCollation.getDirection());
        }
        if (relNode.offset != null) {
            implementor.offset(((RexLiteral) relNode.offset).getValueAs(Long.class));
        }

        if (relNode.fetch != null) {
            implementor.fetch(((RexLiteral) relNode.fetch).getValueAs(Long.class));
        }
    }

    private void genFilter(ElasticsearchRel.Implementor implementor, OqsengineFilter relNode) {
        ObjectMapper mapper = implementor.elasticsearchTable.mapper;
        ElasticsearchFilter.PredicateAnalyzerTranslator translator = new ElasticsearchFilter.PredicateAnalyzerTranslator(mapper, relNode.getRowType());
        try {
            implementor.add(translator.translateMatch(relNode.getCondition()));
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        } catch (PredicateAnalyzer.ExpressionNotAnalyzableException e) {
            throw new RuntimeException(e);
        }
    }

    private void genProject(ElasticsearchRel.Implementor implementor, OqsengineProject oqsengineProject) {
        final List<String> inFields = ElasticsearchRules.elasticsearchFieldNames(oqsengineProject.getInput().getRowType());
        final ElasticsearchRules.RexToElasticsearchTranslator translator =
                new ElasticsearchRules.RexToElasticsearchTranslator(
                        new JavaTypeFactoryImpl(), inFields);

        final List<String> fields = new ArrayList<>();
        final List<String> scriptFields = new ArrayList<>();
        // registers wherever "select *" is present
        boolean hasSelectStar = false;
        for (Pair<RexNode, String> pair : oqsengineProject.getNamedProjects()) {
            final String name = pair.right;
            final String expr = pair.left.accept(translator);

            // "select *" present?
            hasSelectStar |= ElasticsearchConstants.isSelectAll(name);

            //add more function
            if (ElasticsearchRules.isItem(pair.left)) {
                implementor.addExpressionItemMapping(name, expr);
                fields.add(expr);
            } else if (ElasticsearchRules.isExpr(pair.left)) {
                implementor.addExpressionItemMapping(name, expr);
                //ignore the field is ok?
            } else if (expr.equalsIgnoreCase(name)) {
                fields.add(name.toLowerCase());
            } else if (expr.matches("\"literal\":.+")) {
                scriptFields.add(ElasticsearchRules.quote(name)
                        + ":{\"script\": "
                        + expr.split(":")[1] + "}");
            } else {
                //int indexOf = scriptedField.indexOf(".");
                //TODO use one
                scriptFields.add(ElasticsearchRules.quote(name)
                        + ":{\"script\":"
                        // _source (ES2) vs params._source (ES5)
                        + "\"" + implementor.elasticsearchTable.scriptedFieldPrefix() + "['"
                        + expr + "']\"}");
            }
        }

        if (hasSelectStar) {
            // means select * from elastic
            // this does not yet cover select *, _MAP['foo'], _MAP['bar'][0] from elastic
            return;
        }

        final StringBuilder query = new StringBuilder();
//    if (scriptFields.isEmpty()) {
        List<String> newList = fields.stream()
                // _id field is available implicitly
                .filter(f -> !ElasticsearchConstants.ID.equals(f))
                .map(ElasticsearchRules::quote)
                .collect(Collectors.toList());

        final String findString = String.join(", ", newList);
        query.append("\"fields\" : [").append(findString).append("],");
//    } else {
        // if scripted fields are present, ES ignores _source attribute
//      for (String field : fields) {
//        scriptFields.add(ElasticsearchRules.quote(field) + ":{\"script\": "
//            // _source (ES2) vs params._source (ES5)
//            + "\"" + implementor.elasticsearchTable.scriptedFieldPrefix() + "."
//            + field + "\"}");
//      }
        query.append("\"script_fields\": {" + String.join(", ", scriptFields) + "}");
//    }

        implementor.list.removeIf(l -> l.startsWith("\"_source\""));
        implementor.add("{" + query + "}");
    }

    private void genGroupBy(ElasticsearchRel.Implementor implementor, OqsengineAggregate agg) {
        final List<String> inputFields = fieldNames(agg.getRowType());
        for (int group : agg.getGroupSet()) {
            final String name = inputFields.get(group).replace(".", "_");
            implementor.addGroupBy(implementor.expressionItemMap.getOrDefault(name, name));
        }

        final ObjectMapper mapper = implementor.elasticsearchTable.mapper;

        for (AggregateCall aggCall : agg.getAggCallList()) {
            final List<String> names = new ArrayList<>();
            for (int i : aggCall.getArgList()) {
                names.add(inputFields.get(i));
            }

            final ObjectNode aggregation = mapper.createObjectNode();
            final ObjectNode field = aggregation.with(toElasticAggregate(aggCall));

            final String name = names.isEmpty() ? ElasticsearchConstants.ID : names.get(0);
            field.put("field", implementor.expressionItemMap.getOrDefault(name, name));
            if (aggCall.getAggregation().getKind() == SqlKind.ANY_VALUE) {
                field.put("size", 1);
            }

            implementor.addAggregation(aggCall.getName(), aggregation.toString());
        }
    }

    private static List<String> fieldNames(RelDataType relDataType) {
        List<String> names = new ArrayList<>();

        for (RelDataTypeField rdtf : relDataType.getFieldList()) {
            names.add(rdtf.getName());
        }
        return names;
    }

    private static String toElasticAggregate(AggregateCall call) {
        final SqlKind kind = call.getAggregation().getKind();
        switch (kind) {
            case COUNT:
                // approx_count_distinct() vs count()
                return call.isDistinct() && call.isApproximate() ? "cardinality" : "value_count";
            case SUM:
                return "sum";
            case MIN:
                return "min";
            case MAX:
                return "max";
            case AVG:
                return "avg";
            case ANY_VALUE:
                return "terms";
            default:
                throw new IllegalArgumentException("Unknown aggregation kind " + kind + " for " + call);
        }
    }

    /***
     *检索SelectConfig是否带有 hint show_count标识
     * @param selectConfig
     * @return
     */
    private boolean hasCount(SelectConfig selectConfig) {
        return selectConfig.getHints().stream()
                .anyMatch(x -> x.hintName.equalsIgnoreCase("show_count"));
    }


    private ElasticsearchTable getSearchTable(String appCode, String code) {
        String profile = fetcher.getProfile(contextService.getAll());
        //TODO Transport shutdown
        RestClient client = ((ElasticsearchTransportExecutor) transportExecutor).executor(profile)
                .getLowLevelClient();
        Tuple2<String, String> searchIndex = manageBocpMetadataService.getSearchSegmentIndex(profile, code);

        /**
         * 适配需要路由的segment index
         * ***/
        //TODO refresh
        Tuple2<String, String> key = Tuple.of(code.toLowerCase(Locale.ROOT), searchIndex._2);
        ElasticsearchTransport elasticsearchTransport = transportMap
                .get(key);
        ElasticsearchTransport transport;
        String prefix = manageBocpMetadataService.getIndexPrefix(profile, appCode);


        if (elasticsearchTransport == null) {
            transport = new ElasticsearchTransport(client, mapper,
                    searchIndex, code.toLowerCase(Locale.ROOT), fetchSize);
            transportMap.put(key, transport);
        } else {
            transport = elasticsearchTransport;
        }

        return new ElasticsearchTable(classEngine, transport, contextService, fetcher, prefix);
    }
    
    
    class ToElasticsearchQueryShuttle extends RelShuttleImpl {

        private ElasticsearchRel.Implementor implementor;
        
        public ToElasticsearchQueryShuttle(ElasticsearchRel.Implementor implementor) {
            this.implementor = implementor;
        }
        
        @Override
        public RelNode visit(RelNode other) {
            RelNode relNode = super.visit(other);
            if (relNode instanceof OqsengineAggregate) {
                genGroupBy(implementor, (OqsengineAggregate) relNode);
            } else if (relNode instanceof OqsengineProject) {
                genProject(implementor, (OqsengineProject) relNode);
            } else if (relNode instanceof OqsengineFilter) {
                genFilter(implementor, (OqsengineFilter) relNode);
            } else if (relNode instanceof OqsengineSort) {
                genSort(implementor, (OqsengineSort) relNode);
            }
            return relNode;
        }
    }
}

