package com.xforceplus.ultraman.adapter.elasticsearch.service.impl;

import com.xforceplus.ultraman.adapter.elasticsearch.service.IndexOperation;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.FieldMappingType;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.SettingProperties;
import com.xforceplus.ultraman.adapter.elasticsearch.service.entity.FieldMapping;
import com.xforceplus.ultraman.adapter.elasticsearch.service.utils.ElasticSearchMappingBuildUtils;
import com.xforceplus.ultraman.adapter.elasticsearch.transport.ElasticsearchTransportExecutor;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.sdk.core.datasource.route.TransportExecutor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.indices.*;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
public class ElasticSearchIndexImpl implements IndexOperation {

    private int shards;

    private int replicas;

    private int retryOnConflict;

    private int threadHandleBatch = 1000;

    @Autowired
    private TransportExecutor elasticsearchTransportExecutor;

    public ElasticSearchIndexImpl(int shards, int replicas, int retryOnConflict, int threadHandleBatch) {
        this.shards = shards;
        this.replicas = replicas;
        this.retryOnConflict = retryOnConflict;
        this.threadHandleBatch = threadHandleBatch;
    }

    public static void buildSetting(CreateIndexRequest request, int number_of_shards,
                                    int number_of_replicas) {
        if (number_of_shards == 0 && number_of_replicas == 0) {
            /**当前端没有传递索引分片参数时，使用内置常量参数**/
            number_of_replicas = SettingProperties.NUMBER_OF_REPLICAS;
            number_of_shards = SettingProperties.NUMBER_OF_SHARDS;
        }
        request.settings(Settings.builder()
                .put("index.number_of_shards", number_of_shards)
                .put("index.number_of_replicas", number_of_replicas).
                put("index.max_script_fields", 1000)
                .put("index.max_docvalue_fields_search", 1000));
    }

    @Override
    public boolean createIndexAndCreateMapping(String idxName, Collection<IEntityField> allFields, String profile) {
        try {
            if (!indexExists(idxName, profile)) {
                CreateIndexRequest request = new CreateIndexRequest(idxName);
                buildSetting(request, shards, replicas);
                request.mapping(getxContentBuilder(allFields, null));
                RestHighLevelClient executor = ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile);
                CreateIndexResponse res = executor.indices().create(request, RequestOptions.DEFAULT);
                return res.isAcknowledged();
            }
            return true;
        } catch (Exception ex) {
            log.error(ex.getMessage());
            throw new RuntimeException(ex.getCause());
        }
    }

    @Override
    public boolean indexExists(String idxName, String profile) {
        try {
            GetIndexRequest request = new GetIndexRequest(idxName.toLowerCase(Locale.ROOT));
            boolean exists =
                    ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile)
                            .indices()
                            .exists(request, RequestOptions.DEFAULT);
            return exists;
        } catch (Exception ex) {
            log.error(ex.getMessage());
            throw new RuntimeException(ex.getCause());
        }
    }

    @Override
    public boolean indexExists(String profile, String... idxNames) {
        try {
            String[] lowercaseIndices = new String[idxNames.length];
            int index = 0;
            for (String idxName : idxNames) {
                lowercaseIndices[index++] = idxName.toLowerCase(Locale.ROOT);
            }
            GetIndexRequest request = new GetIndexRequest(lowercaseIndices);
            boolean exists =
                    ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile)
                            .indices()
                            .exists(request, RequestOptions.DEFAULT);
            return exists;
        } catch (Exception ex) {
            log.error(ex.getMessage());
            throw new RuntimeException(ex.getCause());
        }
    }

    @Override
    public Set<String> getAllIndex(String profile) {
        try {
            GetIndexRequest request = new GetIndexRequest("*");
            GetIndexResponse response =    ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile)
                    .indices().get(request, RequestOptions.DEFAULT);
            String[] indices = response.getIndices();
            return new HashSet<>(Arrays.asList(indices));
        } catch (Exception ex) {
            log.error(ex.getMessage());
            throw new RuntimeException(ex.getCause());
        }
    }

    @Override
    public boolean joinMapping(String indexName, Set<String> relations, String entityClassCode, String profile) {
        try {
            PutMappingRequest putMappingRequest = new PutMappingRequest(indexName);
            AcknowledgedResponse acknowledgedResponse =
                    ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile)
                            .indices().putMapping(
                                    putMappingRequest.source(getxContentBuilder(relations, entityClassCode)),
                                    RequestOptions.DEFAULT);
            return acknowledgedResponse.isAcknowledged();
        } catch (Exception ex) {
            log.error(ex.getMessage());
            throw new RuntimeException(ex.getCause());
        }
    }

    private XContentBuilder getxContentBuilder(Collection<IEntityField> allFields, String prefixName)
            throws IOException {
        XContentBuilder properties = XContentFactory.jsonBuilder().startObject()
                .field("dynamic", "true") //注释掉这条命令，不能动态扩展字段
                .field("properties").startObject();
        getxContentBuilder(allFields, properties, prefixName);
        properties.endObject();
        properties.endObject();
        return properties;
    }

    private XContentBuilder getxContentBuilder(Set<String> relations, String entityClassCode)
            throws IOException {
        XContentBuilder properties = XContentFactory.jsonBuilder().startObject()
                .field("dynamic", "true") //注释掉这条命令，不能动态扩展字段
                .field("properties").startObject();

        properties = properties.startObject(entityClassCode.concat("_join")).field("type", "join")
                .startObject("relations");
        if (relations != null && relations.size() > 0) {
            properties = properties.field(entityClassCode, relations.toArray());
        }
        properties.endObject();
        properties.endObject();
        properties.endObject();
        properties.endObject();
        return properties;
    }

    private XContentBuilder getxContentBuilder(Collection<IEntityField> allFields,
                                               XContentBuilder properties, String prefixName) throws IOException {
        for (FieldMapping field : ElasticSearchMappingBuildUtils.getFieldInfos(allFields)) {
            String fieldName = field.getField().contains(".") ? field.getField().replace(".", "_")
                    : field.getField();
            fieldName = StringUtils.isNotEmpty(prefixName) ? prefixName.concat(".").concat(fieldName) : fieldName;
//            fieldName = fieldName.toLowerCase();
            if (StringUtils.equalsIgnoreCase(field.getType(), FieldMappingType.TEXT.getType())) {
                properties = properties.startObject(fieldName)
                        .field("type", FieldMappingType.TEXT.getType())
                        .field("analyzer", field.getAnalyzer()).field("fielddata", "true").endObject();
            } else if (StringUtils.equalsIgnoreCase(field.getType(),
                    FieldMappingType.KEYWORD.getType())) {
                properties = properties.startObject(fieldName)
                        .field("type", FieldMappingType.KEYWORD.getType()).endObject();
                if (field.getOriginField().type() == FieldType.STRINGS) {
                    //should add a new field for store raw value
                    properties = properties.startObject(fieldName.concat("@raw"))
                            .field("type", FieldMappingType.TEXT.getType()).endObject();
                }
            } else if (StringUtils.equalsIgnoreCase(field.getType(), FieldMappingType.LONG.getType())) {
                properties = properties.startObject(fieldName)
                        .field("type", FieldMappingType.LONG.getType()).endObject();
            } else if (StringUtils.equalsIgnoreCase(field.getType(), FieldMappingType.DOUBLE.getType())) {
                properties = properties.startObject(fieldName)
                        .field("type", FieldMappingType.DOUBLE.getType()).endObject();
            } else if (StringUtils.equalsIgnoreCase(field.getType(),
                    FieldMappingType.BOOLEAN.getType())) {
                properties = properties.startObject(fieldName)
                        .field("type", FieldMappingType.BOOLEAN.getType()).endObject();
            }
        }
        return properties;
    }

    public boolean deleteIndex(String idxName, String profile) {
        try {
            if (!indexExists(idxName, profile)) {
                log.error(" idxName={} 不存在", idxName);
                return true;
            }
            AcknowledgedResponse delete =
                    ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile)
                            .indices()
                            .delete(new DeleteIndexRequest(idxName), RequestOptions.DEFAULT);
            return delete.isAcknowledged();
        } catch (Exception ex) {
            log.error(ex.getMessage());
            throw new RuntimeException(ex);
        }
    }

    /**
     * 支持索引新增字段
     *
     * @param idxName
     * @param appendFields
     * @param profile
     * @return
     */
    public boolean putMapping(String idxName, Collection<IEntityField> appendFields, String prefixName, String profile) {
        try {
            Map<String, String> mappings = getMappings(idxName, profile);
            List<IEntityField> filterFields = appendFields.stream().filter(fields -> {
                if (mappings.get(fields.name().toLowerCase(Locale.ROOT)) == null) {
                    return true;
                }
                return false;
            }).collect(Collectors.toList());
            PutMappingRequest putMappingRequest = new PutMappingRequest(idxName);
            AcknowledgedResponse acknowledgedResponse =
                    ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile)
                            .indices().putMapping(
                                    putMappingRequest.source(getxContentBuilder(filterFields, prefixName)),
                                    RequestOptions.DEFAULT);
            return acknowledgedResponse.isAcknowledged();
        } catch (IOException ex) {
            log.error(ex.getMessage());
            throw new RuntimeException(ex.getCause());
        }
    }

    public Map<String, String> getMappings(String idxName, String profile) throws IOException {
        Map<String, String> fieldTypes = new HashMap<>();
        GetMappingsRequest getMappings = new GetMappingsRequest().indices(idxName);
        GetMappingsResponse getMappingResponse =
                ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile)
                        .indices().getMapping(getMappings, RequestOptions.DEFAULT);
        Map<String, MappingMetadata> mappings = getMappingResponse.mappings();
        Map<String, Object> indexFields = mappings.get(idxName).sourceAsMap();
        /**
         * in case multi properties will appear
         */
        indexFields.entrySet().stream().filter(x -> "properties".equalsIgnoreCase(x.getKey()))
                .forEach(entry -> {
                    Map<String, Map> fieldProperties = (Map<String, Map>) entry.getValue();
                    fieldProperties.entrySet().stream()
                            .forEach(
                                    field -> fieldTypes.put(field.getKey().toLowerCase(Locale.ROOT), (String) field.getValue().get("type")));

                });
        return fieldTypes;
    }
}
