package com.xforceplus.ultraman.adapter.elasticsearch.service.impl;

import static com.xforceplus.ultraman.adapter.elasticsearch.service.constant.SegmentIndexRule.QUARTER;
import static com.xforceplus.ultraman.adapter.elasticsearch.service.constant.SegmentIndexRule.YEAR;

import akka.stream.ActorMaterializer;
import akka.stream.javadsl.Source;
import com.google.common.collect.Sets;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.adapter.elasticsearch.CustomElasticSearchTransport;
import com.xforceplus.ultraman.adapter.elasticsearch.query.dto.ElasticTenantProfile;
import com.xforceplus.ultraman.adapter.elasticsearch.query.po.BocpElasticConfigPo;
import com.xforceplus.ultraman.adapter.elasticsearch.service.IndexOperation;
import com.xforceplus.ultraman.adapter.elasticsearch.service.ManageBocpMetadataService;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.CommonProperty;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.SegmentFieldType;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.SegmentIndexRule;
import com.xforceplus.ultraman.adapter.elasticsearch.service.utils.BocpMetabaseCacheUtils;
import com.xforceplus.ultraman.adapter.elasticsearch.utils.DynamicConfigUtils;
import com.xforceplus.ultraman.cdc.utils.TimeWaitUtils;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.sdk.core.datasource.route.dynamic.config.DynamicConfig;
import io.vavr.Tuple;
import io.vavr.Tuple2;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.SpringApplication;
import org.springframework.context.ApplicationContext;

/**
 * @ClassName ManageBocpMetabaseServiceImpl
 * @description:
 * @author: WanYi
 * @create: 2023-08-14 17:15
 * @Version 1.0
 **/
@Slf4j
public class ManageBocpMetadataServiceImpl implements ManageBocpMetadataService, InitializingBean {

    @Autowired
    private IndexOperation engineAdapterService;

    @Autowired
    private DynamicConfig dynamicConfig;
    @Resource
    private ApplicationContext applicationContext;
    @Autowired
    private CustomElasticSearchTransport customElasticSearchTransport;

    @Autowired
    private IndexOperation indexOperation;
    
/*  @Autowired
  private EntityClassRelationService entityClassRelationService;*/

    @Autowired
    private EntityClassEngine entityClassEngine;

    @Autowired
    private ActorMaterializer mat;

    private static int TICK = 0;

    private volatile boolean loadFinish = false;

    public void setEngineAdapterService(IndexOperation engineAdapterService) {
        this.engineAdapterService = engineAdapterService;
        if (engineAdapterService instanceof ElasticSearchServiceImpl) {
            ((ElasticSearchServiceImpl) engineAdapterService).setManageBocpMetadataService(this);
        }
    }

    public void setDynamicConfig(DynamicConfig dynamicConfig) {
        this.dynamicConfig = dynamicConfig;
    }

    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    public void setCustomElasticSearchTransport(CustomElasticSearchTransport customElasticSearchTransport) {
        this.customElasticSearchTransport = customElasticSearchTransport;
    }

/*
  public void setEntityClassRelationService(EntityClassRelationService entityClassRelationService) {
    this.entityClassRelationService = entityClassRelationService;
  }
*/

    public void setEntityClassEngine(EntityClassEngine entityClassEngine) {
        this.entityClassEngine = entityClassEngine;
    }

    /**
     * 更新bocp元数据缓存信息
     *
     * @param elasticTenantProfiles
     * @param iEntityClass
     * @param errors
     **/
    @Override
    public void updateMetadataCache(Collection<ElasticTenantProfile> elasticTenantProfiles, IEntityClass iEntityClass,
                                    List<Tuple2<String, String>> errors) {
        try {
            if (elasticTenantProfiles.isEmpty()) {
                String profile = CommonProperty.defaultProfile;
                String indexName = getIndexName(profile, iEntityClass);
                createOrUpdateIndexMapping(errors, iEntityClass, profile, indexName);
                BocpElasticConfigPo bocpElasticConfigPo = BocpElasticConfigPo.builder()
                        .enableSegment(false)
                        .enableSync(true)
                        .entityClassCode(iEntityClass.code())
                        .entityClassId(iEntityClass.id())
                        .tenantCode(profile)
                        .build();
                Map<String, String> currentSegmentIndexs = new HashMap<>();
                currentSegmentIndexs.put(indexName, indexName);
                updateMetadataCache(iEntityClass, profile, currentSegmentIndexs, bocpElasticConfigPo);
            } else {
                elasticTenantProfiles.stream().forEach(tenantProfile -> {
                    if (!StringUtils.isEmpty(tenantProfile.getTenantCode())) {
                        String profile = tenantProfile.getTenantCode();
                        String indexName = genBaseIndexName(profile, iEntityClass);
                        SegmentIndexRule segmentIndexRule = null;
                        Map<String, String> currentSegmentIndexs = new HashMap<>();
                        if (tenantProfile.isEnableSegment()) {
                            segmentIndexRule = getSegmentIndexRule(tenantProfile.getSegmentRule().toLowerCase(Locale.ROOT));
                            String expression = getSegmentMatch(indexName, segmentIndexRule);
                            Set<String> matchIndexs = customElasticSearchTransport.getMatchIndexs(indexName, tenantProfile.getTenantCode(), errors);

                            for (String matchIndex : matchIndexs) {
                                Pattern pattern = Pattern.compile(expression, Pattern.CASE_INSENSITIVE);
                                Matcher matcher = pattern.matcher(matchIndex);
                                if (matcher.matches()) {
                                    createOrUpdateIndexMapping(errors, iEntityClass, profile, matchIndex);
                                    currentSegmentIndexs.put(matcher.group(1), matchIndex);
                                }
                            }
                        } else {
                            indexName = getIndexName(profile, iEntityClass);
                            createOrUpdateIndexMapping(errors, iEntityClass, profile, indexName);
                            currentSegmentIndexs.put(indexName, indexName);
                        }

                        BocpElasticConfigPo bocpElasticConfigPo = BocpElasticConfigPo.builder()
                                .enableSegment(tenantProfile.isEnableSegment())
                                .enableSync(true)
                                .segmentFieldName(tenantProfile.getSegmentField())
                                .entityClassCode(iEntityClass.code())
                                .entityClassId(iEntityClass.id())
                                .segmentDateFormat(tenantProfile.getSegmentDateFormat())
                                .segmentRule(segmentIndexRule)
                                .segmentFieldType(getSegmentFieldType(tenantProfile.getSegmentFieldType(), tenantProfile.getSegmentDateFormat()))
                                .tenantCode(profile)
                                .build();
                        updateMetadataCache(iEntityClass, profile, currentSegmentIndexs, bocpElasticConfigPo);
                    }
                });

            }
        } catch (Throwable e) {
            errors.add(new Tuple2<>(String.valueOf(iEntityClass.id()), e.getMessage()));
        }
    }

    private String genBaseIndexName(String tenantProfile, IEntityClass iEntityClass) {
        if (StringUtils.isEmpty(tenantProfile) || "default".equalsIgnoreCase(tenantProfile)) {
            return iEntityClass.ref().getAppCode().concat("_").concat(iEntityClass.code()).toLowerCase();
        } else {
            return tenantProfile.concat("_").concat(iEntityClass.ref().getAppCode()).concat("_").concat(iEntityClass.code()).toLowerCase();
        }
    }


    /**
     * @param iEntityClass
     * @param profile
     * @param currentSegmentIndexs
     * @param bocpElasticConfigPo
     **/
    private void updateMetadataCache(IEntityClass iEntityClass, String profile,
                                     Map<String, String> currentSegmentIndexs, BocpElasticConfigPo bocpElasticConfigPo) {
        if (BocpMetabaseCacheUtils.getExactlyIndexMapping(profile) != null) {
            BocpMetabaseCacheUtils.getExactlyIndexMapping(profile).put(iEntityClass.id(), currentSegmentIndexs);
        } else {
            Map<Long, Map<String, String>> segmentIndex = new HashMap<>();
            segmentIndex.put(iEntityClass.id(), currentSegmentIndexs);
            BocpMetabaseCacheUtils.putIndexMapping(profile, segmentIndex);
        }
        if (BocpMetabaseCacheUtils.getExactlyBocpConfig(profile) != null) {
            BocpMetabaseCacheUtils.getExactlyBocpConfig(profile).put(iEntityClass.id(), bocpElasticConfigPo);
        } else {
            Map<Long, BocpElasticConfigPo> segmentBocpConfig = new HashMap<>();
            segmentBocpConfig.put(iEntityClass.id(), bocpElasticConfigPo);
            BocpMetabaseCacheUtils.putBocpConfig(profile, segmentBocpConfig);
        }
    }


    /**
     * get tenant index name from config
     *
     * @param iEntityClass
     * @param tenantProfile
     * @return
     **/
    @NotNull
    private String getIndexName(String tenantProfile, IEntityClass iEntityClass) {
        String dynamicAwareIndexName = DynamicConfigUtils.insulateTenant(dynamicConfig, tenantProfile, iEntityClass.code(),
                iEntityClass.ref().getAppCode());

        if (dynamicAwareIndexName != null && dynamicAwareIndexName.toLowerCase().contains(tenantProfile.toLowerCase())) {
            return dynamicAwareIndexName;
        }

        //if the tenant is not configured as a separate datasource, find the name from elasticsearch config
        Map<Long, BocpElasticConfigPo> exactlyBocpConfig = BocpMetabaseCacheUtils.getExactlyBocpConfig(tenantProfile);
        if (exactlyBocpConfig != null) {
            BocpElasticConfigPo bocpElasticConfigPo = exactlyBocpConfig.get(iEntityClass.id());
            if (bocpElasticConfigPo != null) {
                if (StringUtils.isEmpty(tenantProfile) || "default".equalsIgnoreCase(tenantProfile)) {
                    return iEntityClass.ref().getAppCode().concat("_").concat(iEntityClass.code()).toLowerCase();
                } else {
                    return tenantProfile.concat("_").concat(iEntityClass.ref().getAppCode()).concat("_").concat(iEntityClass.code()).toLowerCase();
                }
            }
        }
        //else
        return dynamicAwareIndexName;
    }

    /**
     * 根据拆分索引规则,生成正则表达式 过滤特定的indexName
     *
     * @param indexName
     * @param segmentIndexRule
     **/
    @Nullable
    private String getSegmentMatch(String indexName, SegmentIndexRule segmentIndexRule) {
        String expression;
        /**根据正则表达式匹配当前拆分索引的集合**/
        switch (segmentIndexRule) {
            case QUARTER:
                expression = "^".concat(indexName).concat(String.format("_(\\d{4}_\\d{1}_%s)$", QUARTER.getSegmentRuleName()));
                break;
            case YEAR:
            default:
                expression = "^".concat(indexName).concat(String.format("_(\\d{4}_%s)$", YEAR.getSegmentRuleName()));
                break;
        }
        return expression;
    }

    /**
     * elastic index索引信息上报时，根据bocp配置 返回多个对应的segment index索引列表
     *
     * @param iEntityClass
     * @param profile
     * @return
     **/
    @Override
    public List<String> getEntityClassMappingIndexs(String profile, IEntityClass iEntityClass) {
        List<String> indexs = new ArrayList<>();
        profile = getProfile(profile);
        if (BocpMetabaseCacheUtils.getBocpConfig(profile) != null && BocpMetabaseCacheUtils.getBocpConfig(profile).get(iEntityClass.id()) != null) {
            if (BocpMetabaseCacheUtils.getIndexMapping(profile) != null && BocpMetabaseCacheUtils.getIndexMapping(profile).get(iEntityClass.id()) != null) {
                Map<String, String> segmentIndexs = BocpMetabaseCacheUtils.getIndexMapping(profile).get(iEntityClass.id());
                indexs = new ArrayList<>(segmentIndexs.values());
            }
        } else {
            indexs.add(getIndexName(profile, iEntityClass));
        }
        return indexs;
    }


    /**
     * 检查entityClass binlog记录否在在bocp配置项，不在跳过不同步至elasticsearch index库中
     *
     * @param iEntityClass
     * @param profile
     **/
    @Override
    public boolean checkPassage(String profile, IEntityClass iEntityClass) {
        loadFinish();
        String targetProfile = getProfile(profile);
        Map<Long, BocpElasticConfigPo> bocpConfig = BocpMetabaseCacheUtils.getBocpConfig(targetProfile);
        if (bocpConfig != null) {
            BocpElasticConfigPo targetConfig = bocpConfig.get(iEntityClass.id());
            if (targetConfig == null && !"default".equalsIgnoreCase(targetProfile)) {
                Map<Long, BocpElasticConfigPo> defaultConfig = BocpMetabaseCacheUtils.getBocpConfig("default");
                if (defaultConfig != null) {
                    targetConfig = defaultConfig.get(iEntityClass.id());
                }
            }

            return targetConfig != null;
        }
        return false;
    }

    /**
     * 等待bocp配置解析同步至缓存完成
     **/
    private boolean loadFinish() {
        while (!loadFinish) {
            TimeWaitUtils.wakeupAfter(CommonProperty.RECONNECT_WAIT_IN_SECONDS, TimeUnit.MILLISECONDS);
        }
        return loadFinish;
    }

    /**
     * @param segmentFiledType
     * @return
     **/
    private SegmentFieldType getSegmentFieldType(String segmentFiledType, String dateFormat) {
        SegmentFieldType fieldType;

        if (StringUtils.isEmpty(segmentFiledType)) {
            return null;
        }
        if (StringUtils.isEmpty(segmentFiledType) && StringUtils.isNotEmpty(dateFormat)) {
            fieldType = SegmentFieldType.STRING;
        } else {
            switch (segmentFiledType) {
                case "string":
                    fieldType = SegmentFieldType.STRING;
                    break;
                case "bigint":
                default:
                    fieldType = SegmentFieldType.BIGINT;
                    break;
            }
        }

        return fieldType;
    }

    /**
     * @param segmentRule
     * @return
     **/
    private SegmentIndexRule getSegmentIndexRule(String segmentRule) {
        SegmentIndexRule indexRule;
        switch (segmentRule) {
            case "quarter":
                indexRule = QUARTER;
                break;
            case "year":
            default:
                indexRule = YEAR;
                break;
        }
        return indexRule;
    }

    /**
     * 解析 iEntityClass 的relation关联关系,对索引进行字段冗余
     *
     * @param profile
     * @param indexName
     * @param iEntityClass
     **/
    @Override
    public void buildRedundantFields(String profile, IEntityClass iEntityClass, String indexName, List<Tuple2<String, String>> errors) {
        Set<String> relations = new HashSet<>();
        iEntityClass.relations().forEach(iRelation -> {
            IEntityClass relationEntityClass = entityClassEngine.load(String.valueOf(iRelation.getEntityClassId()), profile).get();
            if (!StringUtils.equalsIgnoreCase(iEntityClass.code(), relationEntityClass.code())) {
                String prefixName = relationEntityClass.code();
                Collection<IEntityField> appendFields = entityClassEngine.describe(relationEntityClass, profile).getAllFields();
                if (engineAdapterService.indexExists(indexName, profile)) {
                    logInfo(engineAdapterService.putMapping(indexName, appendFields, prefixName, profile), iEntityClass, errors, "related code error");
                    relations.add(relationEntityClass.code());
                }
            }
        });
        if (relations.size() > 0 || relations.size() > 0) {
            logInfo(engineAdapterService.joinMapping(indexName, relations, iEntityClass.code(), profile), iEntityClass, errors,
                    "relations is not well build");
        }
    }

    /**
     * 不存在的索引表进行创建与存在新增字段进行mapping, 根据元数据的relation mapping parent-child join对应关系
     *
     * @param iEntityClass
     * @param errors
     * @param profile
     **/
    @Override
    public void createOrUpdateIndexMapping(List<Tuple2<String, String>> errors, IEntityClass iEntityClass, String profile, String indexName) {
        //String indexName = getIndexName(tenantCode, iEntityClass);
        try {
            Collection<IEntityField> allFields = entityClassEngine.describe(iEntityClass, profile).getAllFields();
            boolean indexAndCreateMapping;
            if (engineAdapterService.indexExists(indexName, profile)) {
                indexAndCreateMapping = engineAdapterService.putMapping(indexName, allFields, null, profile);
            } else {
                indexAndCreateMapping = engineAdapterService.createIndexAndCreateMapping(indexName, allFields, profile);
            }
            if (!indexAndCreateMapping) {
                errors.add(new Tuple2<>(iEntityClass.code(),
                        String.format("elasticsearch execute createOrUpdateIndexMapping method exception,index:%s", indexName)));
            }
            buildRedundantFields(profile, iEntityClass, indexName, errors);
        } catch (Throwable e) {
            logInfo(false, iEntityClass, errors, e.getMessage());
        }
    }

    /**
     * CDC同步时，根据bocp配置信息，路由segment index
     *
     * @param iEntityClass
     * @param profile
     **/
    @Override
    public String getWriteSegmentIndex(String profile, IEntityClass iEntityClass, Map<String, Object> attribute) {
        loadFinish();
        try {
            profile = getProfile(profile);
            if (BocpMetabaseCacheUtils.getBocpConfig(profile) != null) {
                //profile related
                BocpElasticConfigPo bocpElasticConfigPo = BocpMetabaseCacheUtils.getBocpConfig(profile).get(iEntityClass.id());
                if (bocpElasticConfigPo == null && !"default".equalsIgnoreCase(profile)) {
                    Map<Long, BocpElasticConfigPo> targetConfig = BocpMetabaseCacheUtils.getBocpConfig("default");
                    if (targetConfig != null) {
                        bocpElasticConfigPo = targetConfig.get(iEntityClass.id());
                    }
                }

                Map<String, String> existeSegmentIndexs = Optional.ofNullable(getSegmentIndexMapping(profile, iEntityClass)).orElseGet(HashMap::new);
                if (bocpElasticConfigPo.isEnableSegment()) {
                    SegmentIndexRule segmentRule = bocpElasticConfigPo.getSegmentRule();
                    Object segmentFiledValue = attribute.get(bocpElasticConfigPo.getSegmentFieldName());
                    SegmentFieldType segmentFieldType = bocpElasticConfigPo.getSegmentFieldType();
                    Calendar calendar = Calendar.getInstance();
                    calendar.setTime(dateFormat(bocpElasticConfigPo, segmentFiledValue, segmentFieldType));
                    String suffixDataFormat = indexSuffixMatch(segmentRule, calendar);
                    if (existeSegmentIndexs.keySet().contains(suffixDataFormat)) {
                        return existeSegmentIndexs.get(suffixDataFormat);
                    } else {
                        return createSegmentIndex(profile, iEntityClass, existeSegmentIndexs, suffixDataFormat);
                    }
                } else if (!bocpElasticConfigPo.isEnableSegment() && existeSegmentIndexs.size() == 1) {
                    return existeSegmentIndexs.values().stream().findFirst().get();
                } else {
                    createSegmentIndex(profile, iEntityClass, existeSegmentIndexs, null);
                }
            }
            return getIndexName(profile, iEntityClass);
        } catch (Exception e) {
            log.info(e.getMessage());
            throw e;
        }

    }


    /**
     * 创建elastic 分拆索引
     *
     * @param profile
     * @param iEntityClass
     * @param existeSegmentIndexs
     * @param suffixDataFormat
     * @return
     **/
    @NotNull
    private String createSegmentIndex(String profile, IEntityClass iEntityClass, Map<String, String> existeSegmentIndexs, String suffixDataFormat) {
        List<Tuple2<String, String>> errors = new ArrayList<>();
        /**当缓不存在segment index**/
        String indexName = getIndexName(profile, iEntityClass);
        if (StringUtils.isNotEmpty(suffixDataFormat)) {
            indexName = indexName.concat("_").concat(suffixDataFormat);
        }
        createOrUpdateIndexMapping(errors, iEntityClass, profile, indexName);
        if (errors.size() == 0) {
            if (StringUtils.isEmpty(suffixDataFormat)) {
                suffixDataFormat = indexName;
            }
            /**segment index创建成功后，更新bocp元数据缓存**/
            existeSegmentIndexs.put(suffixDataFormat, indexName);
            Map<Long, Map<String, String>> indexMapping = BocpMetabaseCacheUtils.getIndexMapping(profile);
            if (indexMapping == null) {
                indexMapping = new HashMap<>();
                BocpMetabaseCacheUtils.putIndexMapping(profile, indexMapping);
            }
            Map<String, String> cacheIndexMapping = indexMapping.get(iEntityClass.id());
            if (cacheIndexMapping != null) {
                existeSegmentIndexs.entrySet().forEach(entry -> cacheIndexMapping.put(entry.getKey(), entry.getValue()));
            } else {
                indexMapping.put(iEntityClass.id(), existeSegmentIndexs);
            }
            return indexName;
        } else {
            String printErrors = errorsBuild(errors).toString();
            log.error("FAILURE:cdc sync elstic execute getSegmentIndex mehtod failed,cause by:", printErrors);
            throw new RuntimeException(printErrors);
        }
    }

    /**
     * 根据bocp segment 拆分配置 生成索引名称拆分后缀
     *
     * @param segmentRule
     * @param calendar
     **/
    @NotNull
    private String indexSuffixMatch(SegmentIndexRule segmentRule, Calendar calendar) {
        String suffixDataFormat;
        switch (segmentRule) {
            case QUARTER:
                suffixDataFormat =
                        String.valueOf(calendar.get(Calendar.YEAR)).concat("_").concat(String.valueOf((calendar.get(Calendar.MONTH) + 2) / 3)).concat("_")
                                .concat(QUARTER.getSegmentRuleName());
                break;
            case YEAR:
            default:
                suffixDataFormat = String.valueOf(calendar.get(Calendar.YEAR)).concat("_").concat(YEAR.getSegmentRuleName());
                break;
        }
        return suffixDataFormat;
    }


    /**
     * 索引拆分后根据bocp配置找到对应的index索引
     *
     * @param profile
     * @param entityClassCode
     * @return
     **/
    @Override
    public Tuple2<String, String> getSearchSegmentIndex(String profile, String entityClassCode) {
        IEntityClass iEntityClass = entityClassEngine.loadByCode(entityClassCode, profile).get();
        Map<Long, BocpElasticConfigPo> bocpConfig = BocpMetabaseCacheUtils.getBocpConfig(getProfile(profile));
        String indexName = getIndexName(profile, iEntityClass);
        String segmentIndex = indexName;
        Map<Long, Map<String, String>> indexMapping = BocpMetabaseCacheUtils.getIndexMapping(getProfile(profile));
        ;
        if (bocpConfig != null) {
            BocpElasticConfigPo targetPo = bocpConfig.get(iEntityClass.id());
            if (targetPo == null && !"default".equalsIgnoreCase(profile)) {
                //use fallback general config
                Map<Long, BocpElasticConfigPo> defaultBOCPConfig = BocpMetabaseCacheUtils.getBocpConfig("default");

                if (defaultBOCPConfig != null) {
                    targetPo = defaultBOCPConfig.get(iEntityClass.id());
                }
            }

            if (targetPo == null) {
                throw new RuntimeException(
                        "The synchronization information of the index is not configured in the bocp configuration. Therefore, the query is not supported");
            }

            Map emptyMap = new HashMap<>();
            if (targetPo.isEnableSegment()) {
                SegmentIndexRule segmentRule = targetPo.getSegmentRule();
                String suffixDataFormat = indexSuffixMatch(segmentRule, Calendar.getInstance());
                if (indexMapping == null || indexMapping.get(iEntityClass.id()) == null) {
                    segmentIndex = createSegmentIndex(profile, iEntityClass, emptyMap, suffixDataFormat);
                } else {
                    //TODO if first has no value may be no index so this should be handled
                    Optional<String> segmentIndexOp = indexMapping.get(iEntityClass.id()).values().stream().findFirst();
                    if (segmentIndexOp.isPresent()) {
                        segmentIndex = segmentIndexOp.get();
                    } else {
                        return new Tuple2(indexName, "$TEMP$");
                    }
                }
                switch (segmentRule) {
                    case YEAR:
                        indexName = indexName.concat("*_").concat(YEAR.getSegmentRuleName());
                        break;
                    case QUARTER:
                        indexName = indexName.concat("*_").concat(QUARTER.getSegmentRuleName());
                        break;
                }
            } else {
                Map<String, String> segmentIndexMapping = indexMapping.get(iEntityClass.id());
                if (segmentIndexMapping != null) {
                    segmentIndex = segmentIndexMapping.get(indexName);
                } else {
                    segmentIndex = createSegmentIndex(profile, iEntityClass, emptyMap, null);
                }
            }
        } else {
            throw new RuntimeException(
                    "The synchronization information of the index is not configured in the bocp configuration. Therefore, the query is not supported");
        }
        return new Tuple2(indexName, segmentIndex);
    }

    /**
     * @param profile
     * @param appCode
     * @return
     **/
    @Override
    public String getIndexPrefix(String profile, String appCode) {
        return DynamicConfigUtils.insulateTenant(dynamicConfig, profile, appCode);
    }

    /**
     * profile not null utils method
     *
     * @param profile
     * @return
     **/
    @NotNull
    private String getProfile(String profile) {
        if (StringUtils.isEmpty(profile)) {
            profile = CommonProperty.defaultProfile;
        }
        return profile;
    }


    /**
     * get segment indexs for profile and entityclassId
     *
     * @param profile
     * @param iEntityClass
     **/
    private Map<String, String> getSegmentIndexMapping(String profile, IEntityClass iEntityClass) {
        if (BocpMetabaseCacheUtils.getIndexMapping(profile) != null) {
            return BocpMetabaseCacheUtils.getIndexMapping(profile).get(iEntityClass.id());
        }
        return null;
    }


    /**
     * @param bocpElasticConfigPo
     * @param segmentFieldType
     * @param segmentFiledValue
     * @return
     **/
    private Date dateFormat(BocpElasticConfigPo bocpElasticConfigPo, Object segmentFiledValue, SegmentFieldType segmentFieldType) {
        switch (segmentFieldType) {
            case BIGINT:
                Long date = Long.valueOf(String.valueOf(segmentFiledValue));
                return new Date(date);
            case STRING:
                String dateStr = String.valueOf(segmentFiledValue);
                String dateFormat = verifyDateFormate(bocpElasticConfigPo);
                SimpleDateFormat formatter = new SimpleDateFormat(StringUtils.trim(dateFormat));
                try {
                    return formatter.parse(dateStr);
                } catch (ParseException e) {
                    log.error("FAILURE:bocp config parse exception,tenantCode:{},tenantCode:{},dataStr:{} formatter:{},cause by:{}",
                            bocpElasticConfigPo.getTenantCode(), bocpElasticConfigPo.getEntityClassCode(), dateStr, formatter, e.getMessage());
                    throw new RuntimeException(e.getMessage());
                }
            default:
                String error = String.format("FAILURE:bocp config parse exception,segmentFieldType:{%s} fileds not match bigint or string.",
                        bocpElasticConfigPo.getSegmentFieldType());
                log.error(error);
                throw new RuntimeException(error);
        }
    }

    @NotNull
    private String verifyDateFormate(BocpElasticConfigPo bocpElasticConfigPo) {
        String dateFormat = "yyyy-mm-dd hh:mm:ss";
        switch (bocpElasticConfigPo.getSegmentDateFormat().toLowerCase()) {
            case "yyyy-mm-dd":
                dateFormat = "yyyy-MM-dd";
                break;
            case "yyyy-mm-dd hh:mm:ss":
                dateFormat = "yyyy-MM-dd HH:mm:ss";
                break;
            case "yyyy-mm-dd hh:mm:ss.sss":
                dateFormat = "yyyy-MM-dd HH:mm:ss.SSS";
                break;
            case "yyyymmdd":
                dateFormat = "yyyyMMdd";
                break;
        }
        return dateFormat;
    }

    /**
     * 表创建失败或异常时进行日志打印
     *
     * @param iEntityClass
     * @param status
     **/
    public void logInfo(boolean status, IEntityClass iEntityClass, List<Tuple2<String, String>> errors, String message) {
        if (!status) {
            errors.add(Tuple.of(iEntityClass.code(), Optional.ofNullable(message).orElse("NullPoint Exception")));
        }
    }

    /**
     * 打印errors异常，并退出applicationContext
     *
     * @param errors
     **/
    public void printErrors(List<Tuple2<String, String>> errors) {
        if (!errors.isEmpty()) {
            StringBuilder sb = errorsBuild(errors);
            log.error("FATAL --- elasticsearch index mapping modification or creation failure. {}{}", System.lineSeparator(), sb);
            int exitCode = SpringApplication.exit(applicationContext, () -> 0);
            System.exit(exitCode);
        }
    }

    @NotNull
    private StringBuilder errorsBuild(List<Tuple2<String, String>> errors) {
        StringBuilder builder = new StringBuilder();
        errors.stream().forEach(x -> {
            String entityCode = x._1;
            String reason = x._2;
            builder.append(entityCode).append(":").append(reason).append("|")
                    .append(System.lineSeparator());
        });
        return builder;
    }

    public void setLoadFinish(boolean loadFinish) {
        this.loadFinish = loadFinish;
    }

    @Override
    public boolean getLoadFinish() {
        return loadFinish();
    }

    /**
     * refresh from elasticsearch
     *
     * @return
     */
    @Override
    public void initSync() {
        Source.tick(Duration.ZERO, Duration.ofSeconds(30), TICK)
                .runForeach(i -> {
                    try {
                        this.sync();
                    } catch (Throwable throwable) {
                        log.error("{}", throwable);
                    }
                }, mat);
    }

    private void sync() {
        if (loadFinish) {
            log.debug("Start to Sync elasticsearch indics");
            //get elastic index
            List<String> bocpConfigAllKey = BocpMetabaseCacheUtils.getBocpConfigAllKey();
            Map<String, Tuple2<String, IEntityClass>> searchMapping = new HashMap<>();
            bocpConfigAllKey.forEach(k -> {
                Map<Long, BocpElasticConfigPo> bocpConfig = BocpMetabaseCacheUtils.getBocpConfig(k);
                Set<String> expectedIndics = bocpConfig.entrySet().stream().map(x -> {
                    Long key = x.getKey();
                    Optional<IEntityClass> targetOp = entityClassEngine.load(key.toString(), null);
                    if (targetOp.isPresent()) {
                        IEntityClass entityClass = targetOp.get();
                        boolean enableSegment = x.getValue().isEnableSegment();
                        //only sync the non-segment index
                        if (!enableSegment) {
                            Tuple2<String, String> searchSegmentIndex = this.getSearchSegmentIndex(k, entityClass.code());
                            searchMapping.put(searchSegmentIndex._1, Tuple.of(k, entityClass));
                            return searchSegmentIndex._1;
                        }
                    }
                    return null;
                }).filter(Objects::nonNull).collect(Collectors.toSet());

                Set<String> allIndic = indexOperation.getAllIndex(k);

                Sets.difference(expectedIndics, allIndic).forEach(x -> {
                    //String profile, IEntityClass iEntityClass, Map<String, String> existeSegmentIndexs, String suffixDataFormat
                    Tuple2<String, IEntityClass> tuple = searchMapping.get(x);
                    if(tuple != null) {
                        createSegmentIndex(tuple._1, tuple._2,  new HashMap<>(), null);
                    }
                });
            });
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        initSync();
    }
}
