package com.xforceplus.ultraman.adapter.elasticsearch;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.xforceplus.ultraman.adapter.elasticsearch.transport.ElasticsearchTransportExecutor;
import com.xforceplus.ultraman.sdk.core.datasource.route.TransportExecutor;
import com.xforceplus.ultraman.sdk.core.datasource.route.dynamic.config.DynamicConfig;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.client.*;
import org.elasticsearch.client.indices.GetIndexRequest;
import org.elasticsearch.client.indices.GetIndexResponse;
import org.elasticsearch.rest.RestStatus;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.*;
import java.util.function.Function;


/**
 * @ClassName CustomElasticSearchTransport
 * @description:
 * @author: WanYi
 * @create: 2023-08-07 19:51
 * @Version 1.0
 **/
@Slf4j
public class CustomElasticSearchTransport {


  @Autowired
  private DynamicConfig dynamicConfig;
  @Autowired
  private TransportExecutor elasticsearchTransportExecutor;

  public CustomElasticSearchTransport(DynamicConfig dynamicConfig,
      TransportExecutor elasticsearchTransportExecutor) {
    this.dynamicConfig = dynamicConfig;
    this.elasticsearchTransportExecutor = elasticsearchTransportExecutor;
  }

  public JSONObject getIndexStats(String indexName, String profile) {
    try {
      RestHighLevelClient executor =
          ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile);
      String path = String.format(Locale.ROOT, "/%s/_stats?pretty", indexName);
      final HttpGet get;
      URIBuilder builder = new URIBuilder(path);
      get = new HttpGet(builder.build());
      Response apply = new CustomElasticSearchTransport.HttpFunction(executor.getLowLevelClient()).apply(get);
      JSONObject jsonObject = JSON.parseObject(EntityUtils.toString(apply.getEntity()));
      JSONObject primaries = jsonObject.getJSONObject("_all").getJSONObject("primaries");
      JSONObject docs = primaries.getJSONObject("docs");
      JSONObject store = primaries.getJSONObject("store");
      Map<String, JSONObject> indexStatus = new HashMap<>();
      indexStatus.put("docs", docs);
      indexStatus.put("store", store);
      return new JSONObject(indexStatus);
    } catch (Throwable e) {
      throw new RuntimeException(e);
    }
  }

  public Set<String> getMatchIndexs(String matchIndexName, String profile, List<Tuple2<String, String>> errors) {
    try {
      String matchSegmentIndex = matchIndexName.concat("_*");
      RestHighLevelClient executor = ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile);
      GetIndexResponse getIndexResponse = executor.indices().get(new GetIndexRequest(matchSegmentIndex), RequestOptions.DEFAULT);
      return getIndexResponse.getMappings().keySet();
    } catch (Throwable e) {
      if (((ElasticsearchStatusException) e).status() == RestStatus.NOT_FOUND) {
        log.info(e.getMessage());
      } else {
        log.error("FAILURE:elasticSearch execute method,cause by:", e.getMessage());
        errors.add(new Tuple2<>(matchIndexName, e.getMessage()));
      }
    }
    return new HashSet<>();
  }

  /**
   * Basic rest operations interacting with elastic cluster.
   */
  private static class HttpFunction implements Function<HttpRequest, Response> {

    private final RestClient restClient;

    HttpFunction(final RestClient restClient) {
      this.restClient = Objects.requireNonNull(restClient, "restClient");
    }

    @Override
    public Response apply(final HttpRequest request) {
      try {
        return applyInternal(request);
      } catch (IOException e) {
        throw new UncheckedIOException(e);
      }
    }

    private Response applyInternal(final HttpRequest request)
        throws IOException {

      Objects.requireNonNull(request, "request");
      final HttpEntity entity = request instanceof HttpEntityEnclosingRequest
          ? ((HttpEntityEnclosingRequest) request).getEntity() : null;

      final Request r = new Request(
          request.getRequestLine().getMethod(),
          request.getRequestLine().getUri());
      r.setEntity(entity);
      final Response response = restClient.performRequest(r);
      final String payload = entity != null && entity.isRepeatable()
          ? EntityUtils.toString(entity) : "<empty>";

      if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
        final String error = EntityUtils.toString(response.getEntity());

        final String message = String.format(Locale.ROOT,
            "Error while querying Elastic (on %s/%s) status: %s\nPayload:\n%s\nError:\n%s\n",
            response.getHost(), response.getRequestLine(),
            response.getStatusLine(), payload, error);
        throw new RuntimeException(message);
      }

      return response;
    }
  }
}
