/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.xforceplus.ultraman.adapter.elasticsearch;

import static com.xforceplus.metadata.schema.dsl.Step.BO;
import static com.xforceplus.metadata.schema.runtime.MetadataEngine.LABEL_INDEX;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.xforceplus.metadata.schema.dsl.metadata.__;
import com.xforceplus.metadata.schema.runtime.MetadataEngine;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.adapter.elasticsearch.service.ManageBocpMetadataService;
import com.xforceplus.ultraman.adapter.elasticsearch.transport.ElasticsearchTransportExecutor;
import com.xforceplus.ultraman.adapter.elasticsearch.utils.DynamicConfigUtils;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.sdk.core.calcite.LazySchemaMap;
import com.xforceplus.ultraman.sdk.core.datasource.route.TransportExecutor;
import com.xforceplus.ultraman.sdk.core.datasource.route.dynamic.config.DynamicConfig;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import io.vavr.Tuple;
import io.vavr.Tuple2;

import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.apache.calcite.schema.Table;
import org.apache.calcite.schema.impl.AbstractSchema;
import org.elasticsearch.client.RestClient;

/**
 * Each table in the schema is an ELASTICSEARCH index.
 */
public class ElasticsearchSchema extends AbstractSchema {

    private MetadataEngine metadataEngine;
    private EntityClassEngine classEngine;
    private ContextService contextService;

    private ProfileFetcher fetcher;

    /**
     * Default batch size to be used during scrolling.
     */
    private int fetchSize;


    private ObjectMapper mapper;

    private Map<String, Table> tableMap;
    private String appCode;
    private TransportExecutor transportExecutor;
    private ManageBocpMetadataService manageBocpMetadataService;

    private Map<Tuple2<String, String>, ElasticsearchTransport> transportMap = new ConcurrentHashMap<>();

    private LazySchemaMap<String, Table> schemaMap;

    @VisibleForTesting
    ElasticsearchSchema(TransportExecutor transportExecutor
            , ObjectMapper mapper
            , MetadataEngine metadataEngine
            , ProfileFetcher fetcher
            , EntityClassEngine entityClassEngine
            , ContextService contextService
            , ManageBocpMetadataService manageBocpMetadataService
            , String appCode, int fetchSize) {
        this.mapper = Objects.requireNonNull(mapper, "mapper");
        this.metadataEngine = metadataEngine;
        Preconditions.checkArgument(fetchSize > 0,
                "invalid fetch size. Expected %s > 0", fetchSize);
        schemaMap = new LazySchemaMap<>(code -> {

            String profile = fetcher.getProfile(contextService.getAll());

            //TODO Transport shutdown
            RestClient client = ((ElasticsearchTransportExecutor) transportExecutor).executor(profile)
                    .getLowLevelClient();
            Tuple2<String, String> searchIndex = manageBocpMetadataService.getSearchSegmentIndex(profile, code);

            /**
             * 适配需要路由的segment index
             * ***/
            //TODO refresh
            Tuple2<String, String> key = Tuple.of(code.toLowerCase(Locale.ROOT), searchIndex._2);
            ElasticsearchTransport elasticsearchTransport = transportMap
                    .get(key);
            ElasticsearchTransport transport;
            String prefix = manageBocpMetadataService.getIndexPrefix(profile, appCode);


            if (elasticsearchTransport == null) {
                transport = new ElasticsearchTransport(client, mapper,
                        searchIndex, code.toLowerCase(Locale.ROOT), code, fetchSize);
                transportMap.put(key, transport);
            } else {
                transport = elasticsearchTransport;
            }


            return new ElasticsearchTable(classEngine, transport, contextService, fetcher, prefix, this);
        });
        
        schemaMap.keySet(() -> {
            List<Map<String, Object>> boList = metadataEngine.getMulti(__.has(LABEL_INDEX, BO));
            return boList.stream()
                    .map(x -> x.get("code").toString().toLowerCase(Locale.ROOT))
                    .collect(Collectors.toSet());
        });
        this.fetchSize = fetchSize;
        this.fetcher = fetcher;
        this.classEngine = entityClassEngine;
        this.contextService = contextService;
        this.tableMap = createTables();
        this.manageBocpMetadataService = manageBocpMetadataService;
        this.appCode = appCode;
        this.transportExecutor = transportExecutor;

    }

    public Map<Tuple2<String, String>, ElasticsearchTransport> getTransportMap() {
        return transportMap;
    }

    public ElasticsearchSchema(TransportExecutor transportExecutor, ObjectMapper mapper,
                               MetadataEngine metadataEngine,
                               EntityClassEngine entityClassEngine, ProfileFetcher fetcher, ContextService contextService,
                               ManageBocpMetadataService manageBocpMetadataService,
                               String appCode) {
        this(transportExecutor, mapper, metadataEngine, fetcher, entityClassEngine, contextService, manageBocpMetadataService, appCode,
                ElasticsearchTransport.DEFAULT_FETCH_SIZE);
    }

    @Override
    protected Map<String, Table> getTableMap() {
        return tableMap;
    }


    private Map<String, Table> createTables() {
        return schemaMap;
    }
}
