/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.xforceplus.ultraman.adapter.elasticsearch;

import com.alibaba.google.common.base.Supplier;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.google.common.collect.ImmutableMap;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.adapter.elasticsearch.ElasticsearchJson.SearchHit;
import com.xforceplus.ultraman.adapter.elasticsearch.ElasticsearchJson.SqlResult;
import com.xforceplus.ultraman.adapter.elasticsearch.query.utils.ParseSqlNodeUtils;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.entity.IRelation;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.Tuple3;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.DataContext;
import org.apache.calcite.adapter.java.AbstractQueryableTable;
import org.apache.calcite.linq4j.Enumerable;
import org.apache.calcite.linq4j.Enumerator;
import org.apache.calcite.linq4j.Linq4j;
import org.apache.calcite.linq4j.QueryProvider;
import org.apache.calcite.linq4j.Queryable;
import org.apache.calcite.linq4j.function.Function1;
import org.apache.calcite.plan.RelOptCluster;
import org.apache.calcite.plan.RelOptTable;
import org.apache.calcite.rel.RelFieldCollation;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.RelDataTypeFactory;
import org.apache.calcite.schema.SchemaPlus;
import org.apache.calcite.schema.Table;
import org.apache.calcite.schema.TranslatableTable;
import org.apache.calcite.schema.impl.AbstractTableQueryable;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.sql.dialect.MysqlSqlDialect;
import org.apache.calcite.util.Pair;
import org.apache.calcite.util.Util;
import org.apache.commons.lang3.StringUtils;
import scala.concurrent.java8.FuturesConvertersImpl;

/**
 * Table based on an Elasticsearch index.
 */
@Slf4j
public class ElasticsearchTable extends AbstractQueryableTable implements TranslatableTable {

    /**
     * Used for constructing (possibly nested) Elastic aggregation nodes.
     */
    private static final String AGGREGATIONS = "aggregations";
    public final ObjectMapper mapper;
    private final ElasticsearchVersion version;
    public final EntityClassEngine engine;
    private final ContextService contextService;
    public final ProfileFetcher fetcher;
    private final String indexName;
    private final ElasticsearchTransport transport;
    public final String entityCode;

    private final String prefix;

    public List<String> list = new ArrayList<>();

    private ElasticsearchSchema schema;

    /**
     * Creates an ElasticsearchTable.
     */
    public ElasticsearchTable(EntityClassEngine engine, ElasticsearchTransport transport,
                              ContextService contextService, ProfileFetcher fetcher, String prefix, ElasticsearchSchema schema) {
        super(Object[].class);
        this.engine = engine;
        this.transport = transport;
        this.version = Optional.ofNullable(transport).map(x -> x.version).orElse(null);
        this.mapper = Optional.ofNullable(transport).map(x -> x.mapper()).orElse(null);
        this.contextService = contextService;
        this.fetcher = fetcher;
        this.indexName = Optional.ofNullable(transport).map(x -> x.indexName).orElse(null);
        this.entityCode = Optional.ofNullable(transport).map(x -> x.entityCode).orElse(null);
        this.prefix = prefix;
        this.schema = schema;
    }

    /**
     * In ES 5.x scripted fields start with {@code params._source.foo} while in ES2.x {@code _source.foo}. Helper method to build correct query based on
     * runtime version of elastic. Used to keep backwards compatibility with ES2.
     *
     * @return string to be used for scripted fields
     * @see <a href="https://github.com/elastic/elasticsearch/issues/20068">_source variable</a>
     * @see <a href="https://www.elastic.co/guide/en/elasticsearch/reference/master/modules-scripting-fields.html">Scripted
     * Fields</a>
     */
    public String scriptedFieldPrefix() {
        // ES2 vs ES5 scripted field difference
        return version == ElasticsearchVersion.ES2
                ? ElasticsearchConstants.SOURCE_GROOVY
                : ElasticsearchConstants.SOURCE_PAINLESS;
    }

    private void merge(ObjectNode query, ObjectNode newNode, String key) {
        if (query.has(key)) {
            ObjectNode jsonNode = (ObjectNode) query.get(key);
            JsonNode newTargetNode = newNode.get("query");
            //TODO
            JsonNode merge = merge(jsonNode, newTargetNode);
//            query.setAll((ObjectNode) merge);
        } else {
            query.setAll(newNode);
        }
    }

    public static JsonNode merge(JsonNode mainNode, JsonNode updateNode) {

        Iterator<String> fieldNames = updateNode.fieldNames();

        while (fieldNames.hasNext()) {
            String updatedFieldName = fieldNames.next();
            JsonNode valueToBeUpdated = mainNode.get(updatedFieldName);
            JsonNode updatedValue = updateNode.get(updatedFieldName);

            // If the node is an @ArrayNode
            if (valueToBeUpdated != null && valueToBeUpdated.isArray()) {

                if(updatedValue.isArray()) {
                    // running a loop for all elements of the updated ArrayNode
                    for (int i = 0; i < updatedValue.size(); i++) {
                        JsonNode updatedChildNode = updatedValue.get(i);
                        // Create a new Node in the node that should be updated, if there was no corresponding node in it
                        // Use-case - where the updateNode will have a new element in its Array
                        if (valueToBeUpdated.size() <= i) {
                            ((ArrayNode) valueToBeUpdated).add(updatedChildNode);
                        }
                        // getting reference for the node to be updated
                        JsonNode childNodeToBeUpdated = valueToBeUpdated.get(i);
                        merge(childNodeToBeUpdated, updatedChildNode);
                    }
                } else {
                    ((ArrayNode) valueToBeUpdated).add(updatedValue);
                }
                // if the Node is an @ObjectNode
            } else if (valueToBeUpdated != null && valueToBeUpdated.isObject()) {
                merge(valueToBeUpdated, updatedValue);
            } else {
                if (mainNode instanceof ObjectNode) {
                    ((ObjectNode) mainNode).replace(updatedFieldName, updatedValue);
                }
            }
        }
        return mainNode;
    }

    /**
     * Executes a "find" operation on the underlying index.
     *
     * @param ops          List of operations represented as Json strings.
     * @param fields       List of fields to project; or null to return map
     * @param sort         list of fields to sort and their direction (asc/desc)
     * @param aggregations aggregation functions
     * @return Enumerator of results
     */
    public Enumerable<Object> find(List<String> ops,
                                    List<Map.Entry<String, Class>> fields,
                                    List<Map.Entry<String, RelFieldCollation.Direction>> sort,
                                    List<String> groupBy,
                                    List<Map.Entry<String, String>> aggregations,
                                    Map<String, String> mappings,
                                    Map<String, String> rawNameMapping,
                                    Long offset, Long fetch, RelNode rawTree, DataContext dataContext) throws IOException {

        if (transport == null) {
            return Linq4j.emptyEnumerable();
        }

        String profile = fetcher.getProfile(Collections.emptyMap());
        Optional<IEntityClass> targetOp = engine.loadByCode(entityCode, profile);
        if (!targetOp.isPresent()) {
            //TODO
            throw new RuntimeException("ElasticSearch Table Not found " + entityCode);
        }

        EntityClassGroup group = engine.describe(targetOp.get(), profile);


        try {
            if (!aggregations.isEmpty() || !groupBy.isEmpty()) {
                // process aggregations separately
                //TODO
                return aggregate(transport, group, ops, fields, sort, groupBy, aggregations, mappings, rawNameMapping, offset, fetch, false);
            }

            final ObjectNode query = mapper.createObjectNode();
            Boolean join_query = contextService.getAll()
                    .get("join_query") != null && (Boolean) contextService.getAll().get("join_query");
            // manually parse from previously concatenated string;
            String targetIndex = indexName;
            for (String op : ops) {
                if (mapper.readTree(op).get("query") != null) {
                    if (join_query) {
                        /**根据filter过滤条件拆成has_parent与has_child查询方式 过滤parent-child index索引表**/
                        targetIndex = generateJoinModelQuery(group, query, op);
                    } else {
                        query.setAll((ObjectNode) mapper.readTree(op));
                    }
                    if (contextService.getAll().get("hasCount") != null && (boolean) contextService.getAll().get("hasCount")) {
                        //query.put("track_total_hits", true);
                        if (indexName.equals(targetIndex)) {
                            executeQueryCount(transport, group, mappings, rawNameMapping, query.toString(), true);
                        } else {
                            ElasticsearchTable table = (ElasticsearchTable) schema.getTable(targetIndex);
                            executeQueryCount(table.transport, group, mappings, rawNameMapping, query.toString(), true);
                        }
                    }
                    continue;
                }

                if (mapper.readTree(op).get("fields") != null) {
                    if (!targetIndex.equalsIgnoreCase(indexName)) {
                        //query in related index should add prefix for every field
                        JsonNode jsonNode = mapper.readTree(op).get("fields");
                        ObjectNode fieldsNode = mapper.createObjectNode();
                        ArrayNode arrayNode = mapper.createArrayNode();
                        fieldsNode.put("fields", arrayNode);
                        (jsonNode).forEach(x -> {
                            if (x instanceof TextNode) {
                                arrayNode.add(new TextNode(group.getEntityClass().code().concat(".").concat(x.asText())));
                            }
                        });
                        query.setAll(fieldsNode);
                    } else {
                        query.setAll((ObjectNode) mapper.readTree(op));
                    }
                    continue;
                }

                query.setAll((ObjectNode) mapper.readTree(op));
            }

            if (!sort.isEmpty()) {
                ArrayNode sortNode = query.withArray("sort");
                sort.forEach(e -> {
                            String key = e.getKey();
                            if (!StringUtils.isEmpty(key)) {
                                boolean isJson = key.startsWith("__json");
                                JsonNode input = null;
                                if (isJson) {
                                    if (key.contains("$order$")) {
                                        key = key.replace("$order$", e.getValue().isDescending() ? "desc" : "asc");
                                    }
                                    try {
                                        input = mapper.readTree(key.substring(6));
                                    } catch (JsonProcessingException ex) {
                                        ex.printStackTrace();
                                    }
                                } else {
                                    input = mapper.createObjectNode()
                                            .put(key,
                                                    e.getValue().isDescending() ? "desc" : "asc");
                                }
                                if (input != null) {
                                    sortNode.add(input);
                                }
                            }
                        }
                );
            }
            if (offset != null) {
                query.put("from", offset);
            }
            if (fetch != null) {
                query.put("size", fetch);
            }

            query.put("_source", false);

            ElasticsearchTransport targetTransport;
            if (!targetIndex.equalsIgnoreCase(indexName)) {
                targetTransport = ((ElasticsearchTable) schema.getTable(targetIndex)).transport;
            } else {
                targetTransport = transport;
            }

            Iterable<ElasticsearchJson.SearchHit> iter;
            if (offset == null) {
                // apply scrolling when there is no offsets
                String finalTargetIndex = targetIndex;
                iter = () -> new Scrolling(targetTransport, group, contextService).query(finalTargetIndex.equalsIgnoreCase(indexName), query);
            } else {
                final ElasticsearchJson.Result search = targetTransport.search().apply(query);
                ElasticsearchJson.SearchHits searchHits = search.searchHits();
                ElasticsearchJson.SearchTotal total = searchHits.total();
                if (total != null && contextService.getAll().get("show_count") == null) {
                    contextService.getAll().put("show_count", total.value());
                }
                String finalTargetIndex1 = targetIndex;
                iter = () -> search.searchHits().flattenHits(finalTargetIndex1.equalsIgnoreCase(indexName), group).iterator();
            }

            final Function1<ElasticsearchJson.SearchHit, Object> getter =
                    ElasticsearchEnumerators.getter(fields, rawNameMapping, ImmutableMap.copyOf(mappings));

            return Linq4j.asEnumerable(iter).select(getter);

        } catch (Throwable throwable) {
            log.warn("elasticsearch table execute query failed!, {}", throwable.getMessage());
            contextService.getAll().compute("errors", (k, v) -> {
                if (v == null) {
                    v = new ArrayList<>();
                }
                ((List<String>) v).add(throwable.getMessage());
                return v;
            });
            throw throwable;
        }
    }

    /**
     * 构建count查询DQL语句
     *
     * @param mappings
     * @param op
     **/
    private void executeQueryCount(ElasticsearchTransport targetTransport, EntityClassGroup group, Map<String, String> mappings, Map<String, String> rawMapping, String op, boolean ignoreJoin) throws IOException {
        Map<String, String> aggregationMap = new HashMap();
        Map<String, Class> fieldMap = new HashMap();
        aggregationMap.put("c", "{\"value_count\":{\"field\":\"_id\"}}");
        fieldMap.put("c", Long.class);
        List<Entry<String, Class>> fieldEntries = new ArrayList<>();
        List<Entry<String, String>> aggregationEntries = new ArrayList<>();
        aggregationEntries.addAll(aggregationMap.entrySet());
        fieldEntries.addAll(fieldMap.entrySet());
        aggregate(targetTransport, group, Collections.singletonList(op), fieldEntries, new ArrayList<>(), new ArrayList<>(), aggregationEntries, mappings, rawMapping, null, null, ignoreJoin);
    }


    /**
     * TODO current deal bool not others
     * TODO
     *
     * @param sourceNode
     * @param mapper
     * @param node
     * @param nonRelatedPos
     * @param relatedPos
     * @return first is related, second is if handled
     */
    private String visitor(JsonNode sourceNode, ObjectMapper mapper, JsonNode node, Set<Tuple2<JsonNode, JsonNode>> nonRelatedPos
            , Set<Tuple3<String, JsonNode, JsonNode>> relatedPos, Set<Tuple3<String, JsonNode, JsonNode>> relatedAnchorList, Set<Tuple2<JsonNode, JsonNode>> selfAnchorList) {
        if (sourceNode.isArray()) {
            Iterator<JsonNode> elements = sourceNode.elements();
            while (elements.hasNext()) {
                JsonNode next = elements.next();
                visitor(next, mapper, node, nonRelatedPos, relatedPos, relatedAnchorList, selfAnchorList);
            }
        } else if (sourceNode.isObject()) {

            Iterator<Entry<String, JsonNode>> rootFields = sourceNode.fields();

            ObjectNode parent = mapper.createObjectNode();

            //node.put("query", "parent");
            while (rootFields.hasNext()) {
                Entry<String, JsonNode> next = rootFields.next();
                String nextToken = next.getKey();
                if (node != null) {
                    if (node.isArray()) {
                        ((ArrayNode) node).add(parent);
                    } else {
                        ((ObjectNode) node).put(nextToken, parent);
                    }
                }

                switch (nextToken) {
                    case "bool":
                        JsonNode value = next.getValue();
                        Iterator<Entry<String, JsonNode>> subClauses = value.fields();
                        while (subClauses.hasNext()) {
                            Entry<String, JsonNode> subClause = subClauses.next();
                            String subClauseKey = subClause.getKey();
                            switch (subClauseKey) {
                                case "must_not":
                                case "filter":
                                case "must":
                                    JsonNode mustClause = subClause.getValue();
                                    if (mustClause instanceof ArrayNode) {
                                        ArrayNode arrayNode = mapper.createArrayNode();
                                        parent.put(subClauseKey, arrayNode);
                                        Map<String, ArrayNode> relatedMapping = new HashMap<>();

                                        for (JsonNode x : mustClause) {
                                            String relatedCode = visitor(x, mapper, null, nonRelatedPos, relatedPos, relatedAnchorList, selfAnchorList);
                                            if (!StringUtils.isEmpty(relatedCode)) {
                                                relatedMapping.compute(relatedCode, (k, v) -> {
                                                    if (v == null) {
                                                        v = mapper.createArrayNode();
                                                    }
                                                    v.add(x);
                                                    return v;
                                                });
                                            } else {
                                                selfAnchorList.add(Tuple.of(arrayNode, x));
                                            }
                                        }

                                        if (!relatedMapping.isEmpty()) {
                                            for (Entry<String, ArrayNode> entry : relatedMapping.entrySet()) {
                                                relatedAnchorList.add(Tuple.of(entry.getKey(), arrayNode, entry.getValue()));
                                            }
                                        }
                                    } else {
                                        //TODO has no list? 
                                        //TODO check
                                        return visitor(mustClause, mapper, parent, nonRelatedPos, relatedPos, relatedAnchorList, selfAnchorList);
                                    }
                                    break;
                            }
                        }
                        break;
                    case "match":
                    case "range":
                    case "term":
                        return visitor(next.getValue(), mapper, parent, nonRelatedPos, relatedPos, relatedAnchorList, selfAnchorList);
                    case "exists":
                        JsonNode existsFields = next.getValue();
                        JsonNode jsonNode1 = existsFields.get("field");
                        String relatedResult = visitor(jsonNode1, mapper, parent, nonRelatedPos, relatedPos, relatedAnchorList, selfAnchorList);
                        if (!StringUtils.isEmpty(relatedResult)) {
                            relatedPos.add(Tuple.of(relatedResult, sourceNode, existsFields));
                        } else {
                            nonRelatedPos.add(Tuple.of(sourceNode, existsFields));
                        }
                        return relatedResult;
                    default:
                        String key = next.getKey();
                        if (key.contains(".")) {
                            String relatedCode = key.split("\\.")[0];
                            relatedPos.add(Tuple.of(relatedCode, sourceNode, sourceNode));
                            return relatedCode;
                        } else {
                            nonRelatedPos.add(Tuple.of(sourceNode, sourceNode));
                            parent.set(key, next.getValue());
                        }
                }
            }
        } else {
            if (sourceNode.isTextual() && sourceNode.textValue().contains(".")) {
                String[] split = sourceNode.textValue().split("\\.");
                return split[0];
            }
        }

        return null;
    }

    /**
     * 适配join(chind-parent) elastic 索引模型查询语句
     *
     * @param query
     * @param op
     **/
    private String generateJoinModelQuery(EntityClassGroup group, ObjectNode query, String op) throws JsonProcessingException {
        JsonNode queryNodes = mapper.readTree(op).get("query");
        /**取出filter中的过滤条件**/
        JsonNode jsonNode = queryNodes.get("constant_score").get("filter");

//        //check if need reverse
//        jsonNode

        ObjectNode objectNode = mapper.createObjectNode();
        Set<Tuple3<String, JsonNode, JsonNode>> list = new HashSet<>();
        Set<Tuple2<JsonNode, JsonNode>> noneList = new HashSet<>();
        Set<Tuple3<String, JsonNode, JsonNode>> anchorList = new HashSet<>();
        Set<Tuple2<JsonNode, JsonNode>> selfAnchorList = new HashSet<>();
        visitor(jsonNode, mapper, objectNode, noneList, list, anchorList, selfAnchorList);


        boolean needQueryInRelatedIndex = false;
        String targetEntityCode = null;
        Set<String> toOneRelatedSet = new HashSet<>();
        //check
        if (!list.isEmpty()) {
            //record same relation code

            for (Tuple3<String, JsonNode, JsonNode> x : list) {
                JsonNode targetNode = x._2;
                String relatedCode = x._1;
                Optional<IRelation> relation = group.relation(relatedCode);
                if (relation.isPresent()) {
                    boolean isToOne = relation.get().getRelationType().equalsIgnoreCase("TO_ONE");
                    if (isToOne) {
                        //TODO
                        toOneRelatedSet.add(relatedCode.toLowerCase());
                        if (toOneRelatedSet.size() > 1) {
                            throw new RuntimeException("Cannot query TO ONE more than one");
                        }
                        needQueryInRelatedIndex = true;

                    }

                    Optional<IEntityClass> relatedEntityClassOp = group.relatedEntityClass(relatedCode);
                    if (relatedEntityClassOp.isPresent()) {
                        targetEntityCode = relatedEntityClassOp.get().code();
                    }
                }

                //TODO break;
            }


            if (needQueryInRelatedIndex) {
                //TO_ONE QUERY
                //deal related
                if (!anchorList.isEmpty()) {
                    //change to has_parent
                    for (Tuple3<String, JsonNode, JsonNode> tuple3 : anchorList) {

                        JsonNode anchorNode = tuple3._2;
                        //every entry to create a has_parent query
                        ObjectNode hasChildQuery = mapper.createObjectNode();
                        ObjectNode queryNode = mapper.createObjectNode();
                        ObjectNode mustNode = mapper.createObjectNode();
                        queryNode.set("bool", mustNode.set("must", tuple3._3));
                        hasChildQuery.set("parent_type", new TextNode(targetEntityCode));
                        hasChildQuery.set("query", queryNode);
                        hasChildQuery.set("inner_hits", mapper.createObjectNode());

                        if (anchorNode.isArray()) {
                            ObjectNode childRootNode = mapper.createObjectNode();
                            ((ArrayNode) anchorNode).add(childRootNode);
                            childRootNode.set("has_parent", hasChildQuery);
                        } else {
                            ((ObjectNode) anchorNode).set("has_parent", hasChildQuery);
                        }
                    }
                }

                //deal self
                if (!selfAnchorList.isEmpty()) {
                    for (Tuple2<JsonNode, JsonNode> selfTuple : selfAnchorList) {
                        JsonNode container = selfTuple._1;
                        if (container.isArray()) {
                            ((ArrayNode) container).add(selfTuple._2);
                        }
                    }
                }

                String selfCode = group.getEntityClass().code();

                /**
                 * add prefix
                 */
                for (Tuple2<JsonNode, JsonNode> tuple : noneList) {
                    JsonNode jsonNode1 = tuple._1;
                    JsonNode jsonNode2 = tuple._2;
                    if (jsonNode1 == jsonNode2) {
                        //modify self
                        if (jsonNode2.isObject()) {
                            //find related property name
                            Entry<String, JsonNode> next = jsonNode1.fields().next();
                            String newKey = selfCode.concat(".").concat(next.getKey());
                            ((ObjectNode) jsonNode1).removeAll();
                            ((ObjectNode) jsonNode1).set(newKey, next.getValue());
                        } else {
                            //TODO
                            log.warn("current not support non-object");
                        }
                    }
                }

                /**
                 * remove prefix
                 */
                for (Tuple3<String, JsonNode, JsonNode> tuple : list) {
                    //change parent to accept new one
                    JsonNode jsonNode1 = tuple._2;
                    JsonNode jsonNode2 = tuple._3;
                    if (jsonNode1 == jsonNode2) {
                        //modify self
                        if (jsonNode2.isObject()) {
                            //find related property name
                            Entry<String, JsonNode> next = jsonNode1.fields().next();
                            String[] split = next.getKey().split("\\.");
                            String newKey = split[1];
                            ((ObjectNode) jsonNode1).removeAll();
                            ((ObjectNode) jsonNode1).set(newKey, next.getValue());
                        } else {
                            //TODO
                            log.warn("current not support non-object");
                        }
                    } else {
                        boolean field = jsonNode2.has("field");
                        if (field) {
                            JsonNode jsonNode3 = jsonNode2.get("field");
                            if (jsonNode3.isTextual()) {
                                String resultString = jsonNode3.textValue().split("\\.")[1];

                                if (jsonNode1.isObject()) {
                                    Iterator<Entry<String, JsonNode>> fields = jsonNode1.fields();
                                    String targetName = null;
                                    while (fields.hasNext()) {
                                        Entry<String, JsonNode> next = fields.next();
                                        if (next.getValue() == jsonNode2) {
                                            targetName = next.getKey();
                                            break;
                                        }
                                    }

                                    if (targetName != null) {
                                        ObjectNode objectNode1 = mapper.createObjectNode();
                                        objectNode1.set("field", new TextNode(resultString));
                                        ((ObjectNode) jsonNode1).set(targetName, objectNode1);
                                    }
                                } else if (jsonNode1.isArray()) {
                                    Iterator<JsonNode> elements = jsonNode1.elements();
                                    int i = 0;
                                    boolean found = false;
                                    while (elements.hasNext()) {
                                        JsonNode next = elements.next();
                                        if (next == jsonNode2) {
                                            found = true;
                                            break;
                                        }
                                        i++;
                                    }

                                    if (found) {
                                        ((ArrayNode) jsonNode1).remove(i);
                                    }

                                    ((ArrayNode) jsonNode1).add(new TextNode(resultString));
                                }
                            } else {
                                log.warn("current not support non-textual");
                            }
                        }
                    }
                }
            } else {
                //TO_MANY QUERY
                //change code name to related entityClass
                String finalTargetEntityCode = targetEntityCode;
                //wrapper anchor with has_child

                //deal with relatedAnchorList
                if (!anchorList.isEmpty()) {
                    for (Tuple3<String, JsonNode, JsonNode> tuple3 : anchorList) {
                        JsonNode anchorNode = tuple3._2;
                        //every entry to create a has_parent query
                        ObjectNode hasChildQuery = mapper.createObjectNode();
                        ObjectNode queryNode = mapper.createObjectNode();
                        ObjectNode mustNode = mapper.createObjectNode();
                        queryNode.set("bool", mustNode.set("must", tuple3._3));
                        hasChildQuery.set("type", new TextNode(tuple3._1));
                        hasChildQuery.set("query", queryNode);
                        hasChildQuery.set("inner_hits", mapper.createObjectNode());

                        if (anchorNode.isArray()) {
                            ObjectNode childRootNode = mapper.createObjectNode();
                            ((ArrayNode) anchorNode).add(childRootNode);
                            childRootNode.set("has_child", hasChildQuery);
                        } else {
                            ((ObjectNode) anchorNode).set("has_child", hasChildQuery);
                        }
                    }
                }

                if (!selfAnchorList.isEmpty()) {
                    for (Tuple2<JsonNode, JsonNode> selfTuple : selfAnchorList) {
                        JsonNode container = selfTuple._1;
                        if (container.isArray()) {
                            ((ArrayNode) container).add(selfTuple._2);
                        }
                    }
                }

                //related code change from relationCode.xx to entityClassCode.xx
                list.forEach(tuple -> {
                    //change parent to accept new one
                    String relatedCode = tuple._1;
                    JsonNode jsonNode1 = tuple._2;
                    JsonNode jsonNode2 = tuple._3;
                    if (jsonNode1 == jsonNode2) {
                        //modify self
                        if (jsonNode2.isObject()) {
                            //find related property name
                            Entry<String, JsonNode> next = jsonNode1.fields().next();
                            String[] split = next.getKey().split("\\.");
                            String newKey = finalTargetEntityCode.concat(".").concat(split[1]);
                            ((ObjectNode) jsonNode1).removeAll();
                            ((ObjectNode) jsonNode1).set(newKey, next.getValue());
                        } else {
                            //TODO
                            log.warn("current not support non-object");
                        }
                    } else {
                        boolean field = jsonNode2.has("field");
                        if (field) {
                            JsonNode jsonNode3 = jsonNode2.get("field");
                            if (jsonNode3.isTextual()) {
                                String resultString = finalTargetEntityCode.concat(".").concat(jsonNode3.textValue().split("\\.")[1]);

                                if (jsonNode1.isObject()) {
                                    Iterator<Entry<String, JsonNode>> fields = jsonNode1.fields();
                                    String targetName = null;
                                    while (fields.hasNext()) {
                                        Entry<String, JsonNode> next = fields.next();
                                        if (next.getValue() == jsonNode2) {
                                            targetName = next.getKey();
                                            break;
                                        }
                                    }

                                    if (targetName != null) {
                                        ObjectNode objectNode1 = mapper.createObjectNode();
                                        objectNode1.set("field", new TextNode(resultString));
                                        ((ObjectNode) jsonNode1).set(targetName, objectNode1);
                                    }
                                } else if (jsonNode1.isArray()) {
                                    Iterator<JsonNode> elements = jsonNode1.elements();
                                    int i = 0;
                                    boolean found = false;
                                    while (elements.hasNext()) {
                                        JsonNode next = elements.next();
                                        if (next == jsonNode2) {
                                            found = true;
                                            break;
                                        }
                                        i++;
                                    }

                                    if (found) {
                                        ((ArrayNode) jsonNode1).remove(i);
                                    }

                                    ((ArrayNode) jsonNode1).add(new TextNode(resultString));
                                }
                            } else {
                                log.warn("current not support non-textual");
                            }
                        }
                    }
                });
            }
        }

        ((ObjectNode) jsonNode).setAll(objectNode);
        query.set("query", jsonNode);

        if (needQueryInRelatedIndex) {
            return targetEntityCode;
        } else {
            return indexName;
        }
    }

    private ObjectNode constructSelfQuery(String prefix, Map<String, Object> parentChild, Map<String, JsonNode> terms, String type) {
        ObjectNode parentBool = mapper.createObjectNode();
        ObjectNode boolNode = mapper.createObjectNode();
        ArrayNode must = mapper.createArrayNode();
        parentChild.entrySet().forEach(entry -> {
            ObjectNode objectNode = (ObjectNode) terms.get(entry.getKey());
            objectNode.fields().forEachRemaining(e -> {
                String rawKey = e.getKey();
                JsonNode termNode = e.getValue();
                ObjectNode targetNode = mapper.createObjectNode();
                ObjectNode targetTermsNode = mapper.createObjectNode();
                Iterator<Entry<String, JsonNode>> fields = termNode.fields();
                while (fields.hasNext()) {
                    Entry<String, JsonNode> next = fields.next();
                    String key = next.getKey();
                    targetTermsNode.put(prefix.concat(".").concat(key), next.getValue());
                }
                targetNode.set(rawKey, targetTermsNode);
                must.add(targetNode);
            });
        });
        parentBool.put(type, must);
        boolNode.put("bool", parentBool);
        return boolNode;
    }

    private ObjectNode constructReplacePrefixQuery(String prefix, Map<String, Object> parentChild, Map<String, JsonNode> terms, String type) {
        ObjectNode parentBool = mapper.createObjectNode();
        ObjectNode boolNode = mapper.createObjectNode();
        ArrayNode must = mapper.createArrayNode();
        parentChild.entrySet().forEach(entry -> {
            ObjectNode objectNode = (ObjectNode) terms.get(entry.getKey());
            objectNode.fields().forEachRemaining(e -> {
                String rawKey = e.getKey();
                JsonNode termNode = e.getValue();
                ObjectNode targetNode = mapper.createObjectNode();
                ObjectNode targetTermsNode = mapper.createObjectNode();
                Iterator<Entry<String, JsonNode>> fields = termNode.fields();
                while (fields.hasNext()) {
                    Entry<String, JsonNode> next = fields.next();
                    String key = next.getKey();
                    String targetName = null;
                    if (key.contains(".")) {
                        String[] split = key.split("\\.");
                        String fieldName = split[1];
                        targetName = prefix.concat(".").concat(fieldName);
                    } else {
                        //TODO
                        if (!rawKey.equalsIgnoreCase("exists")) {
                            targetName = prefix.concat(".").concat(key);
                        } else {
                            targetName = key;
                        }
                    }
                    targetTermsNode.put(targetName, next.getValue());
                }
                targetNode.set(rawKey, targetTermsNode);
                must.add(targetNode);
            });
        });
        parentBool.put(type, must);
        boolNode.put("bool", parentBool);
        return boolNode;
    }

    private ObjectNode constructRelatedQuery(Map<String, Object> parentChild, Map<String, JsonNode> terms, String type) {
        ObjectNode parentBool = mapper.createObjectNode();
        ObjectNode boolNode = mapper.createObjectNode();
        ArrayNode must = mapper.createArrayNode();
        parentChild.entrySet().forEach(entry -> {
            ObjectNode objectNode = (ObjectNode) terms.get(entry.getKey());
            objectNode.fields().forEachRemaining(e -> {
                String rawKey = e.getKey();
                if (rawKey.equalsIgnoreCase("exists")) {
                    JsonNode termNode = e.getValue();
                    ObjectNode targetNode = mapper.createObjectNode();
                    ObjectNode targetTermsNode = mapper.createObjectNode();
                    Iterator<Entry<String, JsonNode>> fields = termNode.fields();
                    while (fields.hasNext()) {
                        Entry<String, JsonNode> next = fields.next();
                        String value = next.getValue().asText();
                        if (value.contains(".")) {
                            value = value.substring(value.indexOf(".") + 1);
                        }
                        targetTermsNode.put(next.getKey(), value);
                    }
                    targetNode.set(rawKey, targetTermsNode);
                    must.add(targetNode);
                } else {
                    JsonNode termNode = e.getValue();
                    ObjectNode targetNode = mapper.createObjectNode();
                    ObjectNode targetTermsNode = mapper.createObjectNode();
                    Iterator<Entry<String, JsonNode>> fields = termNode.fields();
                    while (fields.hasNext()) {
                        Entry<String, JsonNode> next = fields.next();
                        String key = next.getKey();
                        if (key.contains(".")) {
                            key = key.substring(key.indexOf(".") + 1);
                        }
                        targetTermsNode.put(key, next.getValue());
                    }
                    targetNode.set(rawKey, targetTermsNode);
                    must.add(targetNode);
                }
            });
        });
        parentBool.put(type, must);
        boolNode.put("bool", parentBool);
        return boolNode;
    }

    /**
     * 解析过滤条件
     *
     * @param terms
     * @param term
     **/
    private void parseTerms(EntityClassGroup group, Map<String, Object> selfTerms, Map<String, Object> relatedTerms, Map<String, JsonNode> terms, JsonNode term) {
        Iterator<Entry<String, JsonNode>> fields = term.fields();
        while (fields.hasNext()) {
            Entry<String, JsonNode> result = fields.next();
            JsonNode termTuple = result.getValue();
            String operation = result.getKey();
            Supplier<Iterator<Entry<String, JsonNode>>> termFields = termTuple::fields;
            Iterator<Entry<String, JsonNode>> entryIterator = termFields.get();
            Entry<String, JsonNode> next = entryIterator.next();

            if ("exists".equalsIgnoreCase(operation)) {
                //TODO
                if (!next.getValue().asText().contains(".")) {
                    terms.put(next.getValue().asText(), term);
                    selfTerms.put(next.getValue().asText(), next.getValue());
                } else {
                    terms.put(next.getValue().asText(), term);
                    relatedTerms.put(next.getValue().asText(), next.getValue());
                }
            } else {
                if (!next.getKey().contains(".")) {
                    terms.put(next.getKey(), term);
                    selfTerms.put(next.getKey(), next.getValue());
                } else {
                    terms.put(next.getKey(), term);
                    relatedTerms.put(next.getKey(), next.getValue());
                }
            }
        }
    }

    /**
     * 构建elasticsearch dsl过滤查询条件
     *
     * @param parentChild
     * @param terms
     * @return
     **/
    private ObjectNode constructQuery(Map<String, Object> parentChild, Map<String, JsonNode> terms, String type) {
        ObjectNode parentBool = mapper.createObjectNode();
        ObjectNode boolNode = mapper.createObjectNode();
        ArrayNode must = mapper.createArrayNode();
        parentChild.entrySet().forEach(entry -> {
            must.add(terms.get(entry.getKey()));
        });
        parentBool.put(type, must);
        boolNode.put("bool", parentBool);
        return boolNode;
    }

    private ObjectNode constructExistsCreateTimeQuery() {
        ObjectNode parentBool = mapper.createObjectNode();
        ObjectNode boolNode = mapper.createObjectNode();
        ArrayNode must = mapper.createArrayNode();
        ObjectNode exists = mapper.createObjectNode();
        //{"exists":{"field":"create_time"}
        exists.put("exists", mapper.createObjectNode().put("field", "create_time"));
        must.add(exists);
        parentBool.put("must", must);
        boolNode.put("bool", parentBool);
        return boolNode;
    }

    private Enumerable<Object> aggregate(
            ElasticsearchTransport targetTransport,
            EntityClassGroup group,
            List<String> ops,
            List<Map.Entry<String, Class>> fields,
            List<Map.Entry<String, RelFieldCollation.Direction>> sort,
            List<String> groupBy,
            List<Map.Entry<String, String>> aggregations,
            Map<String, String> mapping,
            Map<String, String> rawMapping,
            Long offset, Long fetch, boolean ignoreJoin) throws IOException {

        if ((null != groupBy && !groupBy.isEmpty()) && offset != null) {
            String message = "Currently ES doesn't support generic pagination "
                    + "with aggregations. You can still use LIMIT keyword (without OFFSET). "
                    + "For more details see https://github.com/elastic/elasticsearch/issues/4915";
            throw new IllegalStateException(message);
        }
        ObjectNode query = mapper.createObjectNode();
        // manually parse into JSON from previously concatenated strings
        for (String op : ops) {
            if (contextService.getAll().get("join_query") != null && (Boolean) contextService.getAll().get("join_query")) {
                /**根据filter过滤条件拆成has_parent与has_child查询方式 过滤parent-child index索引表**/
                if (mapper.readTree(op).get("query") != null && !ignoreJoin) {
                    generateJoinModelQuery(group, query, op);
                } else {
                    query.setAll((ObjectNode) mapper.readTree(op));
                }
            } else {
                query.setAll((ObjectNode) mapper.readTree(op));
            }
        }
        // remove / override attributes which are not applicable to aggregations
        query.put("_source", false);
        query.put("size", 0);
        query.remove("script_fields");
        // set _source = false and size = 0, `FetchPhase` would still be executed
        // to fetch the metadata fields and visit the Lucene stored_fields,
        // which would lead to performance declined dramatically.
        // `stored_fields = _none` can prohibit such behavior entirely
        query.put("stored_fields", "_none_");

        // allows to detect aggregation for count(*)
        final Predicate<Map.Entry<String, String>> isCountStar = e -> e.getValue()
                .contains("\"" + ElasticsearchConstants.ID + "\"");

        // list of expressions which are count(*)
        final Set<String> countAll = aggregations.stream()
                .filter(isCountStar)
                .map(Map.Entry::getKey).collect(Collectors.toSet());

        final Map<String, String> fieldMap = new HashMap<>();

        // due to ES aggregation format. fields in "order by" clause should go first
        // if "order by" is missing. order in "group by" is un-important
        final Set<String> orderedGroupBy = new LinkedHashSet<>();
        orderedGroupBy.addAll(sort.stream().map(Map.Entry::getKey).collect(Collectors.toList()));
        orderedGroupBy.addAll(groupBy);

        // construct nested aggregations node(s)
        ObjectNode parent = query.with(AGGREGATIONS);
        for (String name : orderedGroupBy) {
            final String aggName = "g_" + name;
            fieldMap.put(aggName, name);

            final ObjectNode section = parent.with(aggName);
            final ObjectNode terms = section.with("terms");
            terms.put("field", name);

            transport.mapping.missingValueFor(name).ifPresent(m -> {
                // expose missing terms. each type has a different missing value
                terms.set("missing", m);
            });

            if (fetch != null) {
                terms.put("size", fetch);
            }

            sort.stream().filter(e -> e.getKey().equals(name)).findAny()
                    .ifPresent(s ->
                            terms.with("order")
                                    .put("_key", s.getValue().isDescending() ? "desc" : "asc"));

            parent = section.with(AGGREGATIONS);
        }

        // simple version for queries like "select count(*), max(col1) from table" (no GROUP BY cols)
        if (!groupBy.isEmpty() || !aggregations.stream().allMatch(isCountStar)) {
            for (Map.Entry<String, String> aggregation : aggregations) {
                JsonNode value = mapper.readTree(aggregation.getValue());
                parent.set(aggregation.getKey(), value);
            }
        }

        final Consumer<JsonNode> emptyAggRemover = new Consumer<JsonNode>() {
            @Override
            public void accept(JsonNode node) {
                if (!node.has(AGGREGATIONS)) {
                    node.elements().forEachRemaining(this);
                    return;
                }
                JsonNode agg = node.get(AGGREGATIONS);
                if (agg.size() == 0) {
                    ((ObjectNode) node).remove(AGGREGATIONS);
                } else {
                    this.accept(agg);
                }
            }
        };

        // cleanup query. remove empty AGGREGATIONS element (if empty)
        emptyAggRemover.accept(query);

        // This must be set to true or else in 7.X and 6/7 mixed clusters
        // will return lower bounded count values instead of an accurate count.
        if (groupBy.isEmpty()
                && version.elasticVersionMajor() >= ElasticsearchVersion.ES6.elasticVersionMajor()) {
            query.put("track_total_hits", true);
        }

        ElasticsearchJson.Result res = targetTransport.search(Collections.emptyMap()).apply(query);

        final List<Map<String, Object>> result = new ArrayList<>();
        if (res.aggregations() != null) {
            // collect values
            ElasticsearchJson.visitValueNodes(res.aggregations(), m -> {
                // using 'Collectors.toMap' will trigger Java 8 bug here
                Map<String, Object> newMap = new LinkedHashMap<>();
                for (String key : m.keySet()) {
                    newMap.put(fieldMap.getOrDefault(key, key), m.get(key));
                }
                result.add(newMap);
            });
        } else {
            // probably no group by. add single result
            result.add(new LinkedHashMap<>());
        }

        // elastic exposes total number of documents matching a query in "/hits/total" path
        // this can be used for simple "select count(*) from table"
        final long total = res.searchHits().total().value();
        contextService.getAll().put("show_count", total);
        if (groupBy.isEmpty()) {
            // put totals automatically for count(*) expression(s), unless they contain group by
            for (String expr : countAll) {
                result.forEach(m -> m.put(expr, total));
            }
        }

        final Function1<ElasticsearchJson.SearchHit, Object> getter =
                ElasticsearchEnumerators.getter(fields, rawMapping, ImmutableMap.copyOf(mapping));

        ElasticsearchJson.SearchHits hits =
                new ElasticsearchJson.SearchHits(res.searchHits().total(), result.stream()
                        .map(r -> new ElasticsearchJson.SearchHit("_id", r, null, null))
                        .collect(Collectors.toList()));
        //hits.hits().remove(0);
        return Linq4j.asEnumerable(hits.hits()).select(getter);
    }

    private RelDataType fieldTypeToRelDataType(RelDataTypeFactory relDataTypeFactory, Class type) {
        return relDataTypeFactory.createJavaType(type);
    }

    @Override
    public RelDataType getRowType(RelDataTypeFactory relDataTypeFactory) {
        String profile = fetcher.getProfile(contextService.getAll());
        Optional<IEntityClass> targetClass = engine.loadByCode(entityCode,
                profile);
        IEntityClass iEntityClass = targetClass.get();
        EntityClassGroup describe = engine.describe(targetClass.get(), profile);
        Collection<IEntityField> allFields = describe.getAllFields();
        final List<Map.Entry<String, RelDataType>> names = allFields.stream().map(
                        x -> Pair.of(x.name().replace(".", "_"),
                                fieldTypeToRelDataType(relDataTypeFactory, x.type().getJavaType())))
                .collect(Collectors.toList());
        allFields.stream().forEach(x -> {
            if (x.type() == FieldType.STRINGS) {
                names.add(Pair.of(x.name().replace(".", "_").concat("@raw"),
                        fieldTypeToRelDataType(relDataTypeFactory, x.type().getJavaType())));
            }
        });
        /**当有重复relations对象依赖，对关联关系不一样时，只取一遍relations对象的元数据信息，否则calcite会抛SqlValidatorException异常信息**/
        Map<String, List<Pair<String, RelDataType>>> relationSet = new HashMap<>();
        /**加入宽表所有的依赖字段，避免SQL解析验证元数据时抛出找不到表字段**/
        for (IRelation iRelation : iEntityClass.relations()) {
            Optional<IEntityClass> iRelationEntityClassGroup = engine.load(String.valueOf(iRelation.getEntityClassId()), profile);
            IEntityClass iRelationEntityClass = iRelationEntityClassGroup.get();
            Collection<IEntityField> relationAllFields = engine.describe(iRelationEntityClass, profile).getAllFields();
            List<Pair<String, RelDataType>> relationNames = relationAllFields.stream().map(
                            x -> Pair.of(iRelation.getName().concat(".").concat(x.name().replace(".", "_")),
                                    fieldTypeToRelDataType(relDataTypeFactory, x.type().getJavaType())))
                    .collect(Collectors.toList());
            relationAllFields.stream().forEach(x -> {
                if (x.type() == FieldType.STRINGS) {
                    relationNames.add(Pair.of(iRelation.getName().concat(".").concat(x.name()).replace(".", "_").concat("@raw"),
                            fieldTypeToRelDataType(relDataTypeFactory, x.type().getJavaType())));
                }
            });
            relationSet.put(iRelationEntityClass.code(), relationNames);
        }
        for (List<Pair<String, RelDataType>> relationValue : relationSet.values()) {
            names.addAll(relationValue);
        }
        RelDataType structType = relDataTypeFactory.createStructType(names);
        return structType;
    }

    @Override
    public String toString() {
        return "ElasticsearchTable{" + indexName + "}";
    }

    @Override
    public <T> Queryable<T> asQueryable(QueryProvider queryProvider, SchemaPlus schema,
                                        String tableName) {
        return new ElasticsearchQueryable<>(queryProvider, schema, this, tableName);
    }

    @Override
    public RelNode toRel(RelOptTable.ToRelContext context, RelOptTable relOptTable) {
        final RelOptCluster cluster = context.getCluster();
        return new ElasticsearchTableScan(cluster, cluster.traitSetOf(ElasticsearchRel.CONVENTION),
                relOptTable, this, null);
    }

    /**
     * Implementation of {@link Queryable} based on a {@link ElasticsearchTable}.
     *
     * @param <T> element type
     */
    public static class ElasticsearchQueryable<T> extends AbstractTableQueryable<T> {

        ElasticsearchQueryable(QueryProvider queryProvider, SchemaPlus schema,
                               ElasticsearchTable table, String tableName) {
            super(queryProvider, schema, table, tableName);
        }

        @Override
        public Enumerator<T> enumerator() {
            return null;
        }

        private ElasticsearchTable getTable() {
            return (ElasticsearchTable) table;
        }

        /**
         * Called via code-generation.
         *
         * @param ops    list of queries (as strings)
         * @param fields projection
         * @return result as enumerable
         * @see ElasticsearchMethod#ELASTICSEARCH_QUERYABLE_FIND
         */
        @SuppressWarnings("UnusedDeclaration")
        public Enumerable<Object> find(List<String> ops,
                                       List<Map.Entry<String, Class>> fields,
                                       List<Map.Entry<String, RelFieldCollation.Direction>> sort,
                                       List<String> groupBy,
                                       List<Map.Entry<String, String>> aggregations,
                                       Map<String, String> mappings,
                                       Map<String, String> rawMapping,
                                       Long offset, Long fetch, RelNode rawTree, DataContext dataContext) {
            try {
                long currentTimeMillis = System.currentTimeMillis();
                System.out.println("Come into ElasticSearch Table " + currentTimeMillis);
                Enumerable<Object> objects = getTable().find(ops, fields, sort, groupBy, aggregations, mappings, rawMapping, offset, fetch, rawTree, dataContext);
                System.out.println("Query cost is " + (System.currentTimeMillis() - currentTimeMillis));
                return objects;
            } catch (Throwable e) {
                throw new RuntimeException("Failed to query " + getTable().indexName, e);
            }
        }
    }
}
