/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.xforceplus.ultraman.adapter.elasticsearch;

import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.adapter.elasticsearch.query.utils.ParseSqlNodeUtils;

import java.util.AbstractList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.calcite.DataContext;
import org.apache.calcite.adapter.enumerable.EnumerableRel;
import org.apache.calcite.adapter.enumerable.EnumerableRelImplementor;
import org.apache.calcite.adapter.enumerable.JavaRowFormat;
import org.apache.calcite.adapter.enumerable.PhysType;
import org.apache.calcite.adapter.enumerable.PhysTypeImpl;
import org.apache.calcite.linq4j.tree.BlockBuilder;
import org.apache.calcite.linq4j.tree.Expression;
import org.apache.calcite.linq4j.tree.Expressions;
import org.apache.calcite.linq4j.tree.MethodCallExpression;
import org.apache.calcite.plan.ConventionTraitDef;
import org.apache.calcite.plan.RelOptCluster;
import org.apache.calcite.plan.RelOptCost;
import org.apache.calcite.plan.RelOptPlanner;
import org.apache.calcite.plan.RelTraitSet;
import org.apache.calcite.plan.hep.HepPlanner;
import org.apache.calcite.plan.hep.HepProgram;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.convert.ConverterImpl;
import org.apache.calcite.rel.metadata.RelMetadataQuery;
import org.apache.calcite.rel.rel2sql.RelToSqlConverter;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.sql.dialect.CalciteSqlDialect;
import org.apache.calcite.sql.dialect.MysqlSqlDialect;
import org.apache.calcite.tools.FrameworkConfig;
import org.apache.calcite.tools.RelBuilder;
import org.apache.calcite.util.BuiltInMethod;
import org.apache.calcite.util.Pair;
import org.apache.calcite.util.Util;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Relational expression representing a scan of a table in an Elasticsearch data source.
 */
public class ElasticsearchToEnumerableConverter extends ConverterImpl implements EnumerableRel {

    @Autowired
    private ContextService contextService;

    ElasticsearchToEnumerableConverter(RelOptCluster cluster, RelTraitSet traits,
                                       RelNode input) {
        super(cluster, ConventionTraitDef.INSTANCE, traits, input);
    }

    /**
     * E.g. {@code constantArrayList("x", "y")} returns "Arrays.asList('x', 'y')".
     *
     * @param values list of values
     * @param clazz  runtime class representing each element in the list
     * @param <T>    type of elements in the list
     * @return method call which creates a list
     */
    private static <T> MethodCallExpression constantArrayList(List<T> values, Class clazz) {
        return Expressions.call(BuiltInMethod.ARRAYS_AS_LIST.method,
                Expressions.newArrayInit(clazz, constantList(values)));
    }

    /**
     * E.g. {@code constantList("x", "y")} returns {@code {ConstantExpression("x"), ConstantExpression("y")}}.
     *
     * @param values list of elements
     * @param <T>    type of elements inside this list
     * @return list of constant expressions
     */
    private static <T> List<Expression> constantList(List<T> values) {
        return values.stream().map(Expressions::constant).collect(Collectors.toList());
    }

    @Override
    public RelNode copy(RelTraitSet traitSet, List<RelNode> inputs) {
        return new ElasticsearchToEnumerableConverter(getCluster(), traitSet, sole(inputs));
    }

    @Override
    public @Nullable RelOptCost computeSelfCost(RelOptPlanner planner,
                                                RelMetadataQuery mq) {
        return super.computeSelfCost(planner, mq).multiplyBy(.1);
    }

    @Override
    public Result implement(EnumerableRelImplementor relImplementor, Prefer prefer) {
        long start = System.currentTimeMillis();
        final BlockBuilder block = new BlockBuilder();
        final ElasticsearchRel.Implementor implementor = new ElasticsearchRel.Implementor();
        implementor.visitChild(0, getInput());
        final RelDataType rowType = getRowType();
        final PhysType physType = PhysTypeImpl.of(relImplementor.getTypeFactory(), rowType,
                prefer.prefer(JavaRowFormat.ARRAY));
        final Expression fields = block.append("fields",
                constantArrayList(
                        Pair.zip(ElasticsearchRules.elasticsearchFieldNames(rowType),
                                new AbstractList<Class>() {
                                    @Override
                                    public Class get(int index) {
                                        return physType.fieldClass(index);
                                    }

                                    @Override
                                    public int size() {
                                        return rowType.getFieldCount();
                                    }
                                }),
                        Pair.class));
        final Expression table = block.append("table",
                implementor.table
                        .getExpression(ElasticsearchTable.ElasticsearchQueryable.class));
        final Expression ops = block.append("ops", Expressions.constant(implementor.list));
        final Expression sort = block.append("sort", constantArrayList(implementor.sort, Pair.class));
        final Expression groupBy = block.append("groupBy", Expressions.constant(implementor.groupBy));
        final Expression aggregations = block.append("aggregations",
                constantArrayList(implementor.aggregations, Pair.class));

        final Expression mappings = block.append("mappings",
                Expressions.constant(implementor.expressionItemMap));

        final Expression offset = block.append("offset", Expressions.constant(implementor.offset));
        final Expression fetch = block.append("fetch", Expressions.constant(implementor.fetch));
        final Expression tree = block.append("tree", relImplementor.stash(getInput(), RelNode.class));
        final Expression rawMapping = block.append("rawMapping", relImplementor.stash(implementor.projectMapping, Map.class));
        final Expression dataContext = block.append("dataCtx", DataContext.ROOT);
        Expression enumerable = block.append("enumerable",
                Expressions.call(table, ElasticsearchMethod.ELASTICSEARCH_QUERYABLE_FIND.method, ops,
                        fields, sort, groupBy, aggregations, mappings, rawMapping, offset, fetch, tree, dataContext));
        block.add(Expressions.return_(null, enumerable));
        Result result = relImplementor.result(physType, block.toBlock());
        System.out.println("Converter Table " + (System.currentTimeMillis() - start));
        return result;
    }
}
