package com.xforceplus.ultraman.adapter.elasticsearch.query;

import com.xforceplus.metadata.schema.runtime.MetadataEngine;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.adapter.elasticsearch.ElasticsearchSchemaFactory;
import com.xforceplus.ultraman.adapter.elasticsearch.query.utils.ElasticSearchSqlConverter;
import com.xforceplus.ultraman.adapter.elasticsearch.query.utils.ParseSqlNodeUtils;
import com.xforceplus.ultraman.adapter.elasticsearch.service.ManageBocpMetadataService;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.MysqlSqlDialectEx;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.select.SelectConfig;
import com.xforceplus.ultraman.sdk.core.calcite.oqs.DataQueryProvider;
import com.xforceplus.ultraman.sdk.core.config.ExecutionConfig;
import com.xforceplus.ultraman.sdk.core.datasource.route.TransportExecutor;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import io.micrometer.core.annotation.Timed;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.DataContext;
import org.apache.calcite.jdbc.CalciteConnection;
import org.apache.calcite.rel.RelFieldCollation;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.core.AggregateCall;
import org.apache.calcite.rel.hint.RelHint;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.StructKind;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.schema.Schema;
import org.apache.calcite.schema.SchemaPlus;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.sql.SqlOrderBy;
import org.apache.calcite.sql.SqlSelect;
import org.apache.calcite.sql.parser.SqlParseException;
import org.apache.calcite.util.Pair;
import org.apache.calcite.util.Util;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;

import javax.sql.DataSource;
import java.sql.*;
import java.util.*;

@Slf4j
public class ElasticSearchQueryProvider implements
        DataQueryProvider {

    @Lazy
    @Qualifier("elasticSearchDS")
    @Autowired
    private DataSource esDataSource;

    @Autowired
    private ContextService contextService;
    @Autowired
    private EntityClassEngine engine;
    @Lazy
    @Autowired
    private ExecutionConfig executionConfig;

    @Autowired
    private ManageBocpMetadataService manageBocpMetadataService;

    public ElasticSearchQueryProvider() {
    }

    public void setEsDataSource(DataSource esDataSource) {
        this.esDataSource = esDataSource;
    }

    public void setContextService(ContextService contextService) {
        this.contextService = contextService;
    }

    public void setEngine(EntityClassEngine engine) {
        this.engine = engine;
    }

    public void setExecutionConfig(ExecutionConfig executionConfig) {
        this.executionConfig = executionConfig;
    }

    @Override
    public QueryProviderType type() {
        return QueryProviderType.INDEX;
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"query", "es"}
    )
    @Override
    public List<Object> query(String app, IEntityClass entityClass, String profile, RelDataType type,
                              List<RexNode> ops, List<Map.Entry<String, Tuple2<StructKind, Class>>> fields,
                              List<Pair<RexNode, String>> projects,
                              List<Map.Entry<String, RelFieldCollation.Direction>> sort, Long offset, Long fetch,
                              List<String> groupBy, List<AggregateCall> aggs, List<RelHint> hints, RelNode rawTree,
                              DataContext dataContext) {

        contextService.getAll().put("invocation", "index");
        
        Tuple2<String, String> searchIndex = manageBocpMetadataService.getSearchSegmentIndex(profile, entityClass.code());
        if (searchIndex != null && (searchIndex._2 == null || searchIndex._2.equalsIgnoreCase("$TEMP$"))) {
            return Collections.emptyList();
        }
      
        SelectConfig selectConfig = ParseSqlNodeUtils.getSelectConfig(profile, type, ops, fields, projects, sort,
                offset, fetch, groupBy, aggs, hints, rawTree, dataContext);
        try (Connection connection = esDataSource.getConnection()) {
            /**校验SQL是否超过join上限，超过抛出异常**/
            ElasticCustomShuttle elasticCustomShuttle = new ElasticCustomShuttle();
            rawTree.accept(elasticCustomShuttle);
            ElasticSearchSqlConverter elasticSearchSqlConverter = new ElasticSearchSqlConverter(engine, contextService, elasticCustomShuttle);
            SqlNode sqlNode = elasticSearchSqlConverter.oqsRelNodeConverterElasticSql(entityClass, selectConfig);
            String sql = ParseSqlNodeUtils.converterSqlString(sqlNode);
            //contextService.getAll().put("query", sql);
            log.info(String.format("elasticsearch query engine execute sql:%s.", sql));
            contextService.getAll().put("hasCount", hasCount(selectConfig));
            List<Object> retList;
            try (Statement preparedStatement = connection.createStatement()) {
                ResultSet resultSet = preparedStatement.executeQuery(sql);
                retList = new ArrayList<>();
                while (resultSet.next()) {
                    List<Object> row = new ArrayList<>();
                    fields.forEach(f -> {
                        String key = f.getKey();
                        if (!key.startsWith("_") && key.contains(".")) {
                            key = f.getKey().replace(".", "_");
                        }
                        
                        //handle subItem
                        if(key.startsWith("_")) {
                            key = key.substring(1);
                        }
                        Object rs = MasterStorageHelper.getRs(resultSet, key, f.getValue()._2);
                        row.add(rs);
                    });
                    if (row.size() > 1) {
                        retList.add(row.toArray());
                    } else {
                        retList.add(row.get(0));
                    }
                }
            }
            return retList;
        } catch (Throwable throwable) {
            log.error("elasticsearch execute query failed!,cause by {}", throwable);
            throw new RuntimeException(throwable);
        }
    }

    /***
     *检索SelectConfig是否带有 hint show_count标识
     * @param selectConfig
     * @return
     */
    private boolean hasCount(SelectConfig selectConfig) {
        return selectConfig.getHints().stream()
                .anyMatch(x -> x.hintName.equalsIgnoreCase("show_count"));
    }
}

