package com.xforceplus.ultraman.adapter.elasticsearch.service.impl;


import com.xforceplus.ultraman.adapter.elasticsearch.service.IndexOperation;
import com.xforceplus.ultraman.adapter.elasticsearch.service.ManageBocpMetadataService;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.CommonProperty;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.FieldMappingType;
import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.SettingProperties;
import com.xforceplus.ultraman.adapter.elasticsearch.service.entity.FieldMapping;
import com.xforceplus.ultraman.adapter.elasticsearch.service.utils.ElasticSearchMappingBuildUtils;
import com.xforceplus.ultraman.adapter.elasticsearch.service.utils.ThreadPoolExecutorUtils;
import com.xforceplus.ultraman.adapter.elasticsearch.transport.ElasticsearchTransportExecutor;
import com.xforceplus.ultraman.adapter.elasticsearch.utils.DynamicConfigUtils;
import com.xforceplus.ultraman.cdc.adapter.CDCFilter;
import com.xforceplus.ultraman.cdc.adapter.EngineAdapterService;
import com.xforceplus.ultraman.cdc.adapter.IndexUpsertBeforeCallBack;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.entity.IRelation;
import com.xforceplus.ultraman.sdk.core.datasource.route.TransportExecutor;
import com.xforceplus.ultraman.sdk.core.datasource.route.dynamic.config.DynamicConfig;
import io.vavr.Tuple2;
import io.vavr.Tuple4;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.indices.CreateIndexRequest;
import org.elasticsearch.client.indices.CreateIndexResponse;
import org.elasticsearch.client.indices.GetIndexRequest;
import org.elasticsearch.client.indices.GetMappingsRequest;
import org.elasticsearch.client.indices.GetMappingsResponse;
import org.elasticsearch.client.indices.PutMappingRequest;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.reindex.BulkByScrollResponse;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.join.query.JoinQueryBuilders;
import org.elasticsearch.join.query.ParentIdQueryBuilder;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;


/**
 * @program: ultraman-oqsengine-plus
 * @ClassName ElasticSearchServiceImpl
 * @description:
 * @author: WanYi
 * @create: 2023-05-15 16:36
 * @Version 1.0
 **/
@Slf4j
public class ElasticSearchServiceImpl implements EngineAdapterService {

    @Autowired
    public EntityClassEngine entityClassEngine;
    @Autowired
    private DynamicConfig dynamicConfig;
    @Autowired(required = false)
    private List<CDCFilter> filters = new ArrayList<>();
    private long writeTimeOut = 50;
    @Autowired(required = false)
    private List<IndexUpsertBeforeCallBack> beforeCallBacks = new ArrayList<>();

    private ForkJoinPool pool = new ForkJoinPool(4);
    
    @Autowired
    private ManageBocpMetadataService manageBocpMetadataService;
    
    @Autowired
    private TransportExecutor elasticsearchTransportExecutor;


    private volatile double sync_time = 0;
    private volatile long sync_docs = 0;

    /**
     * 分片数
     **/
    private int shards;
    /**
     * 副本数
     **/
    private int replicas;

    private int retryOnConflict;

    private int threadHandleBatch = 1000;

    private ThreadPoolExecutor executor = ThreadPoolExecutorUtils.executor;

    public ElasticSearchServiceImpl(int shards, int replicas, int threadHandleBatch, int retryOnConflict) {
        this.shards = shards;
        this.replicas = replicas;
        this.threadHandleBatch = threadHandleBatch;
        this.retryOnConflict = retryOnConflict;
    }

    public void setDynamicConfig(DynamicConfig dynamicConfig) {
        this.dynamicConfig = dynamicConfig;
    }

    public void setFilters(List<CDCFilter> filters) {
        this.filters = filters;
    }

    public void setBeforeCallBacks(List<IndexUpsertBeforeCallBack> beforeCallBacks) {
        this.beforeCallBacks = beforeCallBacks;
    }


    public void setManageBocpMetadataService(ManageBocpMetadataService manageBocpMetadataService) {
        this.manageBocpMetadataService = manageBocpMetadataService;
    }

    public void setElasticsearchTransportExecutor(TransportExecutor elasticsearchTransportExecutor) {
        this.elasticsearchTransportExecutor = elasticsearchTransportExecutor;
    }

    public void setEntityClassEngine(EntityClassEngine entityClassEngine) {
        this.entityClassEngine = entityClassEngine;
    }


    /**
     * 根据信息自动创建索引与mapping 构建mapping描述
     *
     * @param idxName   索引名称
     * @param allFields 字段信息
     * @param profile   租户信息
     */



    /***
     * @param allFields
     * @param prefixName
     * @return
     * @throws IOException
     */
    private XContentBuilder getxContentBuilder(Collection<IEntityField> allFields, String prefixName)
            throws IOException {
        XContentBuilder properties = XContentFactory.jsonBuilder().startObject()
                .field("dynamic", "true") //注释掉这条命令，不能动态扩展字段
                .field("properties").startObject();
        getxContentBuilder(allFields, properties, prefixName);
        properties.endObject();
        properties.endObject();
        return properties;
    }

    /***
     * @return
     * @throws IOException
     */
    private XContentBuilder getxContentBuilder(Set<String> relations, String entityClassCode)
            throws IOException {
        XContentBuilder properties = XContentFactory.jsonBuilder().startObject()
                .field("dynamic", "true") //注释掉这条命令，不能动态扩展字段
                .field("properties").startObject();

        properties = properties.startObject(entityClassCode.concat("_join")).field("type", "join")
                .startObject("relations");
        if (relations != null && relations.size() > 0) {
            properties = properties.field(entityClassCode, relations.toArray());
        }
        properties.endObject();
        properties.endObject();
        properties.endObject();
        properties.endObject();
        return properties;
    }

    /**
     * * 构建XContentBuilder对象字段（mapping映射字段名字 字段类型 IK分词 analyzer）
     *
     * @param allFields
     * @param properties
     * @param prefixName
     **/
    private XContentBuilder getxContentBuilder(Collection<IEntityField> allFields,
                                               XContentBuilder properties, String prefixName) throws IOException {
        for (FieldMapping field : ElasticSearchMappingBuildUtils.getFieldInfos(allFields)) {
            String filedName = field.getField().contains(".") ? field.getField().replace(".", "_")
                    : field.getField();
            filedName = StringUtils.isNotEmpty(prefixName) ? prefixName.concat(".").concat(filedName) : filedName;
            if (StringUtils.equalsIgnoreCase(field.getType(), FieldMappingType.TEXT.getType())) {
                properties = properties.startObject(filedName)
                        .field("type", FieldMappingType.TEXT.getType())
                        .field("analyzer", field.getAnalyzer()).field("fielddata", "true").endObject();
            } else if (StringUtils.equalsIgnoreCase(field.getType(),
                    FieldMappingType.KEYWORD.getType())) {
                properties = properties.startObject(filedName)
                        .field("type", FieldMappingType.KEYWORD.getType()).endObject();
                if (field.getOriginField().type() == FieldType.STRINGS) {
                    //should add a new field for store raw value
                    properties = properties.startObject(filedName.concat("@raw"))
                            .field("type", FieldMappingType.TEXT.getType()).endObject();
                }
            } else if (StringUtils.equalsIgnoreCase(field.getType(), FieldMappingType.LONG.getType())) {
                properties = properties.startObject(filedName)
                        .field("type", FieldMappingType.LONG.getType()).endObject();
            } else if (StringUtils.equalsIgnoreCase(field.getType(), FieldMappingType.DOUBLE.getType())) {
                properties = properties.startObject(filedName)
                        .field("type", FieldMappingType.DOUBLE.getType()).endObject();
            } else if (StringUtils.equalsIgnoreCase(field.getType(),
                    FieldMappingType.BOOLEAN.getType())) {
                properties = properties.startObject(filedName)
                        .field("type", FieldMappingType.BOOLEAN.getType()).endObject();
            }
        }
        return properties;
    }

    @Override
    public List<CDCFilter> getFilters() {
        return filters;
    }

    @Override
    public List<IndexUpsertBeforeCallBack> getCallBacks() {
        return beforeCallBacks;
    }


    @Override
    public boolean initMetadataInitStatus() {
        return manageBocpMetadataService.getLoadFinish();
    }

    @Override
    public boolean sync(List<OqsEngineEntity> oqsEngineEntities) {
        List<Future> futures = new ArrayList<>();
        long startTime = System.currentTimeMillis();
        for (int i = 0; i <= oqsEngineEntities.size() / threadHandleBatch; i++) {
            int endSubIndex = i * threadHandleBatch + threadHandleBatch;
            endSubIndex = endSubIndex > oqsEngineEntities.size() ? oqsEngineEntities.size() : endSubIndex;
            submitThreadHandler(oqsEngineEntities.subList(i * threadHandleBatch, endSubIndex), futures);
        }
        for (Future future : futures) {
            try {
                boolean status = (boolean) future.get(writeTimeOut, TimeUnit.SECONDS);
                if (!status) {
                    return false;
                }
            } catch (Exception e) {
                log.error(e.getMessage());
                return false;
            }
        }
        long endTime = (System.currentTimeMillis() - startTime);
        this.sync_time += endTime;
        this.sync_docs += oqsEngineEntities.size();
        log.info("elastc_sync(单位毫秒)-------elastic总共同步时间time:({})----总共同步记录:docs({})-----当前批次同步数docs:({}) 当前批次处理时间time:({})", sync_time, sync_docs,
                oqsEngineEntities.size(), endTime);
        return true;
    }

    /**
     * 对数据按批次大小进行切分，提交至线程处理
     *
     * @param futures
     * @param batchHandlerEntities
     **/
    private void submitThreadHandler(List<OqsEngineEntity> batchHandlerEntities, List<Future> futures) {
        Future<Boolean> future = executor.submit(() -> {
            try {
                List threadBatchDeletes = new ArrayList<>();
                List threadBatchUpserts = new ArrayList<>();
                batchHandlerEntities.forEach(entity -> {
                    if (entity.isDeleted()) {
                        threadBatchDeletes.add(entity);
                    } else {
                        threadBatchUpserts.add(entity);
                    }
                });
                if (threadBatchUpserts.size() > 0) {
                    if (!saveOrUpdate(threadBatchUpserts)) {
                        return false;
                    }
                }
                if (threadBatchDeletes.size() > 0) {
                    if (!deleteBatch(threadBatchDeletes)) {
                        return false;
                    }
                }
            } catch (Exception e) {
                log.error("FAILURE,elasticsearch submitThreadHandler method execute failed!,cause by:{}", e.getMessage());
                return false;
            }
            return true;
        });
        futures.add(future);
    }

    /***
     * 批量插入或更新
     * @param oqsEngineEntitys
     * @return
     * @throws IOException
     * @throws CloneNotSupportedException
     */
    public boolean saveOrUpdate(List<OqsEngineEntity> oqsEngineEntitys) throws IOException {
        Map<String, List<BulkRequest>> tenantBulkRequests = new HashMap<>();
        handleTenantsBatchRows(oqsEngineEntitys).entrySet().stream().forEach(tenantEngineEntitys -> {
            List<BulkRequest> bulkRequests = new ArrayList<>();
            tenantEngineEntitys.getValue().entrySet().stream().forEach(oqsEngineEntities -> {
                BulkRequest bulkRequest = new BulkRequest();
                oqsEngineEntities.getValue().forEach(oqsEngineEntity -> {
                    UpdateRequest updateRequest = new UpdateRequest(oqsEngineEntities.getKey(),
                            String.valueOf(oqsEngineEntity.getId()));
                    IndexRequest indexRequest = new IndexRequest(oqsEngineEntities.getKey()).source(
                            oqsEngineEntity.getAttributes(), XContentType.JSON);
                    indexRequest.id(String.valueOf(oqsEngineEntity.getId()));
                    if (oqsEngineEntity.getRoutingId() != null) {
                        indexRequest.routing(oqsEngineEntity.getRoutingId());
                        updateRequest.routing(oqsEngineEntity.getRoutingId());
                    }
                    updateRequest.doc(indexRequest);
                    updateRequest.retryOnConflict(retryOnConflict);
                    updateRequest.docAsUpsert(true);
                    bulkRequest.add(updateRequest);
                });
                bulkRequests.add(bulkRequest);
            });
            tenantBulkRequests.put(tenantEngineEntitys.getKey(), bulkRequests);
        });

        return sumbitBulk(tenantBulkRequests);
    }

    /**
     * 提交 按租户按索引表分组后的bulk 数据
     *
     * @param tenantBulkRequests
     * @return
     **/
    private boolean sumbitBulk(Map<String, List<BulkRequest>> tenantBulkRequests){
        for (Map.Entry<String, List<BulkRequest>> bulkRequests : tenantBulkRequests.entrySet()) {
            Boolean result = null;
            try {
                result = pool.submit(() -> {
                    String profile = StringUtils.equalsIgnoreCase(bulkRequests.getKey(), CommonProperty.defaultProfile) ? null : bulkRequests.getKey();
                    return bulkRequests.getValue().stream().parallel().map(x -> {
                        //TODO
                        x.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
                        BulkResponse bulk = null;
                        try {
                            bulk = ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile)
                                    .bulk(x, RequestOptions.DEFAULT);
                        } catch (IOException e) {
                            log.error("{}", e);
                            return false;
                        }
                        if (bulk.hasFailures()) {
                            log.error("FAILURE,elasticsearch sumbitBulk method execute failed!,cause by:{}", bulk.buildFailureMessage());
                            return false;
                        }
                        return true;
                    }).filter(x -> !x).findAny().orElse(true);
                }).get();
            } catch (InterruptedException | ExecutionException e) {
                log.error("{}", e);
                return false;
            }
            if (!result) {
                return false;
            }
        }
        return true;
    }

    /**
     * 根据Canal采集的多租户多索引批量数据 按照租户与索引表进行归组分类
     *
     * @param oqsEngineEntitys
     * @return
     **/
    public Map<String, Map<String, List<OqsEngineEntity>>> handleTenantsBatchRows(List<OqsEngineEntity> oqsEngineEntitys) {
        try {
            ConcurrentHashMap<String, Map<String, List<OqsEngineEntity>>> batchRows = new ConcurrentHashMap<>();
            for (OqsEngineEntity oqsEngineEntity : oqsEngineEntitys) {
                Map<String, List<OqsEngineEntity>> orDefault = new HashMap<>();
                String profile = oqsEngineEntity.getEntityClassRef().getProfile();
                IEntityClass iEntityClass = entityClassEngine.load(String.valueOf(oqsEngineEntity.getEntityClassRef().getId()), profile).get();
                /**校验binlog记录是否在bocp配置项中，不存在则跳过此条记录**/
                if (null != null || !manageBocpMetadataService.checkPassage(profile, iEntityClass)) {
                    continue;
                }
                String prefixEntitysKey = getProfile(profile);
                Map<String, List<OqsEngineEntity>> tenantEngineEntities = batchRows.putIfAbsent(
                        prefixEntitysKey, orDefault);
                tenantEngineEntities = tenantEngineEntities == null ? orDefault : tenantEngineEntities;
                List<OqsEngineEntity> defaultTenantEntities = new ArrayList<>();
                /**判断索引是否进行切分**/
                String indexName;
                Tuple2<Boolean, String> writeSegmentIndex = getWriteSegmentIndex(profile, oqsEngineEntity.getAttributes(), iEntityClass);
                if (!writeSegmentIndex._1) {
                    continue;
                } else {
                    indexName = writeSegmentIndex._2;
                }
                List<OqsEngineEntity> oqsEngineEntities = tenantEngineEntities.putIfAbsent(
                        indexName,
                        defaultTenantEntities);
                oqsEngineEntities = oqsEngineEntities == null ? defaultTenantEntities : oqsEngineEntities;
                if (iEntityClass.relations().size() > 0) {
                    Map<String, String> relationMap = new HashMap<>();
                    relationMap.put("name", iEntityClass.code());
                    oqsEngineEntity.getAttributes().put(iEntityClass.code().concat("_join"), relationMap);
                }
                oqsEngineEntities.add(oqsEngineEntity);
                /**根据father，拆分出父记录**/
                buildFatherAttributes(oqsEngineEntity, tenantEngineEntities);
                /**根据TO_ONE冗余宽表**/
                buildWideAttributes(oqsEngineEntity, iEntityClass, tenantEngineEntities);
            }
            return batchRows;
        } catch (Throwable e) {
            log.error("FAILURE,elasticsearch handleTenantsBatchRows method failed,cause by:{}", e.getMessage());
            throw new RuntimeException(e);
        }
    }


    @NotNull
    private String getProfile(String profile) {
        String prefixEntitysKey = CommonProperty.defaultProfile;
        if (!StringUtils.isEmpty(profile)) {
            prefixEntitysKey = profile;
        }
        return prefixEntitysKey;
    }

    /**
     * 根据father 拆分出父记录
     *
     * @param oqsEngineEntity
     * @param tenantEngineEntities
     **/
    private void buildFatherAttributes(OqsEngineEntity oqsEngineEntity, Map<String, List<OqsEngineEntity>> tenantEngineEntities) {
        if (oqsEngineEntity.getFather() != 0 && !oqsEngineEntity.isDeleted()) {
            String profile = oqsEngineEntity.getEntityClassRef().getProfile();
            /**拆分出父子记录**/
            List<OqsEngineEntity> defaultFatherEntities = new ArrayList<>();
            IEntityClass fatherEntities = entityClassEngine.load(String.valueOf(oqsEngineEntity.getFather()), profile).get();
            if (fatherEntities != null || manageBocpMetadataService.checkPassage(profile, fatherEntities)) {
                Map<String, List<OqsEngineEntity>> finalTenantEngineEntities = tenantEngineEntities;
                Map<String, Object> fatherAttributes = new HashMap();
                for (IEntityField field : fatherEntities.fields()) {
                    String fieldName = field.name().replace(".", "_");
                    if (StringUtils.equalsIgnoreCase(fieldName, "id")) {
                        fatherAttributes.put(fieldName, oqsEngineEntity.getId());
                    } else {
                        fatherAttributes.put(fieldName, oqsEngineEntity.getAttributes().get(fieldName));
                    }
                }
                OqsEngineEntity fatherOqsEngineEntity = new OqsEngineEntity();
                fatherOqsEngineEntity.setId(oqsEngineEntity.getId());
                fatherOqsEngineEntity.setAttributes(fatherAttributes);
                String segmentIndex;
                Tuple2<Boolean, String> writeSegmentIndex = getWriteSegmentIndex(profile, oqsEngineEntity.getAttributes(), fatherEntities);
                if (!writeSegmentIndex._1) {
                    return;
                } else {
                    segmentIndex = writeSegmentIndex._2;
                }
                List<OqsEngineEntity> fatherOqsEngineEntities = finalTenantEngineEntities.putIfAbsent(segmentIndex, defaultFatherEntities);
                fatherOqsEngineEntities = fatherOqsEngineEntities == null ? defaultFatherEntities : fatherOqsEngineEntities;
                fatherOqsEngineEntities.add(fatherOqsEngineEntity);
            }
        }
    }

    /**
     * 构建child与parent elastic表结构
     *
     * @param tenantEngineEntities
     * @param iEntityClass
     * @param oqsEngineEntity
     **/
    private void buildWideAttributes(OqsEngineEntity oqsEngineEntity, IEntityClass iEntityClass,
                                     Map<String, List<OqsEngineEntity>> tenantEngineEntities) throws CloneNotSupportedException {
        generateRelationOqsEngine(oqsEngineEntity, iEntityClass, tenantEngineEntities);
    }

    /**
     * 根据relation关系列表，创建oqsengineEntity对象
     *
     * @param oqsEngineEntity
     * @param iEntityClass
     * @param engineEntities
     **/
    private void generateRelationOqsEngine(
            OqsEngineEntity oqsEngineEntity,
            IEntityClass iEntityClass,
            Map<String, List<OqsEngineEntity>> engineEntities
    ) throws CloneNotSupportedException {
        List<Tuple4<String, String, String, Tuple2<Long, Map<String, Object>>>> toOneRelatedList = oqsEngineEntity.getToOneRelatedList();
        String profile = oqsEngineEntity.getEntityClassRef().getProfile();
        Map<Long, Map<String, Object>> relationAttributesMap = new HashMap();
        String prefixCode = iEntityClass.code();
        Collection<IRelation> cdcRelation = iEntityClass.relations();
        toOneRelatedList.forEach(tuple4 -> relationAttributesMap.put(tuple4._4._1, tuple4._4._2));
        if (!cdcRelation.isEmpty()) {
            /**当前entity被其它对象所依赖，迭代当前对象被其它对象所依赖的relation列表，将记录分发回写至所依赖的其它index索引库中**/
            for (IRelation iRelation : cdcRelation) {
                IEntityClass relatedEntityClass = entityClassEngine.load(String.valueOf(iRelation.getEntityClassId()), profile).get();
                /**校验binlog记录是否在bocp配置项中，不存在则跳过此条记录**/

                //self relation should be TO_ONE and related entityClass should have a revert relation
                //omit TO ONE
                if (!(StringUtils.equalsIgnoreCase(iRelation.getRelationType(), "TO_ONE")) || !manageBocpMetadataService.checkPassage(profile, relatedEntityClass)) {
                    continue;
                }

                /**
                 * related relation
                 */
                Collection<IRelation> reverseRelations = relatedEntityClass.relations();

                List<IRelation> reversedRelation = reverseRelations.stream().filter(x -> x.getEntityClassId() == iEntityClass.id())
                        .collect(Collectors.toList());

                if (reversedRelation.isEmpty()) {
                    continue;
                }

                String iRelationName = iRelation.getName();
                String relationIdName = iRelationName.concat("_").concat("id");
                Object relationIdVal = oqsEngineEntity.getAttributes().get(relationIdName);
                if (relationIdVal != null && StringUtils.isNotEmpty(String.valueOf(relationIdVal))) {
                    String value = String.valueOf(relationIdVal);
                    OqsEngineEntity.Builder builder = (OqsEngineEntity.Builder) oqsEngineEntity.clone();
                    OqsEngineEntity wideOqsEngineEntity = builder.build();
                    String indexName;
                    Tuple2<Boolean, String> writeSegmentIndex = getWriteSegmentIndex(profile, relationAttributesMap.get(iRelation.getEntityClassId()),
                            relatedEntityClass);
                    if (!writeSegmentIndex._1) {
                        continue;
                    } else {
                        indexName = writeSegmentIndex._2;
                    }
                    Map<String, Object> attributes = new HashMap<>();
                    if (wideOqsEngineEntity.getAttributes().containsKey(prefixCode.concat("_").concat("join"))) {
                        wideOqsEngineEntity.getAttributes().remove(prefixCode.concat("_").concat("join"));
                    }
                    wideOqsEngineEntity.getAttributes().keySet().stream().forEach(key -> {
                        if (StringUtils.equalsIgnoreCase("id", key)) {
                            attributes.put(key, wideOqsEngineEntity.getAttributes().get(key));
                        }
                        attributes.put(prefixCode.concat(".").concat(key), wideOqsEngineEntity.getAttributes().get(key));
                    });
                    Map<String, String> relationMap = new HashMap<>();
                    relationMap.put("name", iRelationName);
                    relationMap.put("parent", value);
                    attributes.put(relatedEntityClass.code().concat("_join"), relationMap);
                    wideOqsEngineEntity.setAttributes(attributes);
                    wideOqsEngineEntity.setRoutingId(value);
                    commondBuildWideRow(wideOqsEngineEntity, engineEntities, indexName);
                }
            }
        }
    }

    /**
     * 判断索引是否进行切分
     *
     * @param profile
     * @param iEntityClass
     * @param attribute
     **/
    private Tuple2<Boolean, String> getWriteSegmentIndex(String profile, Map<String, Object> attribute,
                                                         IEntityClass iEntityClass) {
        try {
            String indexName = manageBocpMetadataService.getWriteSegmentIndex(profile, iEntityClass, attribute);
            return new Tuple2<>(true, indexName);
        } catch (Exception e) {
            log.warn(e.getMessage());
        }
        return new Tuple2<>(false, null);
    }

    /**
     * 根据parent id条件删除宽表里面冗余的明细信息
     *
     * @param profile
     * @param relationClass
     * @param iEntityClass
     * @param oqsEngineEntity
     **/
    private void parentIdQueryDelete(OqsEngineEntity oqsEngineEntity, IEntityClass relationClass, IEntityClass iEntityClass, String profile) {
        try {
            String indexName = DynamicConfigUtils.insulateTenant(dynamicConfig, profile, iEntityClass.code(),
                    iEntityClass.ref().getAppCode());
            ParentIdQueryBuilder parentIdQuery = JoinQueryBuilders.parentId(relationClass.code(), String.valueOf(oqsEngineEntity.getId()));
            /**  通过DeleteByQueryRequest来构建删除请求，setQuery来装载条件，indices来指定索引**/
            DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest();
            deleteByQueryRequest.setRouting(String.valueOf(oqsEngineEntity.getId()));
            deleteByQueryRequest.setQuery(parentIdQuery);
            deleteByQueryRequest.indices(indexName);
            deleteByQueryRequest.setConflicts("proceed");
            RestHighLevelClient executor = ((ElasticsearchTransportExecutor) elasticsearchTransportExecutor).executor(profile);
            BulkByScrollResponse deleteResponse = null;
            deleteResponse = executor.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
            if (deleteResponse.getDeleted() >= 1) {
                log.info("delete parent-child 冗余明细明细文档成立,删除文档条数: " + deleteResponse.getDeleted() + ",indexName：" + indexName);
            }
        } catch (Throwable e) {
            log.error("FAILURE,elasticsearch parentIdQueryDelete method failed,cause by:{}", e.getMessage());
            throw new RuntimeException(e);
        }
    }

    /**
     * child 记录冗余到对应的relation parent表中
     *
     * @param oqsEngineEntity
     * @param indexName
     * @param tenantEngineEntities
     **/
    private void commondBuildWideRow(OqsEngineEntity oqsEngineEntity,
                                     Map<String, List<OqsEngineEntity>> tenantEngineEntities,
                                     String indexName) {
        List<OqsEngineEntity> defaultChildEntities = new ArrayList<>();
        List<OqsEngineEntity> oqsEngineEntities = tenantEngineEntities.putIfAbsent(
                indexName, defaultChildEntities);
        oqsEngineEntities =
                oqsEngineEntities == null ? defaultChildEntities : oqsEngineEntities;
        oqsEngineEntities.add(oqsEngineEntity);
    }

    /***
     * 批量删除
     * @param oqsEngineEntitys
     * @return
     * @throws IOException
     */
    private Boolean deleteBatch(List<OqsEngineEntity> oqsEngineEntitys) throws IOException {
        Map<String, List<BulkRequest>> tenantBulkRequests = new HashMap<>();
        handleTenantsBatchRows(oqsEngineEntitys).entrySet().stream().forEach(tenantEngineEntitys -> {
            List<BulkRequest> bulkRequests = new ArrayList<>();
            tenantEngineEntitys.getValue().entrySet().stream().forEach(oqsEngineEntities -> {
                BulkRequest bulkRequest = new BulkRequest();
                oqsEngineEntities.getValue().forEach(oqsEngineEntity -> {
                    DeleteRequest deleteRequest = new DeleteRequest(oqsEngineEntities.getKey()).id(
                            String.valueOf(oqsEngineEntity.getId()));
                    if (oqsEngineEntity.getRoutingId() != null) {
                        deleteRequest.routing(oqsEngineEntity.getRoutingId());
                    }
                    bulkRequest.add(deleteRequest);
                });
                bulkRequests.add(bulkRequest);
            });
            tenantBulkRequests.put(tenantEngineEntitys.getKey(), bulkRequests);
        });
        return sumbitBulk(tenantBulkRequests);
    }
}
