package com.xforceplus.ultraman.adapter.elasticsearch.service.utils;

import com.xforceplus.ultraman.adapter.elasticsearch.service.constant.FieldMappingType;
import com.xforceplus.ultraman.adapter.elasticsearch.service.entity.FieldMapping;
import com.xforceplus.ultraman.metadata.entity.FieldConfig;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.calcite.sql.validate.SqlValidatorUtil;
import org.apache.commons.lang.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @program: ultraman-oqsengine-plus
 * @ClassName ElasticSearchMappingBuildUtils
 * @description:
 * @author: WanYi
 * @create: 2023-05-12 17:37
 * @Version 1.0
 **/
public class ElasticSearchMappingBuildUtils {

  /**
   * @param allFields elasticsearch宽表字段集合类型
   * @return 返回每个elasticsearch 字段的映射的类型
   */
  public static List<FieldMapping> getFieldInfos(Collection<IEntityField> allFields) {
    List<String> uniquify = SqlValidatorUtil.uniquify(
        new AbstractList<String>() {
          @Override
          public String get(int index) {
            return new ArrayList<>(allFields).get(index).name().trim();
          }

          @Override
          public int size() {
            return new ArrayList<>(allFields).size();
          }
        }, true
    );

    AtomicInteger index = new AtomicInteger(0);
    return allFields.stream().map(fieldInfo -> {
      String fieldName = uniquify.get(index.get());
      FieldMapping build;
      if (fieldInfo.type().getJavaType().getSuperclass() == Number.class || fieldInfo.type().getJavaType() == Boolean.class) {
        build = FieldMapping.builder().field(fieldName).type(getType(null, fieldInfo.type()))
            .build();
      } else {
        build = FieldMapping.builder().field(fieldName).type(getType(fieldInfo.config().getFuzzyType(), fieldInfo.type()))
            .analyzer(FieldMappingType.SETANALYZER_IK_SMART.getType()).build();
      }
      build.setOriginField(fieldInfo);
      index.getAndIncrement();
      return build;
    }).collect(Collectors.toList());
  }

  /**
   * * @string text ,keyword * @Numeric long, integer, short, byte, double, float, half_float, scaled_float * @date date(分 datetime, timestamp 两种情况处理)
   * * @Object object
   *
   * @param fuzzyType 字段是否可以迷糊匹配
   * @param fieldType 字段类型
   * @return 返回elasticsearch mapping字段映射类型
   */
  private static String getType(FieldConfig.FuzzyType fuzzyType, FieldType fieldType) {
    switch (fieldType) {
      case ENUM:
      case STRING:
        return getStringType(fuzzyType, fieldType);
      case BOOLEAN:
        return FieldMappingType.BOOLEAN.getType();
      case LONG:
      case DATETIME:
        return FieldMappingType.LONG.getType();
      case DECIMAL:
        return FieldMappingType.DOUBLE.getType();
      case STRINGS:
      default:
        return FieldMappingType.KEYWORD.getType();
    }
  }


  /***
   * 根据迷糊匹配,判断字段是否需要分词 Textl默认分词，Keyword默认不分词
   * @param fuzzyType  模糊匹配类型
   * @param fieldType  字段类型
   * @return 返回elasticsearch 字段映射类型
   */
  private static String getStringType(FieldConfig.FuzzyType fuzzyType, FieldType fieldType) {
    if (!Objects.isNull(fuzzyType)) {
      switch (fuzzyType) {
        case NOT:
        case UNKNOWN:
        case WILDCARD:
          if (StringUtils.equalsIgnoreCase(fieldType.getType(), FieldType.STRING.getType())) {
            return FieldMappingType.KEYWORD.getType();
          }
        case SEGMENTATION:
          if (StringUtils.equalsIgnoreCase(fieldType.getType(), FieldType.STRING.getType())) {
            return FieldMappingType.TEXT.getType();
          }
        default:
          return FieldMappingType.KEYWORD.getType();
      }
    }
    return FieldMappingType.KEYWORD.getType();
  }


}
