package com.xforceplus.ultraman.extensions.auth.plus.util;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import io.vavr.Tuple;
import io.vavr.Tuple3;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.Constructor;
import org.yaml.snakeyaml.nodes.Node;
import org.yaml.snakeyaml.nodes.SequenceNode;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AuthTemplateProcessor {

    private final static TypeReference<List<Map<String, String>>> LIST_MAP_TYPE = new TypeReference<List<Map<String, String>>>() {
    };

    private final static TypeReference<Map<String, Object>> MAP_TYPE = new TypeReference<Map<String, Object>>() {
    };

    private final static TypeReference<Map<String, String>> MAP_STR_TYPE = new TypeReference<Map<String, String>>() {
    };

    private final String KEY_KEY = "variableKey";

    private final String VAL_KEY = "variableValue";

    private final static String DEFAULT_PATH = "auth-tpl.yaml";

    private Yaml yaml;

    private LoadingCache<Tuple3<String, String, String>, Map<String, Object>> cachedConfig;

    private ObjectMapper mapper;

//    public static void main(String[] args) throws JsonProcessingException {
//        AuthTemplateProcessor processor = new AuthTemplateProcessor();
//        String templateCode = "usercenter"; // Example: template code
//        String envCode = "fat"; // Example: environment code
//        ObjectMapper mapper = new ObjectMapper();
//        String result = processor.renderTemplate("auth-tpl.yaml", templateCode, envCode, mapper);
//        System.out.println(result);
//    }

    public AuthTemplateProcessor(ObjectMapper mapper) {
        this.mapper = mapper;
        yaml = new Yaml(new ListConstructor<>(Template.class));
        cachedConfig = Caffeine.newBuilder()
                .build(env -> {
                    List<Template> templates = load(env._1);
                    YamlConfig yamlConfig = new YamlConfig();
                    yamlConfig.setTemplates(templates);
                    return renderTemplate(yamlConfig, env._2, env._3);
                });
    }

    public List<Template> load(String yamlFilePath) {
        InputStream inputStream = this.getClass()
                .getClassLoader()
                .getResourceAsStream(yamlFilePath);
        List<Template> templates = yaml.load(inputStream);
        return templates;
    }

    public Map<String, Object> get(String path, String envCode, String templateCode) {
        return cachedConfig.get(Tuple.of(path, envCode, templateCode));
    }

    public Map<String, Object> get(String envCode, String templateCode) {
        return cachedConfig.get(Tuple.of(DEFAULT_PATH, envCode, templateCode));
    }

    public Map<String, Object> renderTemplate(YamlConfig config, String envCode, String templateCode) throws JsonProcessingException {
        // Find the template by code
        for (Template template : config.getTemplates()) {
            if (template.getAuthTemplateCode().equals(templateCode)) {
                // Find the environment variables by envCode
                for (Env env : template.getEnvs()) {
                    if (env.getEnvCode().equals(envCode)) {
                        // Parse envVariable as JSON
                        Map<String, String> envVariables = parseEnvVariables(env.getEnvVariable(), mapper);
                        // Render the authTemplateContent with the environment variables
                        return renderAuthTemplate(template.getAuthTemplateContent(), envVariables);
                    }
                }
            }
        }
        return null;
    }

    private Map<String, String> parseEnvVariables(String envVariableString, ObjectMapper objectMapper) throws JsonProcessingException {
        Map<String, String> variablesMap = new HashMap<>();
        envVariableString = envVariableString.replaceAll("'", "\"");
        List<Map<String, String>> maps = objectMapper.readValue(envVariableString, LIST_MAP_TYPE);
        for (Map<String, String> map : maps) {
            String key = map.get(KEY_KEY);
            if (!StringUtils.isEmpty(key)) {
                variablesMap.put(key, map.get(VAL_KEY));
            }
        }
        return variablesMap;
    }

    private Map<String, Object> renderAuthTemplate(String authTemplateContent, Map<String, String> envVariables) {
        // Replace each variable in the template content with the corresponding value
        for (Map.Entry<String, String> entry : envVariables.entrySet()) {
            String keyPlaceholder = entry.getKey(); // Use the original key from the map
            authTemplateContent = authTemplateContent.replace(keyPlaceholder, entry.getValue());
        }

        authTemplateContent = authTemplateContent.replaceAll("'", "\"");

        try {
            return mapper.readValue(authTemplateContent, MAP_TYPE);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    // Java class for the entire YAML configuration
    public static class YamlConfig {
        private List<Template> templates;

        public List<Template> getTemplates() {
            return templates;
        }

        public void setTemplates(List<Template> templates) {
            this.templates = templates;
        }
    }

    public class ListConstructor<T> extends Constructor {
        private final Class<T> clazz;

        public ListConstructor(final Class<T> clazz) {
            this.clazz = clazz;
        }

        @Override
        protected Object constructObject(final Node node) {
            if (node instanceof SequenceNode && isRootNode(node)) {
                ((SequenceNode) node).setListType(clazz);
            }
            return super.constructObject(node);
        }

        private boolean isRootNode(final Node node) {
            return node.getStartMark().getIndex() == 0;
        }
    }

    @Data
    public static class TemplateVo {
        private String authTemplateCode;
        private int authTemplateType;
        private String authTemplateContent;
    }
    
    
    @Data
    // Java class representing the template
    public static class Template {
     
        private TemplateVo template;
        
        private List<Env> envs;

        public String getAuthTemplateCode() {
            return template.authTemplateCode;
        }

        public String getAuthTemplateContent() {
            return template.authTemplateContent;
        }
        

        public List<Env> getEnvs() {
            return envs;
        }

        public void setEnvs(List<Env> envs) {
            this.envs = envs;
        }
    }

    // Java class representing an environment within a template
    public static class Env {
        private String envCode;
        private String envVariable; // String containing key-value pairs

        public String getEnvCode() {
            return envCode;
        }

        public void setEnvCode(String envCode) {
            this.envCode = envCode;
        }

        public String getEnvVariable() {
            return envVariable;
        }

        public void setEnvVariable(String envVariable) {
            this.envVariable = envVariable;
        }
    }
}
