package com.xforceplus.ultraman.extensions.auth.plus.util;

import com.xforceplus.tech.common.utils.JsonHelper;
import com.xforceplus.ultraman.extensions.auth.plus.model.RestOrgData;
import com.xforceplus.ultraman.extensions.auth.plus.model.UserCenterList;
import com.xforceplus.ultraman.extensions.auth.plus.model.UserCenterResponse;
import com.xforceplus.ultraman.extensions.auth.plus.util.usercenter.TokenTask;
import com.xforceplus.ultraman.sdk.infra.query.LazyFetchIterator;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StopWatch;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.RestTemplate;

import java.lang.reflect.Type;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.sdk.infra.utils.IteratorUtils.toStream;

public class RestTemplateHelper {

    private final RestTemplate restTemplate;

    private final Logger log = LoggerFactory.getLogger(RestTemplateHelper.class);

    private final AuthTemplateProcessor templateProcessor;

    private TokenTask tokenTask;

    private final static String USER_CENTER_TPL_CODE = "usercenter";

    public RestTemplateHelper(RestTemplate restTemplate, AuthTemplateProcessor templateProcessor) {
        this.restTemplate = restTemplate;
        this.templateProcessor = templateProcessor;
        this.tokenTask = new TokenTask(restTemplate);
    }

    public <T> UserCenterResponse<UserCenterList<T>> iterate(String env, String subUrl, int step, Class<T> clazz, Object... params) {
        List<UserCenterResponse<UserCenterList<T>>> responseHolder = new ArrayList<>();

        Type genericType = TypeUtil.getGenericType(UserCenterResponse.class, TypeUtil.getGenericType(UserCenterList.class, clazz));
        ParameterizedTypeReference<UserCenterResponse<UserCenterList<T>>> parameterizedTypeReference = ParameterizedTypeReference.forType(genericType);
        Iterator<T> iterator = new LazyFetchIterator<>(Tuple.of(0, step), input -> {
            log.debug("Current input {}", input);
            String pageRowStr = String.format("&page=%s&row=%s", input._1, input._2);
            UserCenterResponse<UserCenterList<T>> fromPaas = this.getFromPaas(env, subUrl + pageRowStr
                    , parameterizedTypeReference,
                    params);
            if (fromPaas.getCode().equals(UserCenterResponse.SUCCESS_CODE)) {
                List<T> content = fromPaas.getResult().getContent();
                fromPaas.getResult().setContent(Collections.emptyList());
                responseHolder.add(fromPaas);
                return content;
            } else {
                //how to handle when query is error?
                responseHolder.add(fromPaas);
                return Collections.emptyList();
            }
        }, (input, last) -> {
            return Tuple.of(input._1 + 1, step);
        }, request -> true, response -> response.isEmpty() || response.size() < step, Function.identity());

        List<T> retList = toStream(iterator).collect(Collectors.toList());

        UserCenterResponse<UserCenterList<T>> container = mergeResponse(responseHolder);
        
        if(UserCenterResponse.SUCCESS_CODE.equals(container.getCode())) {
            container.getResult().setContent(retList);
        }
        return container;
    }

    private <T> UserCenterResponse<UserCenterList<T>> mergeResponse(List<UserCenterResponse<UserCenterList<T>>> responseHolder) {
        UserCenterResponse<UserCenterList<T>> container = new UserCenterResponse<>();
        container.setResult(new UserCenterList<>());
        Optional<UserCenterResponse<UserCenterList<T>>> first = responseHolder.stream().filter(x -> !x.getCode().equals(UserCenterResponse.SUCCESS_CODE)).findFirst();
        UserCenterResponse<UserCenterList<T>> sampleResponse =  first.orElseGet(() -> responseHolder.get(0));;
        container.setCode(sampleResponse.getCode());
        container.setMessage(sampleResponse.getMessage());
        return container;
    }


    /**
     * fun <T> getListWithExceptionDeal(
     *         methodName: String,
     *         pageNo: Int?,
     *         pageSize: Int?,
     *         applyFunc: (String) -> UserCenterResponse<UserCenterList<T>>
     *     ): UserCenterResponse<UserCenterList<T>> {
     *         return try {
     *             var page = if (pageNo == null || pageNo < 1) 0 else pageNo - 1
     *             val row = if (pageSize == null || pageSize > 1000 || pageSize < 0) 1000 else pageSize
     *             val replyList = mutableListOf<T>()
     *             var total = 0
     *             var code: String
     *             var message: String?
     *             var reply: UserCenterList<T>?
     *             do {
     *                 page += 1
     *                 val result = applyFunc("&page=$page&row=$row")
     *                 code = result.code
     *                 message = result.message
     *                 reply = result.result
     *                 reply?.let {
     *                     total = it.totalElements
     *                     it.content.takeIf { r -> !r.isNullOrEmpty() }?.toCollection(replyList)
     *                 } ?: break
     *             } while (replyList.size < total)
     *             UserCenterResponse(code, message, UserCenterList(0, total, total, replyList))
     *         } catch (e: HttpStatusCodeException) {
     *             log.error("$methodName HttpClientErrorException ${e.responseBodyAsString} $e")
     *             JsonUtils.fromJson(e.responseBodyAsString, object : TypeReference<UserCenterResponse<UserCenterList<T>>>(){})
     *         } catch (e: Exception) {
     *             log.error("$methodName exception ${e.message} $e")
     *             UserCenterResponse("0", e.message, null)
     *         }
     *     }
     * @param env
     * @param subUrl
     * @param responseClass
     * @param params
     * @return
     * @param <T>
     */



    //@Retryable(value = HttpStatusCodeException.class, maxAttempts = 3, backoff = @Backoff(delay = 500))
    public <T> T getFromPaas(String env, String subUrl, ParameterizedTypeReference<T> responseClass, Object... params) {
        Map<String, Object> variables = templateProcessor.get(env, USER_CENTER_TPL_CODE);
        if (variables != null) {
            HttpHeaders headers = getUserCenterHeader(env);
            String apiHost = variables.get("apiHost").toString();
            String url = apiHost + subUrl;
            StopWatch stopwatch = new StopWatch();
            stopwatch.start(url);
            ResponseEntity<T> exchange = restTemplate.exchange(url, HttpMethod.GET, new HttpEntity<>(headers), responseClass, params);
            stopwatch.stop();
            return exchange.getBody();
        } else {
            //TODO
            throw new RuntimeException("Configuration Error");
        }
    }

    //    @Retryable(value = HttpStatusCodeException.class, maxAttempts = 3, backoff = @Backoff(delay = 500))
    public <T> UserCenterResponse<T> getFromPaasForUserCenter(String env, String subUrl
            , ParameterizedTypeReference<UserCenterResponse<T>> responseClass, Object... params) {
        HttpHeaders headers = getUserCenterHeader(env);
        Map<String, Object> variables = templateProcessor.get(env, USER_CENTER_TPL_CODE);
        if (variables != null) {
            String apiHost = variables.get("apiHost").toString();
            String url = apiHost + subUrl;
            StopWatch stopwatch = new StopWatch();
            log.info("getFromPaas, url: {}, params: {}", url, JsonHelper.toJsonStr(params));
            stopwatch.start(url);
            ResponseEntity<UserCenterResponse<T>> exchange;
            try {
                exchange = restTemplate.exchange(url, HttpMethod.GET, new HttpEntity<>(headers), responseClass, params);
            } catch (HttpStatusCodeException e) {
                log.info("getFromPaas, error: {}, {}", e.getMessage(), e);
                if (e.getStatusCode().value() == 400) {
                    return JsonHelper.fromJsonStr(e.getResponseBodyAsString(), UserCenterResponse.class).orElse(null);
                } else if (e.getStatusCode().value() == 404) {
                    return new UserCenterResponse<>("0", e.getMessage(), null);
                } else {
                    throw e;
                }
            } finally {
                stopwatch.stop();
                log.info(stopwatch.prettyPrint());
            }
            log.info("getFromPaas, response: {}", exchange);
            return exchange.getBody();
        } else {
            //TODO
            throw new RuntimeException("Configuration Error");
        }
    }

    public <R, T> T postFromPaas(String env, String subUrl, R request, ParameterizedTypeReference<T> responseClass, Object... params) {
        HttpHeaders headers = getUserCenterHeader(env);
        Map<String, Object> variables = templateProcessor.get(env, USER_CENTER_TPL_CODE);
        if (variables != null) {
            String apiHost = variables.get("apiHost").toString();
            String url = apiHost + subUrl;
            log.info("postFromPaas, url: {}, request: {}", url, JsonHelper.toJsonStr(request));
            HttpEntity<R> httpEntity = new HttpEntity<>(request, headers);
            ResponseEntity<T> exchange = restTemplate.exchange(url, HttpMethod.POST, httpEntity, responseClass, params);
            log.info("postFromPaas, response: {}", JsonHelper.toJsonStr(exchange));
            return exchange.getBody();
        } else {
            //TODO
            throw new RuntimeException("Configuration Error");
        }
    }

    //    @Retryable(value = HttpStatusCodeException.class, maxAttempts = 3, backoff = @Backoff(delay = 500))
    public <R, T> T post(String env, String subUrl, R request, Class<T> responseClass, Object... params) {
        HttpHeaders headers = getUserCenterHeader(env);
        Map<String, Object> variables = templateProcessor.get(env, USER_CENTER_TPL_CODE);
        if (variables != null) {
            HttpHeaders httpHeaders = new HttpHeaders();
            httpHeaders.add("Content-Type", "application/json");
            String apiHost = variables.get("apiHost").toString();
            String url = apiHost + subUrl;
            log.info("post, url: {}, request: {}", url, JsonHelper.toJsonStr(request));
            ResponseEntity<T> exchange = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(request, httpHeaders), responseClass, params);
            log.info("post, response: {}", JsonHelper.toJsonStr(exchange));
            return exchange.getBody();
        } else {
            //TODO
            throw new RuntimeException("Configuration Error");
        }
    }

    private HttpHeaders getUserCenterHeader(String env) {
        Map<String, Object> variables = templateProcessor.get(env, USER_CENTER_TPL_CODE);
        if (variables != null) {
            String apiHost = variables.get("apiHost").toString();
            String clientId = variables.get("client").toString();
            String secret = variables.get("secret").toString();
            String token = tokenTask.getClientToken(apiHost, clientId, secret);
            HttpHeaders requestHeaders = new HttpHeaders();
            requestHeaders.add("Content-Type", "application/json");
            requestHeaders.add("accept", "application/json");
            requestHeaders.add("x-app-token", token);
            return requestHeaders;
        } else {
            //TODO
            throw new RuntimeException("Configuration Error");
        }
    }

    public <T> UserCenterResponse<T> getWithExceptionDeal(String methodName, ExceptionSupplier<UserCenterResponse<T>> applyFunc) {
        try {
            return applyFunc.get();
        } catch (HttpStatusCodeException e) {
            log.error("{} HttpClientErrorException {} {}", methodName, e.getResponseBodyAsString(), e);
            return JsonHelper.fromJsonStr(e.getResponseBodyAsString(), UserCenterResponse.class)
                    .orElseThrow(() -> new RuntimeException("Response parse Error from UserCenter"));
        } catch (Exception e) {
            log.error("{} exception {} {}", methodName, e.getMessage(), e);
            return new UserCenterResponse<>("0", e.getMessage(), null);
        }
    }

//    @Retryable(value = HttpStatusCodeException.class, maxAttempts = 3, backoff = @Backoff(delay = 500))
//    public <R, T> T postFromCoordination(String subUrl, R request, ParameterizedTypeReference<T> responseClass, Object... params) {
//        HttpHeaders httpHeaders = new HttpHeaders();
//        httpHeaders.add("Content-Type", "application/json");
//        String url = coordinationProperties.getBaseUrl() + subUrl;
//        log.info("postFromCoordination, url: {}, request: {}", url, JsonUtils.toJson(request));
//        ResponseEntity<T> exchange = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(request, httpHeaders), responseClass, params);
//        log.info("postFromCoordination, response: {}", JsonUtils.toJson(exchange));
//        return exchange.getBody();
//    }

//    public <T> CoordinationResponse<T> getWithExceptionDealForCoordination(String methodName, ExceptionSupplier<CoordinationResponse<T>> applyFunc) {
//        try {
//            return applyFunc.get();
//        } catch (HttpStatusCodeException e) {
//            log.error("{} HttpClientErrorException {} {}", methodName, e.getResponseBodyAsString(), e);
//            return JsonUtils.fromJson(e.getResponseBodyAsString(), new TypeReference<CoordinationResponse<T>>() {
//            });
//        } catch (Exception e) {
//            log.error("{} exception {} {}", methodName, e.getMessage(), e);
//            return new CoordinationResponse<>("0", e.getMessage(), null);
//        }
//    }

//    @Retryable(value = HttpStatusCodeException.class, maxAttempts = 3, backoff = @Backoff(delay = 500))
//    public <T> T getFromOutside(String subUrl, ParameterizedTypeReference<T> responseClass, Object... params) {
//        HttpHeaders httpHeaders = new HttpHeaders();
//        httpHeaders.add("Content-Type", "application/json");
//        String url = outsideProperties.getBaseUrl() + subUrl;
//        log.info("getFromOutside, url: {}, params: {}", url, JsonUtils.toJson(params));
//        ResponseEntity<T> exchange = restTemplate.exchange(url, HttpMethod.GET, new HttpEntity<>(httpHeaders), responseClass, params);
//        log.info("getFromOutside, response: {}", JsonUtils.toJson(exchange));
//        return exchange.getBody();
//    }

//    public <T> OutsideResponse<T> getWithExceptionDealForOutside(String methodName, ExceptionSupplier<OutsideResponse<T>> applyFunc) {
//        try {
//            return applyFunc.get();
//        } catch (HttpStatusCodeException e) {
//            log.error("{} HttpClientErrorException {} {}", methodName, e.getResponseBodyAsString(), e);
//            return JsonUtils.fromJson(e.getResponseBodyAsString(), new TypeReference<OutsideResponse<T>>() {
//            });
//        } catch (Exception e) {
//            log.error("{} exception {} {}", methodName, e.getMessage(), e);
//            return new OutsideResponse<>("0", e.getMessage(), null);
//        }
//    }

    @FunctionalInterface
    public interface ExceptionSupplier<T> {
        T get() throws Exception;
    }
}

