package com.xforceplus.ultraman.extensions.auth.usercenter;

import com.xforceplus.tenant.data.auth.dto.Category;
import com.xforceplus.tenant.data.auth.dto.RuleDTO;
import com.xforceplus.tenant.data.auth.dto.Status;
import com.xforceplus.tenant.data.auth.exception.ClientRuleException;
import com.xforceplus.tenant.data.auth.store.ClientDataRuleProvider;
import com.xforceplus.ultraman.extensions.auth.utils.ExpFactoryEx;
import com.xforceplus.ultraman.sdk.core.auth.AuthBuilder;
import com.xforceplus.ultraman.sdk.core.exception.PermissionFatalException;
import com.xforceplus.ultraman.sdk.core.pipeline.OperationType;
import com.xforceplus.ultraman.sdk.core.pipeline.TransformerPipeline;
import com.xforceplus.ultraman.sdk.core.rel.legacy.*;
import com.xforceplus.ultraman.sdk.core.rel.tree.BinaryTreeNode;
import com.xforceplus.ultraman.sdk.core.rel.tree.builder.TreeBuilder;
import com.xforceplus.ultraman.sdk.core.rel.tree.dsl.ConditionNode;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import java.io.IOException;
import java.time.Duration;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.ultraman.sdk.core.rel.legacy.ExpOperator.AND;

@Slf4j
public class TenantLegacyAuth implements AuthBuilder {

    @Value("${xplat.oqsengine.sdk.tenant.failOnEx:false}")
    private boolean failOnEx = false;

    @Value("${xplat.oqsengine.sdk.tenant.failOnIOEx:false}")
    private boolean failOnIOEx = false;

    @Value("${xplat.oqsengine.sdk.tenant.retry:10}")
    private int maxAttempts = 10;

    private RetryConfig permissionRetryConfig;

    @Autowired
    private TreeBuilder treeBuilder;

    @Autowired
    private TransformerPipeline transformerPipeline;

    @Autowired(required = false)
    private ClientDataRuleProvider clientDataRuleProvider;

    public TenantLegacyAuth() {
        this.permissionRetryConfig = RetryConfig.<List<RuleDTO>>custom()
                .maxAttempts(maxAttempts)
                .waitDuration(Duration.ofMillis(500))
                .retryOnException(x -> x != null && x.getCause() instanceof IOException)
                .build();
    }

    @Override
    public ExpRel getPermissionTreeCondition(Map<Long, Set<String>> involvedIdsMapping
            , ExpContext expContext, String profile, OperationType type) {
        try {
            List<RuleDTO> rules = Collections.emptyList();
            try {
                rules = Optional.ofNullable(clientDataRuleProvider.currentUserDataRules()).orElseGet(Collections::emptyList);
            } catch (Throwable throwable) {
                if (failOnEx) {
                    log.warn("fast fail on permission error");
                    throw new PermissionFatalException();
                } else {
                    if (throwable instanceof ClientRuleException) {
                        if (throwable.getCause() != null) {
                            if (throwable.getCause() instanceof IOException) {
                                if (failOnIOEx) {
                                    log.warn("fast fail on permission IO error");
                                    throw new PermissionFatalException();
                                } else {
                                    log.error("permission process error got IO exception try to retry");
                                    Retry permission = Retry.of("permission", permissionRetryConfig);
                                    rules = permission.executeSupplier(() -> clientDataRuleProvider.currentUserDataRules());
                                }
                            } else {
                                //fast fail
                                throw throwable;
                            }
                        } else {
                            //fast fail
                            throw throwable;
                        }
                    } else {
                        //fast fail
                        throw throwable;
                    }
                }
            }

            //filter rules with current rel related
            Set<Tuple2<String, RuleDTO>> filteredRules = rules.stream().flatMap(
                    x -> {
                        if (x.getStatus() == Status.VALID
                                && involvedIdsMapping.containsKey(x.getMetaDataId())
                                && x.getCategory() == Category.SQL) {

                            Set<String> related = Optional.ofNullable(involvedIdsMapping.get(x.getMetaDataId()))
                                    .orElseGet(Collections::emptySet);
                            
                            if(related.isEmpty()) {
                                return Stream.of(Tuple.of("", x));
                            } else {
                                return related.stream().map(r -> {
                                    return Tuple.of(r, x);
                                });
                            }
                        } else {
                            return null;
                        }
                    }
            ).filter(Objects::nonNull).collect(Collectors.toSet());

            List<ExpNode> conditions = filteredRules.stream().map(x -> {
                BinaryTreeNode<ConditionNode> nodeTree = treeBuilder.build(x._2.getRuleConditions(), x._1());
                ExpCondition condition = ExpFactoryEx.createFrom(nodeTree);
                return condition;
            }).collect(Collectors.toList());

            ExpCondition condition = ExpCondition.call(AND, conditions);
            ExpQuery expQuery = new ExpQuery().filters(condition);
            /**
             * will skip calcite and range restrict
             */
            ExpRel permissionExpRel = transformerPipeline.querySideHandleValue(expQuery
                    , expContext, Collections.singleton("calcite"), Collections.singleton("range"));
//
            return permissionExpRel;
        } catch (Exception ex) {
            log.error("permission process error {}", ex);
            if (ex instanceof PermissionFatalException) {
                throw ex;
            }
        }

        return null;
    }
}
