package com.xforceplus.ultraman.extensions.auth.plus.impl;

import akka.stream.ActorMaterializer;
import akka.stream.OverflowStrategy;
import akka.stream.javadsl.Sink;
import akka.stream.javadsl.Source;
import akka.stream.javadsl.SourceQueueWithComplete;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.xforceplus.tech.common.utils.JsonHelper;
import com.xforceplus.ultraman.extensions.auth.plus.UserCenterCache;
import com.xforceplus.ultraman.extensions.auth.plus.UserCenterCacheKey;
import com.xforceplus.ultraman.extensions.auth.plus.WriteTask;
import com.xforceplus.ultraman.sdk.core.config.ExecutionConfig;
import com.xforceplus.ultraman.sdk.infra.utils.ThreadFactoryHelper;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;


@Slf4j
public class DBUserCenterCacheImpl<T> implements UserCenterCache<T> {

    private final DBCacheServiceImpl dbCacheService;

    private LoadingCache<Tuple2<Long, String>, Object> loadingCache;

    private final Class<T> clazz;

    private final boolean isList;

    private final UserCenterCacheKey userCenterCacheKey;

    private SourceQueueWithComplete<WriteTask> asyncTaskEngine;

    private ExecutorService executorService;
    
    private int timeout;

    private ActorMaterializer mat;

    public DBUserCenterCacheImpl(ActorMaterializer mat, UserCenterCacheKey userCenterCacheKey, DBCacheServiceImpl dbCacheService, Class<T> clazz, boolean isList, int timeout) {
        this.dbCacheService = dbCacheService;
        this.clazz = clazz;
        this.isList = isList;
        this.userCenterCacheKey = userCenterCacheKey;
        this.executorService = ThreadFactoryHelper.buildThreadPool(10, 10, "db-sync", false);
        this.mat = mat;
        this.loadingCache = Caffeine.newBuilder()
                .maximumSize(50000)
                .refreshAfterWrite(Duration.ofSeconds(timeout))
                .build(key -> {
                    log.info("Loading with {}", key);
                    if (isList) {
                        return dbCacheService.readListFromDb(userCenterCacheKey.getValue(), key._1, key._2, clazz);
                    } else {
                        return dbCacheService.readFromDb(userCenterCacheKey.getValue(), key._1, key._2, clazz);
                    }
                });

        asyncTaskEngine = Source.<WriteTask>queue(50000, OverflowStrategy.backpressure())
                .groupedWithin(1000, Duration.ofSeconds(10))
                .mapAsync(10, list -> {
                    return CompletableFuture.supplyAsync(() -> {
                        dbCacheService.writeToDbBatch(list);
                        return 1;
                    }, executorService);
                }).to(Sink.ignore()).run(this.mat);
    }

    @Override
    public T getGroupByTenantId(Long tenantId, String currentKey) {
        if (!isList) {
            return (T) loadingCache.get(Tuple.of(tenantId, currentKey));
        }
        
        throw new RuntimeException("Cannot get Singleton from List");
    }
    
    @Override
    public List<T> getGroupByTenantIdInList(Long tenantId, String currentKey) {
        if (isList) {
            return (List<T>) loadingCache.get(Tuple.of(tenantId, currentKey));
        }

        throw new RuntimeException("Cannot get List from singleton");
    }

    @Override
    public void trySetGroupByTenantId(Long tenantId, String currentKey, T payload) {
        trySetGroupByTenantId(tenantId, currentKey, payload, false);
    }

    @Override
    public void trySetGroupByTenantId(Long tenantId, String currentKey, T payload, boolean writeThrough) {
        loadingCache.put(Tuple.of(tenantId, currentKey), payload);
        if (writeThrough) {
            asyncTaskEngine.offer(new WriteTask(userCenterCacheKey.getValue(), currentKey, tenantId, JsonHelper.toJsonStr(payload)));
        }
    }

    @Override
    public void clearKeyGroupByTenantId(Long tenantId, String currentKey) {
        loadingCache.invalidate(Tuple.of(tenantId, currentKey));
        dbCacheService.deleteCache(userCenterCacheKey.getValue(), tenantId, currentKey);
    }

    @Override
    public void clearByTenantId(Long tenantId) {
        List<Tuple2<Long, String>> keySet = new ArrayList<>();
        loadingCache.asMap().forEach((k,v) -> {
            if(k._1().equals(tenantId)) {
                keySet.add(k);
            }
        });
        
        loadingCache.invalidateAll(keySet);
        dbCacheService.deleteCacheByTenantId(userCenterCacheKey.getValue(), tenantId);
    }
}
