package com.xforceplus.ultraman.extensions.auth.plus.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xforceplus.ultraman.extensions.auth.plus.UserCenterCache;
import com.xforceplus.ultraman.extensions.auth.plus.UserCenterCacheKey;
import com.xforceplus.ultraman.extensions.auth.plus.UserCenterCacheService;
import com.xforceplus.ultraman.extensions.auth.plus.UserCenterOrgJService;
import com.xforceplus.ultraman.extensions.auth.plus.model.*;
import com.xforceplus.ultraman.extensions.auth.plus.util.RestTemplateHelper;
import com.xforceplus.ultraman.extensions.auth.plus.util.UserCenterJHelper;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.extensions.auth.plus.util.CommonHelper.distinctByKey;


@Slf4j
public class UserCenterOrgJServiceImpl implements UserCenterOrgJService {

    private final UserCenterCacheService cacheService;

    private final RestTemplateHelper restTemplateHelper;

    private String env;

    public UserCenterOrgJServiceImpl(String env, UserCenterCacheService cacheService, RestTemplateHelper restTemplateHelper) {
        this.cacheService = cacheService;
        this.restTemplateHelper = restTemplateHelper;
        this.env = env;
    }

    @Override
    public UserCenterResponse<MultiScaleUserOrgData> getUserMultiScaleOrgIdMap(Long tenantId, Long userId) {
        final UserCenterCacheKey userCenterCacheKey = UserCenterCacheKey.GET_USER_MULTI_SCALE_ORG_ID_MAP;
        final String currentKey = String.format(userCenterCacheKey.getPattern(), userId);
        UserCenterCache<MultiScaleUserOrgData> scaleUserCache = cacheService.getUserCenterCache(userCenterCacheKey);
        final MultiScaleUserOrgData cacheValue = scaleUserCache.getGroupByTenantId(tenantId, currentKey);
        if (cacheValue != null) {
            log.info("Success Got Cache within GetUserMultiScaleOrgIdMap");
            //空集合也认为是缓存成功，仅不存在认为是无缓存
            return new UserCenterResponse<>(UserCenterResponse.SUCCESS_CODE, "成功", cacheValue);
        }

        final String subUrl = "/{tenantId}/v2/orgs?userId={userId}&status=1";
        UserCenterResponse<UserCenterList<OrgDataForMultiScale>> response = restTemplateHelper.iterate(env, subUrl, 1000, OrgDataForMultiScale.class, tenantId, userId);
        if (!UserCenterResponse.SUCCESS_CODE.equals(response.getCode())
                || response.getResult() == null
                || response.getResult().getContent() == null
                || response.getResult().getContent().isEmpty()) {
            return new UserCenterResponse<>(response.getCode(), response.getMessage(), null);
        }
        //按3种分割符拆分组织树
        final Map<Integer, List<Tuple2<List<Long>, String>>> groupByParentIdsSize = response.getResult().getContent().stream()
                .map(OrgDataForMultiScale::getParentIds)
                .filter(StringUtils::isNotBlank)
                .map(r -> Tuple.of(UserCenterJHelper.splitParentIds(r), r))
                .collect(Collectors.groupingBy(r -> r._1().size()));
        //如果父节点在返回结果里，过滤返回的其下全部子节点
        final Set<Integer> keySet = groupByParentIdsSize.keySet();
        final Iterator<Map.Entry<Integer, List<Tuple2<List<Long>, String>>>> iterator = groupByParentIdsSize.entrySet().stream().sorted(Map.Entry.comparingByKey()).iterator();
        while (iterator.hasNext()) {
            final Map.Entry<Integer, List<Tuple2<List<Long>, String>>> next = iterator.next();
            final List<Integer> filteredKeys = keySet.stream().filter(k -> k > next.getKey()).collect(Collectors.toList());
            next.getValue().forEach(v -> filteredKeys.forEach(key -> groupByParentIdsSize.getOrDefault(key, Lists.newArrayList()).removeIf(s -> s._2().startsWith(v._2()))));
        }

        Map<Integer, List<Long>> map = groupByParentIdsSize.entrySet().stream()
                .filter(r -> CollectionUtils.isNotEmpty(r.getValue()))
                .collect(Collectors.toMap(x -> x.getKey(), y -> y.getValue().stream().map(t -> t._1().get(t._1().size() - 1)).collect(Collectors.toList())));

        final MultiScaleUserOrgData multiScaleUserOrgData = new MultiScaleUserOrgData(map);
        scaleUserCache.trySetGroupByTenantId(tenantId, currentKey, multiScaleUserOrgData, true);
        return new UserCenterResponse<>(response.getCode(), response.getMessage(), multiScaleUserOrgData);
    }

    @Override
    public UserCenterResponse<List<MultiScaleUserCompanyData>> getUserMultiScaleCompanyList(Long tenantId, Long userId) {
        final UserCenterCacheKey userCenterCacheKey = UserCenterCacheKey.GET_USER_MULTI_SCALE_COMPANY_LIST;
        final String currentKey = String.format(userCenterCacheKey.getPattern(), userId);
        UserCenterCache<List<MultiScaleUserCompanyData>> cache = cacheService.getUserCenterCache(userCenterCacheKey);
        final List<MultiScaleUserCompanyData> cacheValue = cache.getGroupByTenantId(tenantId, currentKey);
        if (cacheValue != null) {
            //空集合也认为是缓存成功，仅不存在认为是无缓存
            return new UserCenterResponse<>(UserCenterResponse.SUCCESS_CODE, "成功", cacheValue);
        }

        final String subUrl = "/{tenantId}/v2/orgs?userId={userId}&status=1";
        UserCenterResponse<UserCenterList<OrgDataForMultiScale>> response = restTemplateHelper.iterate(env, subUrl, 1000, OrgDataForMultiScale.class, tenantId, userId);
        if (!UserCenterResponse.SUCCESS_CODE.equals(response.getCode())
                || response.getResult() == null
                || CollectionUtils.isEmpty(response.getResult().getContent())) {
            return new UserCenterResponse<>(response.getCode(), response.getMessage(), null);
        }
        final List<MultiScaleUserCompanyData> taxNoList = response.getResult().getContent().stream()
                //TODO
                .filter(r -> r.getOrgType() == 1)
                .map(r -> new MultiScaleUserCompanyData(r.getCompanyId(), r.getTaxNum(), r.getCompanyName()))
                .distinct()
                .collect(Collectors.toList());

        cache.trySetGroupByTenantId(tenantId, currentKey, taxNoList);
        return new UserCenterResponse<>(response.getCode(), response.getMessage(), taxNoList);
    }

    @Override
    public UserCenterResponse<OrgDataForDataCompletion> getOrgDataForDataCompletion(Long tenantId, Long companyId, String orgCode, Long orgId) {
        if ((Objects.isNull(companyId) || companyId <= 0L) && StringUtils.isBlank(orgCode) && (Objects.isNull(orgId) || orgId <= 0L)) {
            return new UserCenterResponse<>(UserCenterResponse.FAIL_CODE, "公司id或组织代码或组织id不能同时为空", null);
        }

        final UserCenterCacheKey userCenterCacheKey = UserCenterCacheKey.GET_ORG_DATA_FOR_DATA_COMPLETION;
        final String currentKey = String.format(userCenterCacheKey.getPattern(), companyId, orgCode, orgId);
        UserCenterCache<OrgDataForDataCompletion> cache = cacheService.getUserCenterCache(userCenterCacheKey);
        final OrgDataForDataCompletion cacheValue = cache.getGroupByTenantId(tenantId, currentKey);
        if (cacheValue != null) {
            //空集合也认为是缓存成功，仅不存在认为是无缓存
            log.info("Success Got Cache within OrgDataForDataCompletion");
            return new UserCenterResponse<>(UserCenterResponse.SUCCESS_CODE, "成功", cacheValue);
        }

        String subTempUrl = "/{tenantId}/v2/orgs?status=1";
        if (Objects.nonNull(companyId) && companyId > 0L) {
            subTempUrl += "&companyId=" + companyId;
        }
        if (StringUtils.isNotBlank(orgCode)) {
            subTempUrl += "&orgCode=" + orgCode;
        }
        if (Objects.nonNull(orgId) && orgId > 0L) {
            subTempUrl += "&orgId=" + orgId;
        }
        final String subUrl = subTempUrl;
        final UserCenterResponse<UserCenterList<RestOrgData>> response = restTemplateHelper.iterate(env, subUrl, 10, RestOrgData.class, tenantId);

        if (!UserCenterResponse.SUCCESS_CODE.equals(response.getCode())
                || response.getResult() == null
                || CollectionUtils.isEmpty(response.getResult().getContent())) {
            return new UserCenterResponse<>(response.getCode(), response.getMessage(), null);
        }

        final RestOrgData restOrgData = response.getResult().getContent().get(0);
        final List<Long> parentIdList = UserCenterJHelper.splitParentIds(restOrgData.getParentIds());
        final Map<Integer, Long> parentIdScaleMap = new LinkedHashMap<>();
        for (int i = 1; i <= parentIdList.size(); i++) {
            parentIdScaleMap.put(i, parentIdList.get(i - 1));
        }
        final OrgDataForDataCompletion orgDataForDataCompletion = new OrgDataForDataCompletion(
                restOrgData.getOrgId(),
                restOrgData.getOrgCode(),
                parentIdScaleMap
        );

        //能走到这一步的，都是非长效缓存租户，则默认缓存时间为120秒
        cache.trySetGroupByTenantId(tenantId, currentKey, orgDataForDataCompletion, false);
        return new UserCenterResponse<>(response.getCode(), response.getMessage(), orgDataForDataCompletion);
    }

    @Override
    public UserCenterResponse<List<MultiScaleUserCompanyData>> getUserOrgAllParentCompanies(Long tenantId, Long userId) {
        final UserCenterCacheKey userCenterCacheKey = UserCenterCacheKey.GET_USER_ORG_ALL_PARENT_COMPANIES;
        final String currentKey = String.format(userCenterCacheKey.getPattern(), userId);
        UserCenterCache<List<MultiScaleUserCompanyData>> cache = cacheService.getUserCenterCache(userCenterCacheKey);
        final List<MultiScaleUserCompanyData> cacheValue = cache.getGroupByTenantId(tenantId, currentKey);

        if (cacheValue != null) {
            //空集合也认为是缓存成功，仅不存在认为是无缓存
            return new UserCenterResponse<>(UserCenterResponse.SUCCESS_CODE, "成功", cacheValue);
        }

        final String subUrl = String.format("/%s/v2/orgs?userId=%s&status=1", tenantId, userId);
        final UserCenterResponse<UserCenterList<OrgDataForMultiScale>> response = restTemplateHelper.iterate(env, subUrl, 1000, OrgDataForMultiScale.class, tenantId, userId);

        if (!UserCenterResponse.SUCCESS_CODE.equals(response.getCode()) || response.getResult() == null || CollectionUtils.isEmpty(response.getResult().getContent())) {
            return new UserCenterResponse<>(response.getCode(), response.getMessage(), null);
        }

        final List<OrgDataForMultiScale> orgList = response.getResult().getContent();
        final Map<Long, OrgDataForMultiScale> orgIdToOrgMap = orgList.stream().collect(Collectors.toMap(OrgDataForMultiScale::getOrgId, r -> r));

        //已查询到的公司层级的组织id集合
        final List<OrgDataForMultiScale> companyOrgList = orgList.stream().filter(r -> r.getOrgType() == 1).collect(Collectors.toList());

        //组织层级的组织，如果其全部的父级组织id不在已查询的结果里，则发起二次查询，确认其是否是公司
        final Set<Long> notExistOrgIdSet = Sets.newHashSet();
        orgList.stream().filter(r -> StringUtils.isNotBlank(r.getParentIds()))
                .forEach(org -> {
                    final List<Long> orgIdList = UserCenterJHelper.splitParentIds(org.getParentIds());
                    orgIdList.forEach(r -> {
                        final OrgDataForMultiScale parentOrg = orgIdToOrgMap.get(r);
                        if (parentOrg == null) {
                            notExistOrgIdSet.add(r);
                        }
                    });
                });

        //父级组织未确认的，发起二次查询，确认其是否是公司
        if (CollectionUtils.isNotEmpty(notExistOrgIdSet)) {
            Lists.partition(Lists.newArrayList(notExistOrgIdSet), 50)
                    .forEach(partition -> {
                        final String subUrl2 = "/{tenantId}/v2/orgs?orgType=1&status=1&ids={ids}";
                        final UserCenterResponse<UserCenterList<OrgDataForMultiScale>> response2 = restTemplateHelper.iterate(env, subUrl2, 100, OrgDataForMultiScale.class, tenantId, partition.stream().map(String::valueOf).collect(Collectors.joining(",")));
                        if (UserCenterResponse.SUCCESS_CODE.equals(response2.getCode()) && response2.getResult() != null && CollectionUtils.isNotEmpty(response2.getResult().getContent())) {
                            companyOrgList.addAll(response2.getResult().getContent());
                        }
                    });
        }

        final List<MultiScaleUserCompanyData> taxNoList = companyOrgList.stream()
                .filter(distinctByKey(OrgDataForMultiScale::getCompanyId))
                .map(r -> new MultiScaleUserCompanyData(r.getCompanyId(), r.getTaxNum(), r.getCompanyName()))
                .distinct()
                .collect(Collectors.toList());

        cache.trySetGroupByTenantId(tenantId, currentKey, taxNoList);
        return new UserCenterResponse<>(response.getCode(), response.getMessage(), taxNoList);
    }


    @Override
    public UserCenterResponse<List<MultiScaleUserCompanyData>> getUserDirectParentCompanies(Long tenantId, Long userId) {
        final UserCenterCacheKey userCenterCacheKey = UserCenterCacheKey.GET_USER_DIRECT_PARENT_COMPANIES;
        final String currentKey = String.format(userCenterCacheKey.getPattern(), userId);
        UserCenterCache<List<MultiScaleUserCompanyData>> cache = cacheService.getUserCenterCache(userCenterCacheKey);
        final List<MultiScaleUserCompanyData> cacheValue = cache.getGroupByTenantId(tenantId, currentKey);
        if (cacheValue != null) {
            //空集合也认为是缓存成功，仅不存在认为是无缓存
            return new UserCenterResponse<>(UserCenterResponse.SUCCESS_CODE, "成功", cacheValue);
        }

        final String subUrl = "/{tenantId}/v2/orgs?userId={userId}&status=1";
        final UserCenterResponse<UserCenterList<OrgDataForMultiScale>> response = restTemplateHelper.iterate(env, subUrl, 1000, OrgDataForMultiScale.class, tenantId, userId);
        if (!UserCenterResponse.SUCCESS_CODE.equals(response.getCode()) || response.getResult() == null || CollectionUtils.isEmpty(response.getResult().getContent())) {
            return new UserCenterResponse<>(response.getCode(), response.getMessage(), null);
        }

        final List<OrgDataForMultiScale> orgList = response.getResult().getContent();
        final Map<Long, OrgDataForMultiScale> orgIdToOrgMap = orgList.stream().collect(Collectors.toMap(OrgDataForMultiScale::getOrgId, r -> r));

        //组织层级的组织，如果其全部的父级组织id不在已查询的结果里，则发起二次查询，确认其是否是公司
        final Set<Long> notExistOrgIdSet = Sets.newHashSet();
        orgList.stream().filter(r -> StringUtils.isNotBlank(r.getParentIds()))
                .forEach(org -> {
                    final List<Long> orgIdList = UserCenterJHelper.splitParentIds(org.getParentIds());
                    orgIdList.forEach(r -> {
                        final OrgDataForMultiScale parentOrg = orgIdToOrgMap.get(r);
                        if (parentOrg == null) {
                            notExistOrgIdSet.add(r);
                        }
                    });
                });

        //父级组织未确认的，发起二次查询，确认其是否是公司
        if (CollectionUtils.isNotEmpty(notExistOrgIdSet)) {
            Lists.partition(Lists.newArrayList(notExistOrgIdSet), 50)
                    .forEach(partition -> {
                        final String subUrl2 = "/{tenantId}/v2/orgs?orgType=1&status=1&ids={ids}";
                        final UserCenterResponse<UserCenterList<OrgDataForMultiScale>> response2 = restTemplateHelper.iterate(env, subUrl2, 100, OrgDataForMultiScale.class, tenantId,
                                partition.stream().map(String::valueOf).collect(Collectors.joining(",")));
                        if (UserCenterResponse.SUCCESS_CODE.equals(response2.getCode()) && response2.getResult() != null
                                && CollectionUtils.isNotEmpty(response2.getResult().getContent())) {
                            orgIdToOrgMap.putAll(response2.getResult()
                                    .getContent().stream().collect(Collectors
                                            .toMap(OrgDataForMultiScale::getOrgId, r -> r, (a, b) -> a)));
                        }
                    });
        }

        //获取用户组织权限的所有直接上级公司
        final List<OrgDataForMultiScale> companyOrgList = Lists.newArrayList();
        orgList.stream().filter(r -> StringUtils.isNotBlank(r.getParentIds()))
                .forEach(org -> {
                    final List<Long> orgIdList = UserCenterJHelper.splitParentIds(org.getParentIds());
                    for (int i = orgIdList.size() - 1; i >= 0; i--) {
                        final Long orgId = orgIdList.get(i);
                        final OrgDataForMultiScale parentOrg = orgIdToOrgMap.get(orgId);
                        if (parentOrg != null && parentOrg.getOrgType() == 1) {
                            companyOrgList.add(parentOrg);
                            break;
                        }
                    }
                });

        final List<MultiScaleUserCompanyData> taxNoList = companyOrgList.stream()
                .filter(distinctByKey(OrgDataForMultiScale::getCompanyId))
                .map(r -> new MultiScaleUserCompanyData(r.getCompanyId(), r.getTaxNum(), r.getCompanyName()))
                .distinct()
                .collect(Collectors.toList());

        cache.trySetGroupByTenantId(tenantId, currentKey, taxNoList);

        return new UserCenterResponse<>(response.getCode(), response.getMessage(), taxNoList);
    }


}
