package com.xforceplus.ultraman.extensions.cdc.status.impl;

import com.xforceplus.ultraman.extensions.cdc.status.domain.CurrentStatus;
import com.xforceplus.ultraman.extensions.cdc.status.domain.ProfiledEntityClassStatus;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.sdk.infra.base.id.LongIdGenerator;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.ParameterizedPreparedStatementSetter;
import org.springframework.jdbc.core.RowMapper;

import javax.sql.DataSource;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.metadata.values.DateTimeValue.ZONE_ID;
import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;

/**
 * DB implementation
 */
@Slf4j
public class DBStatusServiceImpl extends StatusServiceSupport {

    private final static String CREATE_SQL = "INSERT INTO CDC_STATUS(ID, PROFILE, ENTITYCLASS_ID, UPDATE_TIME) VALUES (?,?,?,?)";

    private final static String CLEAN_SQL = "DELETE FROM CDC_STATUS WHERE UPDATE_TIME <= ? ";

    private final static String CLEAN_SQL_FOR_SPECIAL = "DELETE FROM CDC_STATUS WHERE UPDATE_TIME <= ? AND ENTITYCLASS_ID = ? AND PROFILE = ?";

    private final static String SELECT_SQL = "SELECT PROFILE, ENTITYCLASS_ID, MIN(UPDATE_TIME) AS UPDATE_TIME FROM CDC_STATUS GROUP BY PROFILE, ENTITYCLASS_ID";
    private final Map<Tuple2<Long, String>, List<Tuple2<Long, CountDownLatch>>> lockMap = new ConcurrentHashMap<>();
    private final ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(1);
    private final ScheduledExecutorService scheduledExecutorServiceNotifier = Executors.newScheduledThreadPool(1);
    private int WINDOW = 30;
    
    private boolean enableTimeout = false;
    
    private JdbcTemplate jdbcTemplate;
    @Autowired
    private LongIdGenerator longIdGenerator;

    public DBStatusServiceImpl(ExecutorService executorService, DataSource dataSource, int window, boolean enableTimeout) {
        super(executorService);
        this.jdbcTemplate = new JdbcTemplate(dataSource);
        this.WINDOW = window;
        this.enableTimeout = enableTimeout;
    }

    public void setMainJdbcTemplate(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    public void setLongIdGenerator(LongIdGenerator longIdGenerator) {
        this.longIdGenerator = longIdGenerator;
    }

    public void startNotify() {
        scheduledExecutorServiceNotifier.scheduleAtFixedRate(() -> {
            try {
                CurrentStatus overview = this.getOverview();
                Map<Tuple2<Long, String>, ProfiledEntityClassStatus> overviewMapping = overview.getOverview();
                //using current lock map
                lockMap.forEach((k, v) -> {
                    Tuple2<Long, String> target = Tuple.of(k._1, "");
                    ProfiledEntityClassStatus classStatus = overviewMapping.get(target);
                    if (classStatus != null) {
                        long timestamp = classStatus.getTimestamp();
                        //count down all
                        v.stream().filter(x -> x._1 < timestamp).forEach(x -> x._2().countDown());
                    } else {
                        v.forEach(x -> x._2().countDown());
                    }
                });
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                        , "StatusServiceSync"
                        , throwable);
            }
        }, 20, 1, TimeUnit.SECONDS);
    }

    @Override
    protected void save(Map<Tuple2<String, Long>, ?> grouped, long timestamp) {
        try {
      
            Set<Tuple2<String, Long>> entityRefs = grouped.keySet();
            jdbcTemplate.batchUpdate(CREATE_SQL, entityRefs, entityRefs.size(), (ps, argument) -> {
                ps.setLong(1, longIdGenerator.next());
//                ps.setString(2, argument._1);
                //clear profile
                log.debug("Save Status at {}, {}", timestamp, argument._2);
                ps.setString(2, "");
                ps.setLong(3, argument._2);
                ps.setLong(4, timestamp);
            });
        } catch (Throwable throwable) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , "SaveStatus"
                    , throwable);
        }
    }

    @Override
    protected void clear(Collection<OqsEngineEntity> oqsEngineEntity) {
        try {
            //group the oqsEngineEntity with profile and entity
            Map<Tuple2<String, Long>, List<OqsEngineEntity>> group = oqsEngineEntity.stream().collect(Collectors
                    .groupingBy(x -> Tuple.of(
                            x.getEntityClassRef().getProfile()
                            , x.getEntityClassRef().getId())));


            Set<Tuple2<String, Long>> entityRefs = group.keySet();

            Instant instant = LocalDateTime.now().atZone(ZONE_ID).toInstant();
            long currentTime = instant.toEpochMilli();
//
            jdbcTemplate.batchUpdate(CLEAN_SQL_FOR_SPECIAL, entityRefs, entityRefs.size(), new ParameterizedPreparedStatementSetter<Tuple2<String, Long>>() {
                @Override
                public void setValues(PreparedStatement ps, Tuple2<String, Long> argument) throws SQLException {
                    List<OqsEngineEntity> oqsEngineEntities = group.get(argument);
                    long lastUpdateTime = findLastUpdateTime(oqsEngineEntities);
                    log.debug("Clear Status at {} {}", lastUpdateTime, argument._2);
                    ps.setLong(1, lastUpdateTime);
                    ps.setLong(2, argument._2);
//                    ps.setString(3, argument._1);
                    ps.setString(3, "");
                }

                private long findLastUpdateTime(List<OqsEngineEntity> oqsEngineEntities) {
                    OptionalLong max = oqsEngineEntities.stream().mapToLong(x -> {
                        return x.getUpdateTime();
                    }).max();

                    return max.orElseGet(() -> {
                        log.error("Cannot find UPDATE_TIME in RECORD");
                        return currentTime;
                    });
                }
            });
        } catch (Throwable throwable) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , "ClearStatus"
                    , throwable);
        }
    }

    @Override
    public void clearStatus(long timestamp) {

    }

    @Override
    public CurrentStatus getOverview() {
        try {
            List<ProfiledEntityClassStatus> query = jdbcTemplate.query(SELECT_SQL, new RowMapper<ProfiledEntityClassStatus>() {

                @Override
                public ProfiledEntityClassStatus mapRow(ResultSet rs, int rowNum) throws SQLException {
                    ProfiledEntityClassStatus statusItem = new ProfiledEntityClassStatus();
                    String profile = rs.getString("PROFILE");
                    long entityClassId = rs.getLong("ENTITYCLASS_ID");
                    long timestamp = rs.getLong("UPDATE_TIME");
                    statusItem.setProfile(profile);
                    statusItem.setEntityClassId(entityClassId);
                    statusItem.setTimestamp(timestamp);
                    return statusItem;
                }
            });

            CurrentStatus currentStatus = new CurrentStatus();
            Map<Tuple2<Long, String>, ProfiledEntityClassStatus> mapped = query.stream()
                    .collect(Collectors.toMap(x -> Tuple.of(x.getEntityClassId(), x.getProfile()), y -> y, (a, b) -> a));
            currentStatus.setOverview(mapped);

            return currentStatus;
        } catch (Throwable throwable) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , "GetOverview"
                    , throwable);
            return null;
        }
    }

    @Override
    public void query(List<Tuple2<Long, String>> profiledEntityClass, long requestTimestamp) {
        log.debug("Query Status at {}", requestTimestamp);
        CurrentStatus overview = this.getOverview();
        log.debug("Current Pending List {}", overview);
        List<CountDownLatch> countDownLatches = new ArrayList<>();
        /**
         * check if has time before
         */
        for (Tuple2<Long, String> entityClass : profiledEntityClass) {
            Map<Tuple2<Long, String>, ProfiledEntityClassStatus> overviewMapping = overview.getOverview();
            //flatten to non profile one TODO
            Tuple2<Long, String> target = Tuple.of(entityClass._1, "");
            ProfiledEntityClassStatus classStatus = overviewMapping.get(target);
            if (classStatus != null) {
                long timestamp = classStatus.getTimestamp();
                if (timestamp <= requestTimestamp) {
                    //need a lock here
                    CountDownLatch countDownLatch = new CountDownLatch(1);
                    countDownLatches.add(countDownLatch);
                    lockMap.compute(entityClass, (k, v) -> {
                        if (v == null) {
                            v = new ArrayList<>();
                        }
                        v.add(Tuple.of(timestamp, countDownLatch));
                        return v;
                    });
                }
            }
        }

        for (CountDownLatch countDownLatch : countDownLatches) {
            try {
                if(enableTimeout) {
                    countDownLatch.await(WINDOW - 1, TimeUnit.SECONDS);
                } else {
                    countDownLatch.await();
                }
            } catch (InterruptedException e) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                        , "QueryInterceptor"
                        , e);
            }
        }
    }

    @Override
    public void onEmpty() {
        
    }

    /**
     * clean the timeout record
     */
    public void startClean() {
        scheduledExecutorService.scheduleAtFixedRate(this::cleanTimeout, 100, WINDOW, TimeUnit.SECONDS);
    }

    public void shutDown() {

    }

    private void cleanTimeout() {
        try {
            LocalDateTime minus = LocalDateTime.now().minusSeconds(WINDOW);
            Instant instant = minus.atZone(ZONE_ID).toInstant();
            long currentTime = instant.toEpochMilli();
            jdbcTemplate.update(CLEAN_SQL, currentTime);
            lockMap.values().removeIf(x -> {
                x.removeIf(y -> y._2.getCount() == 0);
                return x.size() == 0;
            });
        } catch (Throwable throwable) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , "cleanTimeout"
                    , throwable);
        }
    }
}
