package com.xforceplus.ultraman.extensions.cdc.status.impl;

import com.xforceplus.ultraman.extensions.cdc.status.domain.CurrentStatus;
import com.xforceplus.ultraman.extensions.cdc.status.domain.ProfiledEntityClassStatus;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.sdk.infra.base.id.LongIdGenerator;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.checkerframework.checker.units.qual.Current;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.RowMapper;

import javax.sql.DataSource;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.metadata.values.DateTimeValue.ZONE_ID;
import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;

/**
 * save with binlog time and do not record when entity changed is a simple mode
 * but will block when other records insert
 */
@Slf4j
public class SimpleDBStatusServiceImpl extends StatusServiceSupport {

    private final static String CREATE_SQL = "INSERT INTO CDC_STATUS(ID, UPDATE_TIME) VALUES (?,?)";

    private final static String CLEAN_ALL_SQL = "TRUNCATE TABLE CDC_STATUS";

    private final static String CLEAN_SQL_BY_UPDATE_TIME = "DELETE FROM CDC_STATUS WHERE UPDATE_TIME <= ? ";

    private final static String SELECT_SQL = "SELECT MAX(UPDATE_TIME) AS UPDATE_TIME FROM CDC_STATUS ";
    private final Map<Long, List<CountDownLatch>> lockMap = new ConcurrentHashMap<>();

    private int WINDOW = 30;

    private boolean enableTimeout = false;

    private JdbcTemplate jdbcTemplate;
    @Autowired
    private LongIdGenerator longIdGenerator;

    private final ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(1);

    private final ScheduledExecutorService scheduledExecutorServiceNotifier = Executors.newScheduledThreadPool(1);

    public SimpleDBStatusServiceImpl(ExecutorService executorService, DataSource dataSource, int window, boolean enableTimeout) {
        super(executorService);
        this.jdbcTemplate = new JdbcTemplate(dataSource);
        this.WINDOW = window;
        this.enableTimeout = enableTimeout;
    }

    @Override
    public void clearStatus(long timestamp) {

    }

    @Override
    public CurrentStatus getOverview() {
        try {
            ProfiledEntityClassStatus query = jdbcTemplate.queryForObject(SELECT_SQL, (rs, rowNum) -> {
                ProfiledEntityClassStatus statusItem = new ProfiledEntityClassStatus();
                long timestamp = rs.getLong("UPDATE_TIME");
                statusItem.setProfile(null);
                statusItem.setEntityClassId(0L);
                statusItem.setTimestamp(timestamp);
                return statusItem;
            });

            CurrentStatus currentStatus = new CurrentStatus();
            if (query != null) {
                currentStatus.setTimestamp(query.getTimestamp());
            } else {
                currentStatus.setTimestamp(System.currentTimeMillis());
            }

            return currentStatus;
        } catch (Throwable throwable) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , "GetOverview"
                    , throwable);
            return null;
        }
    }

    @Override
    public void query(List<Tuple2<Long, String>> profiledEntityClass, long requestTimestamp) {
        log.debug("Query Status at {}", requestTimestamp);
        CurrentStatus overview = this.getOverview();
        log.debug("Current Sync time {}", overview.getTimestamp());
        List<CountDownLatch> countDownLatches = new ArrayList<>();

        if (overview.getTimestamp() == 0) {
            return;
        }

        if (requestTimestamp >= overview.getTimestamp()) {
            CountDownLatch countDownLatch = new CountDownLatch(1);
            countDownLatches.add(countDownLatch);
            lockMap.compute(requestTimestamp, (k, v) -> {
                if (v == null) {
                    v = new ArrayList<>();
                }
                v.add(countDownLatch);
                return v;
            });
        }

        for (CountDownLatch countDownLatch : countDownLatches) {
            try {
                if (enableTimeout) {
                    countDownLatch.await(WINDOW - 1, TimeUnit.SECONDS);
                } else {
                    countDownLatch.await();
                }
            } catch (InterruptedException e) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                        , "Query"
                        , e);
            }
        }
    }

    @Override
    public void onEmpty() {
        //can clear all
        jdbcTemplate.execute(CLEAN_ALL_SQL);
        lockMap.values().stream().flatMap(x -> x.stream()).forEach(x -> x.countDown());
        lockMap.clear();
    }

    @Override
    protected void save(Map<Tuple2<String, Long>, ?> grouped, long timestamp) {
        //this is save with request
        //DO nothing
    }

    /**
     * insert
     *
     * @param oqsEngineEntity
     */
    @Override
    protected void clear(Collection<OqsEngineEntity> oqsEngineEntity) {
        //记录
        try {

            /**
             * (ps, argument) -> {
             *                 ps.setLong(1, longIdGenerator.next());
             * //                ps.setString(2, argument._1);
             *                 //clear profile
             *                 log.debug("Save Status at {}, {}", timestamp, argument._2);
             *                 ps.setString(2, "");
             *                 ps.setLong(3, argument._2);
             *                 ps.setLong(4, timestamp);
             *             }
             */
            OptionalLong maxTime = oqsEngineEntity.stream().mapToLong(x -> x.getUpdateTime()).max();
            jdbcTemplate.update(CREATE_SQL, ps -> {
                ps.setLong(1, longIdGenerator.next());
                ps.setLong(2, maxTime.orElse(System.currentTimeMillis()));
            });
        } catch (Throwable throwable) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , "Clear"
                    , throwable);
        }
    }

    private void cleanTimeout() {
        try {
            LocalDateTime minus = LocalDateTime.now().minusSeconds(WINDOW);
            Instant instant = minus.atZone(ZONE_ID).toInstant();
            long currentTime = instant.toEpochMilli();
            jdbcTemplate.update(CLEAN_SQL_BY_UPDATE_TIME, currentTime);
            lockMap.values().removeIf(x -> {
                x.removeIf(y -> y.getCount() == 0);
                return x.isEmpty();
            });
        } catch (Throwable throwable) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , "CleanTimeout"
                    , throwable);
        }
    }

    public void startClean() {
        scheduledExecutorService.scheduleAtFixedRate(this::cleanTimeout, 100, WINDOW, TimeUnit.SECONDS);
    }

    public void startNotify() {
        scheduledExecutorServiceNotifier.scheduleAtFixedRate(() -> {
            try {
                CurrentStatus overview = this.getOverview();
                long newSyncTime = overview.getTimestamp();
                
                if(newSyncTime == 0) {
                    //done all
                    lockMap.values().stream().flatMap(Collection::stream).forEach(CountDownLatch::countDown);
                    return;
                }
                
                //using current lock map
                lockMap.forEach((k, v) -> {
                    if (k <= newSyncTime) {
                        v.forEach(CountDownLatch::countDown);
                    }
                });
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                        , "StarSync"
                        , throwable);
            }

        }, 20, 1, TimeUnit.SECONDS);
    }
}
